/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
2006 Andy Jefferson - moved to Core. Removed RDBMS mappings.
2007 Andy Jefferson - support roleForMember so it handles collection elements etc
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.math.BigInteger;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.ColumnMetaData;
import org.datanucleus.metadata.FieldRole;
import org.datanucleus.metadata.MetaDataUtils;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.expression.IntegerLiteral;
import org.datanucleus.store.mapped.expression.Literal;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.NumericExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.StringExpression;
import org.datanucleus.store.mapped.expression.StringLiteral;
import org.datanucleus.store.mapped.mapping.SingleFieldMapping;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Mapping for JDK1.5 Enum type.
 **/
public class EnumMapping extends SingleFieldMapping
{
    protected String datastoreJavaType = ClassNameConstants.JAVA_LANG_STRING;

    /**
     * Initialize this JavaTypeMapping with the given DatastoreAdapter for the given FieldMetaData.
     * @param container The datastore container storing this mapping (if any)
     * @param clr the ClassLoaderResolver
     * @param fmd FieldMetaData for the field to be mapped (if any)
     */
    public void initialize(AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
        if (fmd != null && fmd.isSerialized())
        {
            datastoreJavaType = ClassNameConstants.JAVA_IO_SERIALIZABLE;
        }
        else if (fmd != null)
        {
            ColumnMetaData[] colmds = getColumnMetaDataForMember(fmd, roleForMember);
            if (colmds != null)
            {
                // Check if the user requested an INTEGER based datastore type
                if (MetaDataUtils.isJdbcTypeNumeric(colmds[0].getJdbcType()))
                {
                    datastoreJavaType = ClassNameConstants.JAVA_LANG_INTEGER;
                }
            }
        }

        super.initialize(fmd, container, clr);
    }

    /**
     * Accessor for the valid values for this mapping (if any restriction is imposed).
     * @param index The index of the datastore column
     * @return The valid value(s)
     */
    public Object[] getValidValues(int index)
    {
        // this block defines check constraints based on the values of enums
        if (mmd != null)
        {
            if (mmd.getColumnMetaData() != null && 
                mmd.getColumnMetaData().length > 0 && 
                mmd.getColumnMetaData()[0].hasExtension("enum-check-constraint") &&
                mmd.getColumnMetaData()[0].getValueForExtension("enum-check-constraint").equalsIgnoreCase("true"))
            {
                try
                {
                    Enum[] values = (Enum[])mmd.getType().getMethod("values", (Class[])null).invoke((Object)null, (Object[])null);
                    if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_STRING))
                    {
                        String[] valueStrings = new String[values.length];
                        for (int i=0;i<values.length;i++)
                        {
                            valueStrings[i] = values[i].toString();
                        }
                        return valueStrings;
                    }
                    else
                    {
                        Integer[] valueInts = new Integer[values.length];
                        for (int i=0;i<values.length;i++)
                        {
                            valueInts[i] = values[i].ordinal();
                        }
                        return valueInts;
                    }
                }
                catch (Exception e)
                {
                    NucleusLogger.PERSISTENCE.warn(StringUtils.getStringFromStackTrace(e));
                }
            }
        }
        return super.getValidValues(index);
    }

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        return datastoreJavaType;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapped.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return Enum.class;
    }

    /**
     * Method to set the Enum in the datastore statement.
     * @param om ObjectManager
     * @param preparedStatement Statement for the datastore
     * @param exprIndex Index position(s) to set the Enum at in the statement
     * @param value The Enum value to set
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
        if (value == null)
        {
            getDatastoreMapping(0).setObject(preparedStatement, exprIndex[0], null);
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_INTEGER))
        {
            if (value instanceof Enum)
            {
                int intVal = ((Enum)value).ordinal();
                getDatastoreMapping(0).setInt(preparedStatement, exprIndex[0], intVal);
            }
            else if (value instanceof BigInteger)
            {
                // ordinal value passed in directly (e.g ENUM_FIELD == ? in query)
                getDatastoreMapping(0).setInt(preparedStatement, exprIndex[0], ((BigInteger)value).intValue());
            }
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_STRING))
        {
            String stringVal;
            if (value instanceof String)
            {
                //it may be String when there is a compare between Enums values and Strings in the JDOQL query (for example)
                stringVal = (String)value;
            }
            else
            {
                stringVal = ((Enum)value).name();
            }
            getDatastoreMapping(0).setString(preparedStatement, exprIndex[0], stringVal);
        }
        else
        {
            super.setObject(om, preparedStatement, exprIndex, value);
        }
    }

    /**
     * Method to extract the Enum object from the passed result set.
     * @param om ObjectManager
     * @param resultSet The result set
     * @param exprIndex The index position(s) in the result set to use.
     * @return The Enum
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        if (exprIndex == null)
        {
            return null;
        }

        if (getDatastoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_INTEGER))
        {
            long longVal = getDatastoreMapping(0).getLong(resultSet, exprIndex[0]);
            Class enumType = null;
            if (mmd == null)
            {
                enumType = om.getClassLoaderResolver().classForName(type);
            }
            else
            {
                enumType = mmd.getType();
                if (roleForMember == FieldRole.ROLE_COLLECTION_ELEMENT)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getCollection().getElementType());
                }
                else if (roleForMember == FieldRole.ROLE_ARRAY_ELEMENT)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getArray().getElementType());
                }
                else if (roleForMember == FieldRole.ROLE_MAP_KEY)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getMap().getKeyType());
                }
                else if (roleForMember == FieldRole.ROLE_MAP_VALUE)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getMap().getValueType());
                }
            }
            return enumType.getEnumConstants()[(int)longVal];
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_STRING))
        {
            String stringVal = getDatastoreMapping(0).getString(resultSet, exprIndex[0]);
            Class enumType = null;
            if (mmd == null)
            {
                enumType = om.getClassLoaderResolver().classForName(type);
            }
            else
            {
                enumType = mmd.getType();
                if (roleForMember == FieldRole.ROLE_COLLECTION_ELEMENT)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getCollection().getElementType());
                }
                else if (roleForMember == FieldRole.ROLE_ARRAY_ELEMENT)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getArray().getElementType());
                }
                else if (roleForMember == FieldRole.ROLE_MAP_KEY)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getMap().getKeyType());
                }
                else if (roleForMember == FieldRole.ROLE_MAP_VALUE)
                {
                    enumType = om.getClassLoaderResolver().classForName(mmd.getMap().getValueType());
                }
            }
            return Enum.valueOf(enumType, stringVal);
        }
        else
        {
            return super.getObject(om, resultSet, exprIndex);
        }
    }

    public Object getSampleValue(ClassLoaderResolver clr)
    {
        throw new UnsupportedOperationException();
    }

    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_INTEGER))
        {
            // Return an IntegerLiteral to represent this Enum, but with the original value present just in case
            ScalarExpression expr = new IntegerLiteral(qs, this, BigInteger.valueOf(((Enum)value).ordinal()));
            ((Literal)expr).setRawValue(value);
            return expr;
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_STRING))
        {
            // Return an StringLiteral to represent this Enum, but with the original value present just in case
            ScalarExpression expr = new StringLiteral(qs, this, ((Enum)value).name());
            ((Literal)expr).setRawValue(value);
            return expr;
        }
        else
        {
            // No querying of serialised Enums!
            return null;
        }
    }

    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_INTEGER))
        {
            return new NumericExpression(qs, this, te);
        }
        else if (datastoreJavaType.equals(ClassNameConstants.JAVA_LANG_STRING))
        {
            return new StringExpression(qs, this, te);
        }
        else
        {
            // No querying of serialised Enums!
            return null;
        }
    }
}