/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2003 Andy Jefferson - coding standards
2005 Andy Jefferson - added support for implicit variables
2006 Andy Jefferson - support for using in result clauses
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.IdentifierType;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Representation of an unbound variable in a Query.
 **/
public class UnboundVariable extends ScalarExpression
{
    /** Name of the variable */
    private final String name;

    /** Type of the variable. This may be assigned later if implicit. */
    private Class type;

    /** Reference to the Binder so that we can bind the variable at the appropriate point. */
    private final UnboundVariable.VariableBinder binder;

    /**
     * Constructor.
     * @param qs The Query Expression
     * @param name Name of the variable
     * @param type Type of the variable (if known at this point)
     * @param binder The Binder that we can bind with this variable when we know which expression to bind to.
     */
    public UnboundVariable(QueryExpression qs, String name, Class type, UnboundVariable.VariableBinder binder)
    {
        super(qs);

        this.name = name;
        this.type = type;
        this.binder = binder;
        getExpressionToBindToThis();
        
    }

    /**
     * Accessor for the variable name.
     * @return The variable name
     */
    public String getVariableName()
    {
        return name;
    }
    
    /**
     * @return may return null, if type is unknown
     */
    public LogicSetExpression getLogicSetExpression()
    {
        if (te==null)
        {
            ScalarExpression boundExpr = getExpressionToBindToThis();
            if( boundExpr == null )
            {
                return null;
            }
            return boundExpr.getLogicSetExpression();
        }
        return te;
    }

    /**
     * Accessor for the variable type.
     * @return The variable type
     */
    public Class getVariableType()
    {
        return type;
    }

    /**
     * Mutator for the variable type in the case where we have an implicit variable and
     * its type is not known at construction. Only updates the type if it was null at construction.
     * @param type The type
     */
    public void setVariableType(Class type)
    {
        this.type = type;
        getExpressionToBindToThis();
    }

    /**
     * Method to bind this variable to its expression, using the JDOQL compiler.
     * @param qsc The expression to bind to.
     */
    public void bindTo(ScalarExpression qsc)
    {
        binder.bindVariable(name, qsc);
    }

    /**
     * Change the output statement text to refer to the bound to expression
     * since it has real fields and this is only a variable.
     * @param mode Mode of operation
     * @return The statement text for this unbound variable
     */
    public StatementText toStatementText(int mode)
    {
        ScalarExpression exprToBind = getExpressionToBindToThis();
        if (exprToBind == null)
        {
            throw new NucleusUserException("Unconstrained variable referenced: " + name);
        }
        if( !qs.hasCrossJoin(exprToBind.te))
        {
            qs.crossJoin(exprToBind.te,true);
        }

        return exprToBind.toStatementText(mode);
    }

    public ExpressionList getExpressionList()
    {
        ScalarExpression exprToBind = getExpressionToBindToThis();
        if (exprToBind == null)
        {
            throw new NucleusUserException("Unconstrained variable referenced: " + name);
        }
        if( !qs.hasCrossJoin(exprToBind.te))
        {
            qs.crossJoin(exprToBind.te,true);
        }
        return exprToBind.getExpressionList();
    }
    
    /**
     * StatementText representation of this expression. I.E. A Boolean field may be stored in 
     * boolean format like 0 or 1, and it can also be stored in other formats, like Y or N, TRUE or FALSE, and so on.
     * The projection mode for the boolean field is the real content of the value stored, (e.g. Y or N), and opposed to
     * that the filter mode for the boolean field is always represented by a boolean expression (e.g. Y=Y or N=N)
     * In SQL, the projection can be exemplified as "SELECT BOOLEAN_FIELD ... " and the filter as 
     * "SELECT COLUMNS ... WHERE BOOLEAN_FIELD ='Y'" 
     * 
     * @return the StatementText 
     */    
    public StatementText toStatementText()
    {
        throw new NucleusUserException("Unconstrained variable referenced: " + name);
    }
    
    /**
     * Checks if one expression is equals the other expression
     * @param expr the expression to check if this is equals to the expression
     * @return the BooleanExpression
     */
    public BooleanExpression eq(ScalarExpression expr)
    {
        ScalarExpression boundExpr = getExpressionToBindToThis();
        bindTo(boundExpr);

        //qs.innerJoin(expr, boundExpr, boundExpr.te, true, true);
        if( qs.hasCrossJoin(boundExpr.te))
        {
            //qs.addAlias(boundExpr.te,true);
            qs.andCondition(expr.eq(boundExpr));
        }
        else
        {
            qs.innerJoin(expr, boundExpr, boundExpr.te, true, true);
        }
        return expr.eq(boundExpr);
    }

    /**
     * Checks if one expression is not equals the other expression
     * @param expr the expression to check if this is not equals to the expression
     * @return the BooleanExpression
     */
    public BooleanExpression noteq(ScalarExpression expr)
    {
        ScalarExpression boundExpr = getExpressionToBindToThis();
        bindTo(boundExpr);

        //qs.innerJoin(expr, boundExpr, boundExpr.te, false, true);
        if( qs.hasCrossJoin(boundExpr.te))
        {
            //qs.addAlias(boundExpr.te,true);
           // qs.andCondition(expr.noteq(boundExpr));
        }
        else
        {
            qs.innerJoin(expr, boundExpr, boundExpr.te, false, true);
        }

        
        return expr.noteq(boundExpr);
    }

    /**
     * Method to access a field in the class of the variable.
     * @param fieldName Name of the field to access
     * @param innerJoin whether to use an inner join to access this field
     * @return The scalar expression for the field
     */
    public ScalarExpression accessField(String fieldName, boolean innerJoin)
    {
        ScalarExpression exprToBind = getExpressionToBindToThis();
        bindTo(exprToBind);
        qs.crossJoin(exprToBind.getLogicSetExpression(), true);

        return exprToBind.accessField(fieldName, innerJoin);
    }

    /** Cached expression that we should bind to. */
    ScalarExpression exprToBind = null;

    /**
     * Convenience method to get the expression to bind to for this unbound variable.
     * Caches the variable it finds so that we dont find it multiple times.
     * @return The expression to bind to
     */
    private ScalarExpression getExpressionToBindToThis()
    {
        if (exprToBind != null)
        {
            // Use the cached value
            return exprToBind;
        }

        if (type == null)
        {
            // No type specified, must be implicit parameter
            return null;
        }
        ApiAdapter api = qs.getStoreManager().getApiAdapter();
        if (!api.isPersistable(type))
        {
            // Type is not PC so cannot bind to it
            return null;
        }

        AbstractClassMetaData typeCmd = qs.getStoreManager().getOMFContext().getMetaDataManager().getMetaDataForClass(type,qs.getClassLoaderResolver());
        if (typeCmd.isEmbeddedOnly())
        {
            // No table for this class so cannot bind to it
            return null;
        }

        String jtJavaName = "UNBOUND" + '.' + this.getVariableName();

        DatastoreClass cbt = qs.getStoreManager().getDatastoreClass(type.getName(), qs.getClassLoaderResolver());
        DatastoreIdentifier jtRangeVar = qs.getStoreManager().getIdentifierFactory().newIdentifier(IdentifierType.TABLE, jtJavaName);
        LogicSetExpression jtExpr = qs.getTableExpression(jtRangeVar);
        if (jtExpr == null)
        {
            jtExpr = qs.newTableExpression(cbt, jtRangeVar);
            //qs.addAlias(jtExpr, true);
        }

        // bind unbound variable
        JavaTypeMapping mb = cbt.getIdMapping();
        ScalarExpression exprBindTo = mb.newScalarExpression(qs, jtExpr);

        // Save the mapping in case we are selecting this in the result clause later
        mapping = mb;

        exprToBind = exprBindTo;

        return exprBindTo;
    }
    
    public static interface VariableBinder
    {
        /**
         * Bind a variable to the query.
         * @param name Name of the variable
         * @param expr The expression
         */
        void bindVariable(String name, ScalarExpression expr);
    }
}