/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.HashMap;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.MappedStoreManager;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Expression for a query in language-independent form.
 */
public interface QueryExpression
{
    /**
     * Sets the parent QueryExpression of this query. In SQL it can be exemplified as
     * <code> 
     * SELECT 1 FROM PARENT WHERE EXISTS (SELECT 1 FROM THIS)
     * </code>
     * The parent QueryExpression is the nesting SQL.
     * 
     * @param parentQueryExpr the parent of this query
     **/
    void setParent(QueryExpression parentQueryExpr);

    /**
     * Accessor for the parent QueryExpression if this is a nested expression.
     * @return Parent expression
     */
    QueryExpression getParent();

    /**
     * Method to set the candidate class and alias in use by the query.
     * The expression is created with a candidate table, yet this could store more than 1 class.
     * Additionally the "alias" of the candidate table expression is a DatastoreIdentifier whereas
     * this alias here is a String form.
     * @param cls The candidate class
     * @param alias The alias
     */
    void setCandidateInformation(Class cls, String alias);

    /**
     * Accessor for the candidate class of the query expression.
     * @return Candidate class
     */
    Class getCandidateClass();

    /**
     * Accessor for the candidate alias in use by the query.
     * @return Candidate alias
     */
    String getCandidateAlias();

    /**
     * Accessor for the expression for the main table of this query.
     * This is the same as the default table expression except where this is a subquery.
     * @return Main table expression
     */
    LogicSetExpression getMainTableExpression();

    /**
     * Accessor for the alias of the main table of this query.
     * @return Alias for the main table in the query
     */
    DatastoreIdentifier getMainTableAlias();

    /**
     * Accessor to the table expression for the given alias.
     * @param alias the alias
     * @return the TableExpression
     */
    LogicSetExpression getTableExpression(DatastoreIdentifier alias);

    /**
     * Creates a table expression
     * @param mainTable the main table
     * @param alias the alias
     * @return TableExpression
     */
    LogicSetExpression newTableExpression(DatastoreContainerObject mainTable, DatastoreIdentifier alias);
    
    /**
     * Creates a table expression
     * @param mainTable the main table
     * @param alias the alias
     * @param unionQueries Whether to add to any union
     * @return TableExpression[]
     */
    LogicSetExpression[] newTableExpression(DatastoreContainerObject mainTable, DatastoreIdentifier alias, boolean unionQueries);

    /**
     * Accessor for the store manager associated with this query.
     * @return The store manager
     */
    MappedStoreManager getStoreManager();

    /**
     * Accessor for the ClassLoaderResolver to use with this query statement.
     * @return ClassLoader resolver
     */
    ClassLoaderResolver getClassLoaderResolver();

    /**
     * Set whether this statement returns distinct results.
     * @param distinctResults Whether we return distinct results
     */
    void setDistinctResults(boolean distinctResults);

    /**
     * Method to define an extension for this query statement allowing control over its behaviour
     * in generating a query.
     * @param key Extension key
     * @param value Value for the key
     */
    void addExtension(String key, Object value);

    /**
     * Accessor for the value for an extension.
     * @param key Key for the extension
     * @return Value for the extension (if any)
     */
    Object getValueForExtension(String key);

    /**
     * Accessor for the extensions for this expression.
     * @return Extensions
     */
    HashMap getExtensions();

    /**
     * Whether this query will return a meta data expression in the SELECT clause
     * @return hasMetaDataExpression
     */
    boolean hasNucleusTypeExpression();

    /**
     * Select the datastore identity column.
     * @param alias Alias to use for this column
     * @param unionQueries Whether to select the datastore id column of all unioned tables
     * @return The position of the column in the result set
     */
    int[] selectDatastoreIdentity(String alias, boolean unionQueries);

    /**
     * Select the version column.
     * @param alias Alias to use for this column
     * @param unionQueries Whether to select the version column of all unioned tables
     * @return The position of the column in the result set
     */
    int[] selectVersion(String alias, boolean unionQueries);

    /**
     * Select the column(s) for the specified field in the primary table of the query.
     * @param fieldName Name of the field
     * @param alias Alias to use for these column(s)
     * @param unionQueries Whether to select the field column(s) of all unioned queries.
     * @return The position of the columns in the result set.
     */
    int[] selectField(String fieldName, String alias, boolean unionQueries);

    /**
     * Select the columns for a mapping
     * @param mapping The mapping
     * @return The index of the columns in the select
     **/
    int[] select(JavaTypeMapping mapping);
    
    /**
     * select a new column, add to union queries, if unionQueries is true
     * @param mapping The mapping
     * @param unionQueries Whether to add to any union
     * @return The index of the columns in the select
     */
    int[] select(JavaTypeMapping mapping, boolean unionQueries);
    
    /**
     * select an expression; eg: "'Text' as alias"
     * @param expr The expression to add to the select statement 
     * @return The index of the expression in the select
     */
    int selectScalarExpression(ScalarExpression expr);
    
    /**
     * select an expression; eg: "'Text' as alias"
     * @param expr The expression to add to the select statement
     * @param unionQueries whether to apply the select in all queries unified by the union clause 
     * @return The index of the expression in the select
     */
    int selectScalarExpression(ScalarExpression expr, boolean unionQueries);
    
    /**
     * Select columns, add to union queries, if unionQueries is true
     * @param alias The alias
     * @param mapping The mapping
     * @return The index of the columns in the select
     */
    int[] select(DatastoreIdentifier alias, JavaTypeMapping mapping);
    
    /**
     * select columns, add to union queries, if unionQueries is true
     * @param alias The alias
     * @param mapping The mapping
     * @param unionQueries Whether to add to any union
     * @return The index of the column in the select
     */
    int[] select(DatastoreIdentifier alias, JavaTypeMapping mapping, boolean unionQueries);

    /**
     * add an condition to the query.
     * @param condition the Boolean expression
     */
    void andCondition(BooleanExpression condition);

    /**
     * add an condition to the query and queries involved in the union if unionQuery is true
     * @param condition the Boolean expression 
     * @param unionQueries whether to apply the condition in all queries unified by the union clause 
     */
    void andCondition(BooleanExpression condition, boolean unionQueries);

    /**
     * Method to add tables/crossJoin but no joins to they, will be output as FROM TABLE1,TABLE2,TABLE3
     * @param tableExpr table expression
     * @param unionQueries Whether to apply the alias to unions of this query.
     **/
    void crossJoin(LogicSetExpression tableExpr, boolean unionQueries);

    /**
     * Method to add tables/alias but no joins to they, will be output as FROM TABLE1,TABLE2,TABLE3
     * Checks parent expressions until reach the root expression
     * @param tableExpr table expression
     **/
    boolean hasCrossJoin(LogicSetExpression tableExpr);

    /**
     * Method to do an inner join to another table, and optionally apply it to
     * any unions for this query.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr the 
     * @param equals if the join is applied as filter, if use equals or not equals
     * @param unionQueries whether to apply the inner join in all queries unified by the union clause 
     **/
    void innerJoin(ScalarExpression expr, ScalarExpression expr2, 
                   LogicSetExpression tblExpr, boolean equals, boolean unionQueries);

    /**
     * Method to do an inner join to another table.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr The table expression for the table to apply the join
     * @param equals if the join is applied as filter, if use equals or not equals
     **/
    void innerJoin(ScalarExpression expr, ScalarExpression expr2, LogicSetExpression tblExpr, boolean equals);

    /**
     * Method to do a left outer join to another table, and optionally apply it
     * to any unions for this query.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr The table expression for the table to apply the join
     * @param equals if the join is applied as filter, if use equals or not equals
     * @param unionQueries Whether to apply to unions of this query.
     **/
    void leftOuterJoin(ScalarExpression expr, ScalarExpression expr2, 
                       LogicSetExpression tblExpr, boolean equals, boolean unionQueries);

    /**
     * Method to do a left outer join to another table.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr The table expression
     * @param equals if the join is applied as filter, if use equals or not equals
     **/
    void leftOuterJoin(ScalarExpression expr, ScalarExpression expr2, LogicSetExpression tblExpr, boolean equals);

    /**
     * Method to do a right outer join to another table, and optionally apply it
     * to any unions for this query.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr The table expression for the table to apply the join
     * @param equals if the join is applied as filter, if use equals or not equals
     * @param unionQueries Whether to apply to unions of this query.
     **/
    void rightOuterJoin(ScalarExpression expr, ScalarExpression expr2, 
                        LogicSetExpression tblExpr, boolean equals, boolean unionQueries);

    /**
     * Method to do a right outer join to another table.
     * @param expr the left hand expression
     * @param expr2 the right hand expression
     * @param tblExpr The table expression for the table to apply the join
     * @param equals if the join is applied as filter, if use equals or not equals
     **/
    void rightOuterJoin(ScalarExpression expr, ScalarExpression expr2, LogicSetExpression tblExpr, boolean equals);

    /**
     * Method to add a grouping clause to the statement.
     * Grouping clauses that are implied by the selected columns will be added automatically
     * so this provides a means to supplement them.
     * @param expr The group by expression
     */
    void addGroupingExpression(ScalarExpression expr);

    /**
     * Method to set the having clause of the statement.
     * @param expr The having expression
     */
    void setHaving(BooleanExpression expr);

    /**
     * Mutator for the ordering criteria.
     * @param exprs The expressions to order by
     * @param descending Whether each expression is ascending/descending
     **/
    void setOrdering(ScalarExpression[] exprs, boolean[] descending);

    /**
     * set the update condition(s) for the query.
     * @param exprs the Boolean expression
     */
    void setUpdates(ScalarExpression[] exprs);

    /**
     * Union two QueryExpressions <code>this</code> and <code>qe</code>.
     * Both QueryExpressions must have the same ScalarExpressions selected,
     * and they must be in the same select order. 
     * valid:
     * e.g. a) fieldA, fieldB, fieldE, fieldC
     *      b) fieldA, fieldB, fieldE, fieldC
     * invalid:
     * e.g. a) fieldA, fieldE, fieldB, fieldC      
     *      b) fieldA, fieldB, fieldE, fieldC
     *      
     * @param qe the QueryExpression
     */
    void union(QueryExpression qe);

    /**
     * add an condition to the query.
     * @param condition the Boolean expression 
     */
    void iorCondition(BooleanExpression condition);

    /**
     * add an condition to the query and queries involved in the union if unionQuery is true
     * @param condition the Boolean expression 
     * @param unionQueries whether to apply the condition in all queries unified by the union clause 
     */
    void iorCondition(BooleanExpression condition, boolean unionQueries);

    /**
     * Method to add a range constraint on any SELECT.
     * This typically will use LIMIT/OFFSET where they are supported by
     * the underlying RDBMS.
     * @param offset The offset to start from
     * @param count The number of records to return
     */
    void setRangeConstraint(long offset, long count);

    /**
     * Set this query is to be used as a as set for the Exists function.
     * example WHERE EXISTS( QUERY )
     * @param isExistsSubQuery The isExistsSubQuery to set.
     */
    void setExistsSubQuery(boolean isExistsSubQuery);

    /**
     * Accessor for the number of ScalarExpression projected.
     * @return The number of columns in the SELECT
     **/
    int getNumberOfScalarExpressions();

    /**
     * Method to convert the criteria into a delete statement text.
     * @return The StatementText
     **/
    StatementText toDeleteStatementText();

    /**
     * Method to convert the criteria into an update statement text.
     * @return The StatementText
     **/
    StatementText toUpdateStatementText();

    /**
     * Method to convert the criteria into the statement text.
     * @param lock whether to lock the instances using this statement
     * @return The StatementText
     **/
    StatementText toStatementText(boolean lock); 

    /**
     * Allows reseting the compiled expression
     */
    void reset();
}