/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import org.datanucleus.store.mapped.DatastoreIdentifier;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Interface representation of the querying capabilities of a backing store for a collection.
 * This is part of the JDOQL "expression" design that will be refactored out when the 
 * new query engine replaces it.
 */
public interface CollectionStoreQueryable
{
    /**
     * Create a subquery for the given query that selects elements, joining to the owner table. 
     * This subquery can subsequently be used in an EXISTS expression to determine whether a Collection is empty.
     * @param parentStmt Parent query statement for this subquery
     * @param ownerMapping Id mapping for the owner
     * @param ownerTableExpr Table Expression for the owner that the subquery joins to
     * @param existsTableAlias Alias for this subquery main table
     * @return Subquery returning the existence of elements
     */ 
    QueryExpression getExistsSubquery(QueryExpression parentStmt,
            JavaTypeMapping ownerMapping,
            LogicSetExpression ownerTableExpr,
            DatastoreIdentifier existsTableAlias);

    /**
     * Create a subquery for the size of the collection.
     * @param parentStmt Parent query statement for this subquery
     * @param ownerMapping Id mapping for the owner
     * @param ownerTableExpr Table Expression for the owner in the parent statement that the subquery joins to
     * @param sizeTableAlias Alias for this subquery main table
     * @return Subquery returning the size
     */
    QueryExpression getSizeSubquery(QueryExpression parentStmt,
            JavaTypeMapping ownerMapping,
            LogicSetExpression ownerTableExpr,
            DatastoreIdentifier sizeTableAlias);

    /**
     * Method used in queries when contains() has been invoked.
     * @param stmt The Query Statement
     * @param parentStmt the parent Query Statement. If there is no parent, "parentStmt" must be equals to "stmt".
     * @param ownerMapping the mapping for the owner
     * @param ownerTableExpr Table Expression for the owner
     * @param collectionTableAlias Alias for the "Collection" table. 
     * @param filteredElementType The Class Type for the filtered element
     * @param elementExpr The Expression for the element
     * @param elementTableAlias The SQL alias to assign to the element table expression
     * @param existsQuery Whether this is joining for an EXISTS query
     * @return expression to the join
     **/
    ScalarExpression joinElementsTo(QueryExpression stmt,
            QueryExpression parentStmt,
            JavaTypeMapping ownerMapping,
            LogicSetExpression ownerTableExpr,
            DatastoreIdentifier collectionTableAlias,
            Class filteredElementType,
            ScalarExpression elementExpr,
            DatastoreIdentifier elementTableAlias,
            boolean existsQuery);
}