/**********************************************************************
Copyright (c) 2009 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus.store.connection;

import java.util.ArrayList;
import java.util.List;


/**
 * Abstract implementation of a managed connection.
 */
public abstract class AbstractManagedConnection implements ManagedConnection
{
    /** The underlying Connection. */
    protected Object conn;

    /** if managed by TransactionManager can reuse several times inside the same transaction. */
    protected boolean managed = false;

    protected boolean locked = false;

    protected List<ManagedConnectionResourceListener> listeners = new ArrayList();

    /**
     * Release this connection.
     * Releasing this connection will allow this managed connection to be used one or more times
     * inside the same transaction. If this managed connection is managed by a transaction manager, 
     * release is a no-op, otherwise the physical connection is closed 
     */
    public void release()
    {
        if (!managed)
        {
            close();
        }
    }

    public void setManagedResource()
    {
        managed = true;
    }

    public void addListener(ManagedConnectionResourceListener listener)
    {
        listeners.add(listener);            
    }

    public void removeListener(ManagedConnectionResourceListener listener)
    {
        listeners.remove(listener);            
    }

    public void flush()
    {
        for (int i=0; i<listeners.size(); i++)
        {
            listeners.get(i).managedConnectionFlushed();
        }
    }

    public boolean isLocked()
    {
        return locked;
    }

    public void lock()
    {
        locked = true;
    }

    public void unlock()
    {
        locked = false;
    }

    public String toString()
    {
        // log this object, and connection associated
        return "[" + super.toString() + ", "+conn+"]";
    }
}