/**********************************************************************
Copyright (c) 2004 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
    ...
**********************************************************************/
package org.datanucleus.sco;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import org.datanucleus.StateManager;
import org.datanucleus.store.scostore.ListStore;

/**
 * An iterator for a SCO List object. Takes in the delegate and the backing
 * store, and provides iteration through the objects.
 * Only accessible from package to provide protection.
 */
public class SCOListIterator implements ListIterator
{
    private final ListIterator iter;
    private Object last=null;

    /** The index in the owner SCO. */
    private int currentIndex=-1;

    /** The owning SCO that we are really iterating. */
    private List ownerSCO;

    /**
     * Constructor taking the delegate, backing store and any start index.
     * @param sco Owner SCO 
     * @param sm State Manager of SCO List to iterate 
     * @param theDelegate The delegate list
     * @param theStore The backing store (connected to the DB)
     * @param useDelegate whether to use a delegate
     * @param startIndex The start index position
     */
    public SCOListIterator(List sco,
                           StateManager sm,
                           List theDelegate,
                           ListStore theStore,
                           boolean useDelegate,
                           int startIndex)
    {
        ownerSCO = sco;

        // Populate our entries list
        ArrayList entries = new ArrayList();

        Iterator i=null;
        if (useDelegate)
        {
            i = theDelegate.iterator();
        }
        else
        {
            if (theStore != null)
            {
                i = theStore.iterator(sm);
            }
            else
            {
                i = theDelegate.iterator();
            }
        }
        while (i.hasNext())
        {
            entries.add(i.next());
        }

        if (startIndex >= 0)
        {
            iter = entries.listIterator(startIndex);
        }
        else
        {
            iter = entries.listIterator();
        }
        currentIndex = startIndex;
    }

    public void add(Object o)
    {
        currentIndex = iter.nextIndex();

        ownerSCO.add(currentIndex, o);
        last = null;
    }

    public boolean hasNext()
    {
        return iter.hasNext();
    }

    public boolean hasPrevious()
    {
        return iter.hasPrevious();
    }

    public Object next()
    {
        currentIndex++;
        return last = iter.next();
    }

    public int nextIndex()
    {
        return iter.nextIndex();
    }

    public Object previous()
    {
        currentIndex--;
        return last = iter.previous();
    }

    public int previousIndex()
    {
        return iter.previousIndex();
    }

    public void remove()
    {
        if (last == null)
        {
            throw new IllegalStateException();
        }

        ownerSCO.remove(currentIndex);
        last = null;
        currentIndex--;
    }

    public synchronized void set(Object o)
    {
        if (last == null)
        {
            throw new IllegalStateException("No entry to replace");
        }

        ownerSCO.set(currentIndex-1, o);
        iter.set(o);

        last = o;
    }
}