/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.query.expression;

import org.datanucleus.exceptions.ClassNotResolvedException;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.query.symbol.SymbolTable;

/**
 * Expression representing a cast to a different type.
 */
public class CastExpression extends Expression
{
    /** Name of the class to cast to (may be unqualified at this stage). */
    String className;

    /**
     * Constructor for an expression representing the cast of the left expression.
     * @param left Expression being cast
     * @param className Name of the class to cast to
     */
    public CastExpression(Expression left, String className)
    {
        this.left = left;
        this.className = className;
    }

    /**
     * The name of the class to cast to.
     * @return The class to cast to.
     */
    public String getClassName()
    {
        return className;
    }

    /**
     * Method to bind the expression to the symbol table as appropriate.
     * @param symtbl Symbol table
     * @return The symbol for this expression
     */
    public Symbol bind(SymbolTable symtbl)
    {
        if (left != null)
        {
            symbol = left.bind(symtbl);
        }

        // Make sure the cast class name is fully-qualified/valid
        try
        {
            Class cls = symtbl.getSymbolResolver().resolveClass(className);
            if (cls != null && !cls.getName().equals(className))
            {
                className = cls.getName();
            }
        }
        catch (ClassNotResolvedException cnre)
        {
            // TODO Throw exception since not known ?
        }

        return symbol;
    }

    public String toString()
    {
        return "CastExpression(" + className + "){[" + left + "]}";
    }
}