/**********************************************************************
Copyright (c) 2009 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
   ...
**********************************************************************/
package org.datanucleus;

import org.datanucleus.api.ApiAdapter;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.state.StateManagerFactory;
import org.datanucleus.store.ExecutionContext;
import org.datanucleus.store.FieldValues;
import org.datanucleus.store.FieldValues2;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.Type;
import org.datanucleus.store.types.TypeManager;

/**
 * Context of execution for persistence operations
 */
public class ExecutionContextImpl implements ExecutionContext
{
    ObjectManager om;
    public ExecutionContextImpl(ObjectManager om)
    {
        this.om = om;
    }
    
    public Object findObjectUsingAID(Type pcClass, final FieldValues2 fv, boolean ignoreCache, boolean checkInheritance)
    {
        return om.findObjectUsingAID(pcClass.getType(), new FieldValues()
        {
            
            public FetchPlan getFetchPlanForLoading()
            {
                return fv.getFetchPlanForLoading();
            }
            
            public void fetchNonLoadedFields(StateManager sm)
            {
                fv.fetchNonLoadedFields(sm.getObjectProvider());
            }
            
            public void fetchFields(StateManager sm)
            {
                fv.fetchFields(sm.getObjectProvider());
            }
        }, ignoreCache, checkInheritance);
    }

    public ClassLoaderResolver getClassLoaderResolver()
    {
        return om.getClassLoaderResolver();
    }

    public MetaDataManager getMetaDataManager()
    {
        return om.getMetaDataManager();
    }

    public Transaction getTransaction()
    {
        return om.getTransaction();
    }

    public TypeManager getTypeManager()
    {
        return om.getOMFContext().getTypeManager();
    }

    public ObjectProvider findObjectProvider(Object object)
    {
        StateManager sm = om.findStateManager(object);
        if (sm==null)
        {
            return null;
        }
        return sm.getObjectProvider();
    }
    
    /**
     * @param persist persists the object if not yet persisted. 
     */
    public ObjectProvider findObjectProvider(Object object, boolean persist)
    {
        StateManager sm = om.findStateManager(object);
        if (sm == null && persist)
        {
            int objectType = StateManager.PC;
            Object object2 = om.persistObjectInternal(object, null, null, -1, objectType);
            sm = om.findStateManager(object2);
        }
        else if(sm==null)
        {
            return null;
        }
        return sm.getObjectProvider();
    }
    
    public ApiAdapter getApiAdapter()
    {
        return om.getApiAdapter();
    }
    
    public ObjectProvider findObjectProviderForEmbedded(Object value, ObjectProvider owner, AbstractMemberMetaData mmd)
    {
            StateManager embeddedSM = om.findStateManager(value);
            if (embeddedSM == null)
            {
                // Assign a StateManager to manage our embedded object
                embeddedSM = StateManagerFactory.newStateManagerForEmbedded(om, value, false);
            }
            if (embeddedSM.getEmbeddedOwners() == null || embeddedSM.getEmbeddedOwners().length == 0)
            {
                int absoluteFieldNumber = owner.getClassMetaData().getMetaDataForMember(mmd.getName()).getAbsoluteFieldNumber();
                embeddedSM.getObjectProvider().addEmbeddedOwner(owner, absoluteFieldNumber);
                embeddedSM.setPcObjectType(StateManager.EMBEDDED_PC);
            }
            return embeddedSM.getObjectProvider();
    }
    
    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd)
    {
            Class pcClass = om.getClassLoaderResolver().classForName(mmd.getClassName());
            StateManager sm = StateManagerFactory.newStateManagerForHollow( om, pcClass, null );
            if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null)
            {
                sm.initialiseForEmbedded( sm.getObject(), true );
            }
            return sm.getObjectProvider();
    }

    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, Type effectiveType)
    {
        Class pcClass = om.getClassLoaderResolver().classForName(effectiveType.getName());
        StateManager sm = StateManagerFactory.newStateManagerForHollow( om, pcClass, null );
        if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null)
        {
            sm.initialiseForEmbedded( sm.getObject(), true );
        }
        return sm.getObjectProvider();
    }

    public ObjectProvider newObjectProviderForMember(AbstractMemberMetaData mmd, AbstractClassMetaData effectiveTypeCmd)
    {
        Class pcClass = om.getClassLoaderResolver().classForName(effectiveTypeCmd.getFullClassName());
        StateManager sm = StateManagerFactory.newStateManagerForHollow( om, pcClass, null );
        if (mmd.isEmbedded() || mmd.getEmbeddedMetaData()!=null || effectiveTypeCmd.isEmbeddedOnly())
        {
            sm.initialiseForEmbedded( sm.getObject(), true );
        }
        return sm.getObjectProvider();
    }

    public void deleteObjectInternal(Object pc)
    {
        om.deleteObjectInternal(pc);
    }
    
    /**
     * Method to persist the passed object (internally).
     * @param pc The object
     * @param ownerSM StateManager of the owner when embedded
     * @param ownerFieldNum Field number in the owner where this is embedded (or -1 if not embedded)
     * @param objectType Type of object (see org.datanucleus.StateManager, e.g StateManager.PC)
     * @return The persisted object
     */
    public Object persistObjectInternal(Object pc, ObjectProvider ownerSM, int ownerFieldNum, int objectType)
    {
        if (ownerSM != null)
        {
            StateManager sm = om.findStateManager(ownerSM.getObject());
            return om.persistObjectInternal(pc, null, sm, ownerFieldNum, objectType);
        }
        else
        {
            return om.persistObjectInternal(pc, null, null, ownerFieldNum, objectType);
        }
    }
    
    public boolean isClosed()
    {
        return om.isClosed();
    }
}

