/*
 * This file is part of CycloneDX Core (Java).
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) OWASP Foundation. All Rights Reserved.
 */
package org.cyclonedx.util.deserializer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import org.cyclonedx.model.OrganizationalContact;
import org.cyclonedx.model.OrganizationalEntity;
import org.cyclonedx.model.Property;
import org.cyclonedx.model.vulnerability.Vulnerability;
import org.cyclonedx.util.TimestampUtils;
import org.cyclonedx.util.ToolsJsonParser;


public class VulnerabilityDeserializer
    extends JsonDeserializer<List<Vulnerability>> {

  private final PropertiesDeserializer propertiesDeserializer = new PropertiesDeserializer();
  private final AffectsDeserializer affectsDeserializer = new AffectsDeserializer();

  @Override
  public List<Vulnerability> deserialize(final JsonParser parser, final DeserializationContext context) {
    try {
      JsonNode node = parser.getCodec().readTree(parser);
      return parseVulnerabilities(node.has("vulnerability") ? node.get("vulnerability") : node, parser, context);
    } catch (Exception e) {
      return null;
    }
  }

  private List<Vulnerability> parseVulnerabilities(JsonNode node, JsonParser jsonParser, DeserializationContext ctxt) throws IOException {
    ObjectMapper mapper = getMapper(jsonParser);
    List<Vulnerability> vulnerabilities = new ArrayList<>();
    ArrayNode nodes = DeserializerUtils.getArrayNode(node, mapper);
    for (JsonNode vulnerabilityNode : nodes) {
      vulnerabilities.add(parseVulnerability(vulnerabilityNode, jsonParser, ctxt, mapper));
    }
    return vulnerabilities;
  }

  private Vulnerability parseVulnerability(JsonNode node, JsonParser jsonParser, DeserializationContext ctxt, ObjectMapper mapper) throws IOException {
    JsonParser vulnerabilityParser = node.traverse(jsonParser.getCodec());
    vulnerabilityParser.nextToken();

    Vulnerability vulnerability = new Vulnerability();

    if (node.has("bom-ref")) {
      vulnerability.setBomRef(node.get("bom-ref").asText());
    }

    if (node.has("id")) {
      vulnerability.setId(node.get("id").asText());
    }

    if (node.has("description")) {
      vulnerability.setDescription(node.get("description").asText());
    }

    if (node.has("detail")) {
      vulnerability.setDetail(node.get("detail").asText());
    }

    if (node.has("recommendation")) {
      vulnerability.setRecommendation(node.get("recommendation").asText());
    }

    if (node.has("source")) {
      Vulnerability.Source source = mapper.convertValue(node.get("source"), Vulnerability.Source.class);
      vulnerability.setSource(source);
    }

    if (node.has("references")) {
      JsonNode referencesNode = node.get("references");
      if (referencesNode.isArray()) {
        List<Vulnerability.Reference> references = mapper.convertValue(node.get("references"),
                new TypeReference<List<Vulnerability.Reference>>() {});
        vulnerability.setReferences(references);
      } else if (referencesNode.has("reference")) {
        JsonNode referenceNode = referencesNode.get("reference");
        if (referenceNode.isArray()) {
          List<Vulnerability.Reference> references = mapper.convertValue(referenceNode,
                  new TypeReference<List<Vulnerability.Reference>>() {});
          vulnerability.setReferences(references);
        } else {
          vulnerability.addReference(mapper.convertValue(referenceNode, Vulnerability.Reference.class));
        }
      }
    }

    if (node.has("ratings")) {
      JsonNode ratingsNode = node.get("ratings");
      if (ratingsNode.isArray()) {
        List<Vulnerability.Rating> ratings = mapper.convertValue(node.get("ratings"), new TypeReference<List<Vulnerability.Rating>>() {
        });
        vulnerability.setRatings(ratings);
      } else if (ratingsNode.has("rating")) {
        JsonNode ratingNode = ratingsNode.get("rating");
        if (ratingNode.isArray()) {
          List<Vulnerability.Rating> ratings = mapper.convertValue(ratingNode, new TypeReference<List<Vulnerability.Rating>>() {
          });
          vulnerability.setRatings(ratings);
        }
        else {
          vulnerability.addRating(mapper.convertValue(ratingNode, Vulnerability.Rating.class));
        }
      }
    }

    if (node.has("cwes")) {
      JsonNode cwesNode = node.get("cwes");
      if (cwesNode.isArray()) {
        List<Integer> codes = mapper.convertValue(node.get("cwes"), new TypeReference<List<Integer>>() {});
        vulnerability.setCwes(codes);
      } else if (cwesNode.has("cwe")) {
        JsonNode cweNode = cwesNode.get("cwe");
        if (cweNode.isArray()) {
          List<Integer> codes = mapper.convertValue(cweNode, new TypeReference<List<Integer>>() {
          });
          vulnerability.setCwes(codes);
        } else {
          vulnerability.addCwe(cweNode.asInt());
        }
      }
    }

    if (node.has("advisories")) {
      JsonNode advisoriesNode = node.get("advisories");
      if (advisoriesNode.isArray()) {
        List<Vulnerability.Advisory> advisories = mapper.convertValue(node.get("advisories"),
                new TypeReference<List<Vulnerability.Advisory>>() {});
        vulnerability.setAdvisories(advisories);
      } else if (advisoriesNode.has("advisory")) {
        JsonNode advisoryNode = advisoriesNode.get("advisory");
        if (advisoryNode.isArray()) {
          List<Vulnerability.Advisory> advisories = mapper.convertValue(advisoryNode,
                  new TypeReference<List<Vulnerability.Advisory>>() {});
          vulnerability.setAdvisories(advisories);
        } else {
          vulnerability.addAdvisory(mapper.convertValue(advisoryNode, Vulnerability.Advisory.class));
        }
      }
    }

    if (node.has("created")) {
      vulnerability.setCreated(TimestampUtils.parseTimestamp(node.get("created").textValue()));
    }

    if (node.has("published")) {
      vulnerability.setPublished(TimestampUtils.parseTimestamp(node.get("published").textValue()));
    }

    if (node.has("updated")) {
      vulnerability.setUpdated(TimestampUtils.parseTimestamp(node.get("updated").textValue()));
    }

    if (node.has("rejected")) {
      vulnerability.setRejected(TimestampUtils.parseTimestamp(node.get("rejected").textValue()));
    }

    if (node.has("credits")) {
      parseCredits(node.get("credits"), vulnerability, mapper);
    }

    if (node.has("analysis")) {
      parseAnalysis(node.get("analysis"), vulnerability, mapper);
    }

    if (node.has("affects")) {
      JsonParser affectsParser = node.get("affects").traverse(jsonParser.getCodec());
      affectsParser.nextToken();
      List<Vulnerability.Affect> affects = affectsDeserializer.deserialize(affectsParser, ctxt);
      vulnerability.setAffects(affects);
    }

    if (node.has("properties")) {
      JsonParser propertiesParser = node.get("properties").traverse(jsonParser.getCodec());
      propertiesParser.nextToken();
      List<Property> properties = propertiesDeserializer.deserialize(propertiesParser, ctxt);
      vulnerability.setProperties(properties);
    }

    if (node.has("tools")) {
      ToolsJsonParser toolsParser = new ToolsJsonParser(node, jsonParser, ctxt);
      vulnerability.setTools(toolsParser.getTools());
      vulnerability.setToolChoice(toolsParser.getToolInformation());
    }

    return vulnerability;
  }

  private void parseAnalysis(JsonNode analysisNode, Vulnerability vulnerability, ObjectMapper mapper) {
    if (analysisNode != null) {
      Vulnerability.Analysis analysis = new Vulnerability.Analysis();
      if (analysisNode.has("state")) {
        analysis.setState(mapper.convertValue(analysisNode.get("state"), Vulnerability.Analysis.State.class));
      }
      if (analysisNode.has("justification")) {
        analysis.setJustification(mapper.convertValue(analysisNode.get("justification"), Vulnerability.Analysis.Justification.class));
      }
      if (analysisNode.has("detail")) {
        analysis.setDetail(analysisNode.get("detail").asText());
      }
      if (analysisNode.has("firstIssued")) {
        analysis.setFirstIssued(TimestampUtils.parseTimestamp(analysisNode.get("firstIssued").textValue()));
      }
      if (analysisNode.has("lastUpdated")) {
        analysis.setLastUpdated(TimestampUtils.parseTimestamp(analysisNode.get("lastUpdated").textValue()));
      }
      if (analysisNode.has("response")) {
        JsonNode responsesNode = analysisNode.get("response");
        if (responsesNode.isArray()) {
          List<Vulnerability.Analysis.Response> responses = mapper.convertValue(responsesNode,
                  new TypeReference<List<Vulnerability.Analysis.Response>>() {});
          analysis.setResponses(responses);
        }
      } else if (analysisNode.has("responses")) {
        JsonNode responsesNode = analysisNode.get("responses");
        if (responsesNode.has("response")) {
          JsonNode responseNode = responsesNode.get("response");
          if (responseNode.isArray()) {
            List<Vulnerability.Analysis.Response> responses = mapper.convertValue(responseNode,
                    new TypeReference<List<Vulnerability.Analysis.Response>>() {});
            analysis.setResponses(responses);
          }
          else if (responseNode.isTextual()) {
            Vulnerability.Analysis.Response response =
                Vulnerability.Analysis.Response.fromString(responseNode.asText());
            analysis.addResponse(response);
          }
        }
      }
      vulnerability.setAnalysis(analysis);
    }
  }

  private void parseCredits(JsonNode creditsNode, Vulnerability vulnerability, ObjectMapper mapper) {
    Vulnerability.Credits credits = new Vulnerability.Credits();
    if (creditsNode.has("organizations")) {
      parseOrganizations(creditsNode.get("organizations"), credits, mapper);
    }
    if (creditsNode.has("individuals")) {
      parseIndividuals(creditsNode.get("individuals"), credits, mapper);
    }
    vulnerability.setCredits(credits);
  }

  private void parseOrganizations(JsonNode organizationsNode, Vulnerability.Credits credits, ObjectMapper mapper) {
    if (organizationsNode != null) {
      if (organizationsNode.isArray()) {
        List<OrganizationalEntity> organizations = mapper.convertValue(organizationsNode,
                new TypeReference<List<OrganizationalEntity>>() {});
        credits.setOrganizations(organizations);
      } else if (organizationsNode.isObject()) {
        OrganizationalEntity organization = mapper.convertValue(organizationsNode.get("organization"), OrganizationalEntity.class);
        credits.addOrganization(organization);
      }
    }
  }

  private void parseIndividuals(JsonNode individualsNode, Vulnerability.Credits credits, ObjectMapper mapper) {
    if (individualsNode != null) {
      if (individualsNode.isArray()) {
        List<OrganizationalContact> individuals = mapper.convertValue(individualsNode, new TypeReference<List<OrganizationalContact>>() {});
        credits.setIndividuals(individuals);
      } else if (individualsNode.isObject()) {
        OrganizationalContact individual = mapper.convertValue(individualsNode.get("individual"), OrganizationalContact.class);
        credits.addIndividual(individual);
      }
    }
  }

  private ObjectMapper getMapper(JsonParser jsonParser) {
    if (jsonParser.getCodec() instanceof ObjectMapper) {
      return (ObjectMapper) jsonParser.getCodec();
    } else {
      return new ObjectMapper();
    }
  }
}
