/*
 * Decompiled with CFR 0.152.
 */
package org.csanchez.jenkins.plugins.kubernetes;

import com.google.common.base.Objects;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import hudson.Extension;
import hudson.XmlFile;
import hudson.model.AsyncPeriodicWork;
import hudson.model.PeriodicWork;
import hudson.model.Saveable;
import hudson.model.TaskListener;
import hudson.model.listeners.SaveableListener;
import io.fabric8.kubernetes.client.HttpClientAware;
import io.fabric8.kubernetes.client.KubernetesClient;
import java.io.IOException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateEncodingException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import jenkins.model.Jenkins;
import okhttp3.Dispatcher;
import okhttp3.OkHttpClient;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesCloud;
import org.csanchez.jenkins.plugins.kubernetes.KubernetesFactoryAdapter;
import org.kohsuke.accmod.Restricted;
import org.kohsuke.accmod.restrictions.NoExternalUse;

public class KubernetesClientProvider {
    private static final Logger LOGGER = Logger.getLogger(KubernetesClientProvider.class.getName());
    private static final Integer CACHE_SIZE = Integer.getInteger(KubernetesClientProvider.class.getPackage().getName() + ".clients.cacheSize", 10);
    private static final Long EXPIRED_CLIENTS_PURGE_TIME = Long.getLong(KubernetesClientProvider.class.getPackage().getName() + ".clients.expiredClientsPurgeTime", 3600L);
    private static final Long EXPIRED_CLIENTS_PURGE_PERIOD = Long.getLong(KubernetesClientProvider.class.getPackage().getName() + ".clients.expiredClientsPurgePeriod", EXPIRED_CLIENTS_PURGE_TIME / 2L);
    private static final Integer CACHE_EXPIRATION = Integer.getInteger(KubernetesClientProvider.class.getPackage().getName() + ".clients.cacheExpiration", 86400);
    private static final Queue<Client> expiredClients = new ConcurrentLinkedQueue<Client>();
    private static final Cache<String, Client> clients = CacheBuilder.newBuilder().maximumSize((long)CACHE_SIZE.intValue()).expireAfterWrite((long)CACHE_EXPIRATION.intValue(), TimeUnit.SECONDS).removalListener(rl -> {
        LOGGER.log(Level.FINE, "{0} cache : Removing entry for {1}", new Object[]{KubernetesClient.class.getSimpleName(), rl.getKey()});
        Client client = (Client)rl.getValue();
        if (client != null) {
            client.expired = Instant.now();
            expiredClients.add(client);
        }
    }).build();
    private static volatile int runningCallsCount;
    private static volatile int queuedCallsCount;

    private KubernetesClientProvider() {
    }

    static KubernetesClient createClient(KubernetesCloud cloud) throws NoSuchAlgorithmException, UnrecoverableKeyException, KeyStoreException, IOException, CertificateEncodingException {
        String displayName = cloud.getDisplayName();
        Client c = (Client)clients.getIfPresent((Object)displayName);
        if (c == null) {
            KubernetesClient client = new KubernetesFactoryAdapter(cloud.getServerUrl(), cloud.getNamespace(), cloud.getServerCertificate(), cloud.getCredentialsId(), cloud.isSkipTlsVerify(), cloud.getConnectTimeout(), cloud.getReadTimeout(), cloud.getMaxRequestsPerHost()).createClient();
            clients.put((Object)displayName, (Object)new Client(KubernetesClientProvider.getValidity(cloud), client));
            LOGGER.log(Level.INFO, "Created new Kubernetes client: {0} {1}", new Object[]{displayName, client});
            return client;
        }
        return c.getClient();
    }

    private static int getValidity(KubernetesCloud cloud) {
        return Objects.hashCode((Object[])new Object[]{cloud.getServerUrl(), cloud.getNamespace(), cloud.getServerCertificate(), cloud.getCredentialsId(), cloud.isSkipTlsVerify(), cloud.getConnectTimeout(), cloud.getReadTimeout(), cloud.getMaxRequestsPerHostStr()});
    }

    @Restricted(value={NoExternalUse.class})
    public static boolean closeExpiredClients() {
        boolean b = false;
        if (expiredClients.isEmpty()) {
            return b;
        }
        LOGGER.log(Level.FINE, "Closing expired clients: ({0}) {1}", new Object[]{expiredClients.size(), expiredClients});
        if (expiredClients.size() > 10) {
            LOGGER.log(Level.WARNING, "High number of expired clients, may cause memory leaks: ({0}) {1}", new Object[]{expiredClients.size(), expiredClients});
        }
        Iterator it = expiredClients.iterator();
        while (it.hasNext()) {
            Client expiredClient = (Client)it.next();
            if (Instant.now().minus(EXPIRED_CLIENTS_PURGE_TIME, ChronoUnit.SECONDS).isBefore(expiredClient.getExpired())) break;
            KubernetesClient client = expiredClient.client;
            if (client instanceof HttpClientAware) {
                if (!KubernetesClientProvider.gracefulClose(client, ((HttpClientAware)client).getHttpClient())) continue;
                it.remove();
                b = true;
                continue;
            }
            LOGGER.log(Level.WARNING, "{0} is not {1}, forcing close", new Object[]{client.toString(), HttpClientAware.class.getSimpleName()});
            client.close();
            it.remove();
            b = true;
        }
        return b;
    }

    @Restricted(value={NoExternalUse.class})
    public static boolean gracefulClose(KubernetesClient client, OkHttpClient httpClient) {
        Dispatcher dispatcher = httpClient.dispatcher();
        int runningCallsCount = dispatcher.runningCallsCount();
        int queuedCallsCount = dispatcher.queuedCallsCount();
        if (runningCallsCount == 0 && queuedCallsCount == 0) {
            LOGGER.log(Level.INFO, "Closing {0}", client.toString());
            client.close();
            return true;
        }
        LOGGER.log(Level.INFO, "Not closing {0}: there are still running ({1}) or queued ({2}) calls", new Object[]{client.toString(), runningCallsCount, queuedCallsCount});
        return false;
    }

    public static int getRunningCallsCount() {
        return runningCallsCount;
    }

    public static int getQueuedCallsCount() {
        return queuedCallsCount;
    }

    @Restricted(value={NoExternalUse.class})
    public static void invalidate(String displayName) {
        clients.invalidate((Object)displayName);
    }

    @Extension
    public static class UpdateConnectionCount
    extends PeriodicWork {
        public long getRecurrencePeriod() {
            return TimeUnit.SECONDS.toMillis(5L);
        }

        protected void doRun() {
            int runningCallsCount = 0;
            int queuedCallsCount = 0;
            for (Client client : clients.asMap().values()) {
                KubernetesClient kClient = client.getClient();
                if (!(kClient instanceof HttpClientAware)) continue;
                OkHttpClient httpClient = ((HttpClientAware)kClient).getHttpClient();
                Dispatcher dispatcher = httpClient.dispatcher();
                runningCallsCount += dispatcher.runningCallsCount();
                queuedCallsCount += dispatcher.queuedCallsCount();
            }
            runningCallsCount = runningCallsCount;
            queuedCallsCount = queuedCallsCount;
        }
    }

    @Extension
    public static class SaveableListenerImpl
    extends SaveableListener {
        public void onChange(Saveable o, XmlFile file) {
            if (o instanceof Jenkins) {
                Jenkins jenkins = (Jenkins)o;
                HashSet cloudDisplayNames = new HashSet(clients.asMap().keySet());
                for (KubernetesCloud cloud : jenkins.clouds.getAll(KubernetesCloud.class)) {
                    String displayName = cloud.getDisplayName();
                    Client client = (Client)clients.getIfPresent((Object)displayName);
                    if (client != null && client.getValidity() == KubernetesClientProvider.getValidity(cloud)) {
                        cloudDisplayNames.remove(displayName);
                        continue;
                    }
                    LOGGER.log(Level.INFO, "Invalidating Kubernetes client: {0} {1}", new Object[]{displayName, client});
                }
                for (String displayName : cloudDisplayNames) {
                    LOGGER.log(Level.INFO, "Invalidating Kubernetes client: {0}", displayName);
                    KubernetesClientProvider.invalidate(displayName);
                }
            }
            super.onChange(o, file);
        }
    }

    @Extension
    public static class PurgeExpiredKubernetesClients
    extends AsyncPeriodicWork {
        public PurgeExpiredKubernetesClients() {
            super("Purge expired KubernetesClients");
        }

        public long getRecurrencePeriod() {
            return TimeUnit.SECONDS.toMillis(EXPIRED_CLIENTS_PURGE_PERIOD);
        }

        protected Level getNormalLoggingLevel() {
            return Level.FINEST;
        }

        protected void execute(TaskListener listener) {
            KubernetesClientProvider.closeExpiredClients();
        }
    }

    private static class Client {
        private final KubernetesClient client;
        private final int validity;
        private Instant expired;

        public Client(int validity, KubernetesClient client) {
            this.client = client;
            this.validity = validity;
        }

        public KubernetesClient getClient() {
            return this.client;
        }

        public int getValidity() {
            return this.validity;
        }

        public Instant getExpired() {
            return this.expired;
        }
    }
}

