// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.tools

import java.nio.file.{Path => JavaPath}
import org.apache.hadoop.fs.{Path => HadoopPath}
import org.cert.netsa.mothra.packer.CorePacker

class PackerOptions extends ToolOptions("mothra-packer") {

  note("Required options:\n")

  var incomingDir: HadoopPath = _
  opt[HadoopPath]("incoming-dir").valueName("DIR").required()
    .foreach(incomingDir = _)
    .text("HDFS directory to watch for incoming files.")
    .validate(checkHadoopDirExists("Option --incoming-dir"))

  var outgoingDir: HadoopPath = _
  opt[HadoopPath]("outgoing-dir").valueName("DIR").required()
    .foreach(outgoingDir = _)
    .text("HDFS directory to store packed file repository.")
    .validate(checkHadoopDirExists("Option --outgoing-dir"))

  var workDir: JavaPath = _
  opt[JavaPath]("work-dir").valueName("DIR").required()
    .foreach(workDir = _)
    .text("Local directory to store files while packing.")
    .validate(checkJavaDirExists("Option --work-dir"))

  var packingLogicFile: HadoopPath = _
  opt[HadoopPath]("packing-logic").valueName("FILE").required()
    .foreach(packingLogicFile = _)
    .text("HDFS file containing packing configuration.")
    .validate(checkHadoopFileExists("Option --packing-logic"))

  note("\nOptions:\n")

  help("help")
    .text("Print this message and exit.")

  version("version")
    .text(s"Print the version number of $toolName and exit.")

  note("")

  var oneShot: Boolean = false
  opt[Unit]("one-shot")
    .foreach(_ => oneShot = true)
    .text("Process INCOMING-DIR once and then exit.")
  opt[Unit]("oneshot").hidden()
    .foreach(_ => oneShot = true)

  var archiveDir: Option[HadoopPath] = None
  opt[HadoopPath]("archive-dir").valueName("DIR")
    .foreach(d => archiveDir = Some(d))
    .text("Archive working files to HDFS after use.")
    .validate(checkHadoopDirExists("Option --archive-dir"))

  var maxPackJobs: Option[Int] = None
  opt[Int]("max-pack-jobs").valueName("N")
    .foreach(n => maxPackJobs = Some(n))
    .text(s"Process N input files simultaneously. (Def. 1)")
    .validate(checkPos("Option --max-pack-jobs"))

  var packAttempts: Option[Int] = None
  opt[Int]("pack-attempts").valueName("N")
    .foreach(n => packAttempts = Some(n))
    .text(s"Try N times to pack each input file. (Def. 3)")
    .validate(checkPos("Option --pack-attempts"))

  var pollingInterval: Option[Int] = None
  opt[Int]("polling-interval").valueName("N")
    .foreach(n => pollingInterval = Some(n))
    .text(s"Seconds between scans for incoming files. (Def. 30)")
    .validate(checkPos("Option --polling-interval"))

  note("\nOptions that control the format of packed files:\n")

  var compression: Option[String] = None
  opt[String]("compression").valueName("CODEC")
    .foreach(c => compression = Some(parseCompression(c)))
    .text(s"Compress packed files using this method.${
      if ( CorePacker.DEFAULT_COMPRESSION == "" ) ""
        else s" (Def. ${CorePacker.DEFAULT_COMPRESSION})"}")
    .validate(checkCompression("compression"))

  var hoursPerFile: Option[Int] = None
  opt[Int]("hours-per-file").valueName("N")
    .foreach(hours => hoursPerFile = Some(hours))
    .text(s"Store N hours (1-24) in each output file. (Def. ${CorePacker.DEFAULT_HOURS_PER_FILE})")
    .validate(checkRange("Option --hours-per-file", 1, 24))

  note("\nOptions that control moving packed files to OUTGOING-DIR:\n")

  var workDirCheckInterval: Option[Int] = None
  opt[Int]("check-interval").valueName("N")
    .foreach(secs => workDirCheckInterval = Some(secs))
    .text(s"Seconds between scans for complete files. (Def. 60)")
    .validate(checkPos("Option --check-interval"))

  var workDirMaximumAge: Option[Int] = None
  opt[Int]("maximum-age").valueName("N")
    .foreach(secs => workDirMaximumAge = Some(secs))
    .text(s"Always move files older than N seconds. (Def. ${60*60})")
    .validate(checkPos("Option --maximum-age"))

  var workDirMaximumSize: Option[Long] = None
  opt[Long]("maximum-size").valueName("N")
    .foreach(size => workDirMaximumSize = Some(size))
    .text(s"Always move files larger than N bytes. (Def. ${100*0x100000})")
    .validate(checkPos("Option --maximum-size"))

  var workDirMinimumAge: Option[Int] = None
  opt[Int]("minimum-age").valueName("N")
    .foreach(secs => workDirMinimumAge = Some(secs))
    .text(s"Don't move files younger than N seconds. (Def. ${60*5})")
    .validate(checkPos("Option --minimum-age"))

  var workDirMinimumSize: Option[Long] = None
  opt[Long]("minimum-size").valueName("N")
    .foreach(size => workDirMinimumSize = Some(size))
    .text(s"Don't move files smaller than N bytes. (Def. ${64*0x100000})")
    .validate(checkPos("Option --minimum-size"))

  var fileCacheSize: Option[Int] = None
  opt[Int]("file-cache-size").valueName("N")
    .foreach(n => fileCacheSize = Some(n))
    .text(s"Keep up to N >= ${CorePacker.MINIMUM_FILE_CACHE_SIZE} files open for writing. (Def. ${CorePacker.DEFAULT_FILE_CACHE_SIZE})")
    .validate(checkGE("Option --file-cache-size", CorePacker.MINIMUM_FILE_CACHE_SIZE))

  var numMoveThreads: Option[Int] = None
  opt[Int]("num-move-threads").valueName("N")
    .foreach(n => numMoveThreads = Some(n))
    .text(s"Use N threads to move files to OUTGOING-DIR. (Def. ${CorePacker.DEFAULT_NUM_MOVE_THREADS})")
    .validate(checkPos("Option --num-move-threads"))

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
