// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import java.nio.file.{Path => JPath}
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.{Path => HPath}
import org.apache.hadoop.io.compress.CompressionCodec


/**
  * PackerConfig holds the directories and compression method specified when
  * the packer was started.
  *
  * @param rootDir The root directory of the record repository
  * @param workDir The temporary directory to use as files are created.
  * @param archiveDir If not None, the directory in which to move temporary
  * files after their contents have been copied into the `rootDir`.
  * @param compressCodec The compression to use for output files.
  */
private[mothra] case class PackerConfig(
  rootDir:       HPath,
  workDir:       JPath,
  archiveDir:    Option[HPath] = None,
  compressCodec: Option[CompressionCodec] = None)(
  implicit hadoopConf: Configuration)
{
  /**
    * A file's compression suffix for the specified `compressCodec`.
    */
  val compressSuffix =
    compressCodec.map {c => c.getDefaultExtension}.getOrElse("")

  /**
    * The FileSystem of the `rootDir`.
    */
  lazy val rootDirFileSystem = rootDir.getFileSystem(hadoopConf)

  /**
    * The FileSystem of the `archiveDir` or `null` if `archiveDir` is `None`.
    */
  lazy val archiveDirFileSystem =
    archiveDir.map { _.getFileSystem(hadoopConf) }.orNull;

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
