// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.mothra.packer

import org.cert.netsa.io.ipfix.Record


/**
  * A [[org.cert.netsa.io.ipfix.Record Record]] and the information
  * required to generate the file name where the Record will be stored
  * in a Mothra repository.
  *
  * @param record The record to write to the file.
  * @param relativePackedPath The relative path of the file to return; does
  * not include time prefix or filename suffix, but does include a version
  * number
  * @param startTime The date+time to use for time-based parts of file path
  * @param observationDomain The domain used for all IPFIX Messages in the
  * file; this is needed in case a new file must created and opened.
  */
case class PackableRecord(
  record:               Record,
  relativePackedPath:   String,
  startTime:            RecordStartTime,
  observationDomain:    Int)

/** A PackableRecord factory. */
object PackableRecord {
  /** The observationDomain to use if none is supplied. */
  private[this] val defaultDomain = 0

  /** Creates a new PackableRecord with the `startTime` based on a starting time
    * field in the `record`.  The supplied `observationDomain` is used if
    * non-empty, else a domain of 0 is used. */
  def apply(
    record: Record,
    relativePackedPath: String,
    observationDomain: Option[Int] = None): PackableRecord =
  {
    new PackableRecord(
      record, relativePackedPath, StartTimeCache.ofRecord(record),
      observationDomain.getOrElse(defaultDomain))
  }

  /** Creates a new PackableRecord with a `startTime` based on a starting time
    * field in the `record`. */
  def apply(
    record: Record,
    relativePackedPath: String,
    observationDomain: Int): PackableRecord =
  {
    new PackableRecord(
      record, relativePackedPath, StartTimeCache.ofRecord(record),
      observationDomain)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
