/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.core;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.primitives.UnsignedBytes;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;
import javax.annotation.Nullable;
import org.bitcoin.NativeSecp256k1;
import org.bitcoin.NativeSecp256k1Util;
import org.bitcoin.Secp256k1Context;
import org.bitcoinj.core.Address;
import org.bitcoinj.core.DumpedPrivateKey;
import org.bitcoinj.core.LegacyAddress;
import org.bitcoinj.core.NetworkParameters;
import org.bitcoinj.core.SegwitAddress;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.SignatureDecodeException;
import org.bitcoinj.core.Utils;
import org.bitcoinj.core.VarInt;
import org.bitcoinj.crypto.EncryptableItem;
import org.bitcoinj.crypto.EncryptedData;
import org.bitcoinj.crypto.KeyCrypter;
import org.bitcoinj.crypto.KeyCrypterException;
import org.bitcoinj.crypto.LazyECPoint;
import org.bitcoinj.crypto.LinuxSecureRandom;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.script.Script;
import org.bitcoinj.wallet.Protos;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1TaggedObject;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequenceGenerator;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.Digest;
import org.bouncycastle.crypto.KeyGenerationParameters;
import org.bouncycastle.crypto.digests.SHA256Digest;
import org.bouncycastle.crypto.ec.CustomNamedCurves;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.KeyParameter;
import org.bouncycastle.crypto.signers.DSAKCalculator;
import org.bouncycastle.crypto.signers.ECDSASigner;
import org.bouncycastle.crypto.signers.HMacDSAKCalculator;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.FixedPointUtil;
import org.bouncycastle.math.ec.custom.sec.SecP256K1Curve;
import org.bouncycastle.util.Properties;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ECKey
implements EncryptableItem {
    private static final Logger log = LoggerFactory.getLogger(ECKey.class);
    public static final Comparator<ECKey> AGE_COMPARATOR = new Comparator<ECKey>(){

        @Override
        public int compare(ECKey k1, ECKey k2) {
            if (k1.creationTimeSeconds == k2.creationTimeSeconds) {
                return 0;
            }
            return k1.creationTimeSeconds > k2.creationTimeSeconds ? 1 : -1;
        }
    };
    public static final Comparator<ECKey> PUBKEY_COMPARATOR = new Comparator<ECKey>(){
        private Comparator<byte[]> comparator = UnsignedBytes.lexicographicalComparator();

        @Override
        public int compare(ECKey k1, ECKey k2) {
            return this.comparator.compare(k1.getPubKey(), k2.getPubKey());
        }
    };
    private static final X9ECParameters CURVE_PARAMS = CustomNamedCurves.getByName((String)"secp256k1");
    public static final ECDomainParameters CURVE;
    public static final BigInteger HALF_CURVE_ORDER;
    private static final SecureRandom secureRandom;
    @Nullable
    protected final BigInteger priv;
    protected final LazyECPoint pub;
    protected long creationTimeSeconds;
    protected KeyCrypter keyCrypter;
    protected EncryptedData encryptedPrivateKey;
    private byte[] pubKeyHash;
    @VisibleForTesting
    public static boolean FAKE_SIGNATURES;
    private static final String BITCOIN_SIGNED_MESSAGE_HEADER = "Bitcoin Signed Message:\n";
    private static final byte[] BITCOIN_SIGNED_MESSAGE_HEADER_BYTES;

    public ECKey() {
        this(secureRandom);
    }

    public ECKey(SecureRandom secureRandom) {
        ECKeyPairGenerator generator = new ECKeyPairGenerator();
        ECKeyGenerationParameters keygenParams = new ECKeyGenerationParameters(CURVE, secureRandom);
        generator.init((KeyGenerationParameters)keygenParams);
        AsymmetricCipherKeyPair keypair = generator.generateKeyPair();
        ECPrivateKeyParameters privParams = (ECPrivateKeyParameters)keypair.getPrivate();
        ECPublicKeyParameters pubParams = (ECPublicKeyParameters)keypair.getPublic();
        this.priv = privParams.getD();
        this.pub = new LazyECPoint(pubParams.getQ(), true);
        this.creationTimeSeconds = Utils.currentTimeSeconds();
    }

    protected ECKey(@Nullable BigInteger priv, ECPoint pub, boolean compressed) {
        this(priv, new LazyECPoint((ECPoint)Preconditions.checkNotNull((Object)pub), compressed));
    }

    protected ECKey(@Nullable BigInteger priv, LazyECPoint pub) {
        if (priv != null) {
            Preconditions.checkArgument((priv.bitLength() <= 256 ? 1 : 0) != 0, (String)"private key exceeds 32 bytes: %s bits", (int)priv.bitLength());
            Preconditions.checkArgument((!priv.equals(BigInteger.ZERO) ? 1 : 0) != 0);
            Preconditions.checkArgument((!priv.equals(BigInteger.ONE) ? 1 : 0) != 0);
        }
        this.priv = priv;
        this.pub = (LazyECPoint)Preconditions.checkNotNull((Object)pub);
    }

    @Deprecated
    public static LazyECPoint compressPoint(LazyECPoint point) {
        return point.compress();
    }

    @Deprecated
    public static LazyECPoint decompressPoint(LazyECPoint point) {
        return point.decompress();
    }

    public static ECKey fromASN1(byte[] asn1privkey) {
        return ECKey.extractKeyFromASN1(asn1privkey);
    }

    public static ECKey fromPrivate(BigInteger privKey) {
        return ECKey.fromPrivate(privKey, true);
    }

    public static ECKey fromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return new ECKey(privKey, new LazyECPoint(point, compressed));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes) {
        return ECKey.fromPrivate(new BigInteger(1, privKeyBytes));
    }

    public static ECKey fromPrivate(byte[] privKeyBytes, boolean compressed) {
        return ECKey.fromPrivate(new BigInteger(1, privKeyBytes), compressed);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(BigInteger priv, ECPoint pub, boolean compressed) {
        return new ECKey(priv, pub, compressed);
    }

    public static ECKey fromPrivateAndPrecalculatedPublic(byte[] priv, byte[] pub) {
        Preconditions.checkNotNull((Object)priv);
        Preconditions.checkNotNull((Object)pub);
        return new ECKey(new BigInteger(1, priv), new LazyECPoint(CURVE.getCurve(), pub));
    }

    public static ECKey fromPublicOnly(ECPoint pub, boolean compressed) {
        return new ECKey(null, pub, compressed);
    }

    public static ECKey fromPublicOnly(byte[] pub) {
        return new ECKey(null, new LazyECPoint(CURVE.getCurve(), pub));
    }

    public static ECKey fromPublicOnly(ECKey key) {
        return ECKey.fromPublicOnly(key.getPubKeyPoint(), key.isCompressed());
    }

    public ECKey decompress() {
        if (!this.pub.isCompressed()) {
            return this;
        }
        return new ECKey(this.priv, new LazyECPoint(this.pub.get(), false));
    }

    @Deprecated
    public ECKey(@Nullable byte[] privKeyBytes, @Nullable byte[] pubKey) {
        this(privKeyBytes == null ? null : new BigInteger(1, privKeyBytes), pubKey);
    }

    @Deprecated
    public ECKey(EncryptedData encryptedPrivateKey, byte[] pubKey, KeyCrypter keyCrypter) {
        this((byte[])null, pubKey);
        this.keyCrypter = (KeyCrypter)Preconditions.checkNotNull((Object)keyCrypter);
        this.encryptedPrivateKey = encryptedPrivateKey;
    }

    public static ECKey fromEncrypted(EncryptedData encryptedPrivateKey, KeyCrypter crypter, byte[] pubKey) {
        ECKey key = ECKey.fromPublicOnly(pubKey);
        key.encryptedPrivateKey = (EncryptedData)Preconditions.checkNotNull((Object)encryptedPrivateKey);
        key.keyCrypter = (KeyCrypter)Preconditions.checkNotNull((Object)crypter);
        return key;
    }

    @Deprecated
    public ECKey(@Nullable BigInteger privKey, @Nullable byte[] pubKey, boolean compressed) {
        if (privKey == null && pubKey == null) {
            throw new IllegalArgumentException("ECKey requires at least private or public key");
        }
        this.priv = privKey;
        if (pubKey == null) {
            ECPoint point = ECKey.publicPointFromPrivate(privKey);
            this.pub = new LazyECPoint(point, compressed);
        } else {
            this.pub = new LazyECPoint(CURVE.getCurve(), pubKey);
        }
    }

    @Deprecated
    private ECKey(@Nullable BigInteger privKey, @Nullable byte[] pubKey) {
        this(privKey, pubKey, false);
    }

    public boolean isPubKeyOnly() {
        return this.priv == null;
    }

    public boolean hasPrivKey() {
        return this.priv != null;
    }

    public boolean isWatching() {
        return this.isPubKeyOnly() && !this.isEncrypted();
    }

    public byte[] toASN1() {
        try {
            byte[] privKeyBytes = this.getPrivKeyBytes();
            ByteArrayOutputStream baos = new ByteArrayOutputStream(400);
            DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)baos);
            seq.addObject((ASN1Encodable)new ASN1Integer(1L));
            seq.addObject((ASN1Encodable)new DEROctetString(privKeyBytes));
            seq.addObject((ASN1Encodable)new DERTaggedObject(0, (ASN1Encodable)CURVE_PARAMS.toASN1Primitive()));
            seq.addObject((ASN1Encodable)new DERTaggedObject(1, (ASN1Encodable)new DERBitString(this.getPubKey())));
            seq.close();
            return baos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] publicKeyFromPrivate(BigInteger privKey, boolean compressed) {
        ECPoint point = ECKey.publicPointFromPrivate(privKey);
        return point.getEncoded(compressed);
    }

    public static ECPoint publicPointFromPrivate(BigInteger privKey) {
        if (privKey.bitLength() > CURVE.getN().bitLength()) {
            privKey = privKey.mod(CURVE.getN());
        }
        return new FixedPointCombMultiplier().multiply(CURVE.getG(), privKey);
    }

    public byte[] getPubKeyHash() {
        if (this.pubKeyHash == null) {
            this.pubKeyHash = Utils.sha256hash160(this.pub.getEncoded());
        }
        return this.pubKeyHash;
    }

    public byte[] getPubKey() {
        return this.pub.getEncoded();
    }

    public ECPoint getPubKeyPoint() {
        return this.pub.get();
    }

    public BigInteger getPrivKey() {
        if (this.priv == null) {
            throw new MissingPrivateKeyException();
        }
        return this.priv;
    }

    public boolean isCompressed() {
        return this.pub.isCompressed();
    }

    public ECDSASignature sign(Sha256Hash input) throws KeyCrypterException {
        return this.sign(input, null);
    }

    public ECDSASignature sign(Sha256Hash input, @Nullable KeyParameter aesKey) throws KeyCrypterException {
        KeyCrypter crypter = this.getKeyCrypter();
        if (crypter != null) {
            if (aesKey == null) {
                throw new KeyIsEncryptedException();
            }
            return this.decrypt(aesKey).sign(input);
        }
        if (this.priv == null) {
            throw new MissingPrivateKeyException();
        }
        return this.doSign(input, this.priv);
    }

    protected ECDSASignature doSign(Sha256Hash input, BigInteger privateKeyForSigning) {
        if (Secp256k1Context.isEnabled()) {
            try {
                byte[] signature = NativeSecp256k1.sign(input.getBytes(), Utils.bigIntegerToBytes(privateKeyForSigning, 32));
                return ECDSASignature.decodeFromDER(signature);
            }
            catch (NativeSecp256k1Util.AssertFailException e) {
                log.error("Caught AssertFailException inside secp256k1", (Throwable)e);
                throw new RuntimeException(e);
            }
            catch (SignatureDecodeException e) {
                throw new RuntimeException(e);
            }
        }
        if (FAKE_SIGNATURES) {
            return TransactionSignature.dummy();
        }
        Preconditions.checkNotNull((Object)privateKeyForSigning);
        ECDSASigner signer = new ECDSASigner((DSAKCalculator)new HMacDSAKCalculator((Digest)new SHA256Digest()));
        ECPrivateKeyParameters privKey = new ECPrivateKeyParameters(privateKeyForSigning, CURVE);
        signer.init(true, (CipherParameters)privKey);
        BigInteger[] components = signer.generateSignature(input.getBytes());
        return new ECDSASignature(components[0], components[1]).toCanonicalised();
    }

    public static boolean verify(byte[] data, ECDSASignature signature, byte[] pub) {
        if (FAKE_SIGNATURES) {
            return true;
        }
        if (Secp256k1Context.isEnabled()) {
            try {
                return NativeSecp256k1.verify(data, signature.encodeToDER(), pub);
            }
            catch (NativeSecp256k1Util.AssertFailException e) {
                log.error("Caught AssertFailException inside secp256k1", (Throwable)e);
                return false;
            }
        }
        ECDSASigner signer = new ECDSASigner();
        ECPublicKeyParameters params = new ECPublicKeyParameters(CURVE.getCurve().decodePoint(pub), CURVE);
        signer.init(false, (CipherParameters)params);
        try {
            return signer.verifySignature(data, signature.r, signature.s);
        }
        catch (NullPointerException e) {
            log.error("Caught NPE inside bouncy castle", (Throwable)e);
            return false;
        }
    }

    public static boolean verify(byte[] data, byte[] signature, byte[] pub) throws SignatureDecodeException {
        if (Secp256k1Context.isEnabled()) {
            try {
                return NativeSecp256k1.verify(data, signature, pub);
            }
            catch (NativeSecp256k1Util.AssertFailException e) {
                log.error("Caught AssertFailException inside secp256k1", (Throwable)e);
                return false;
            }
        }
        return ECKey.verify(data, ECDSASignature.decodeFromDER(signature), pub);
    }

    public boolean verify(byte[] hash, byte[] signature) throws SignatureDecodeException {
        return ECKey.verify(hash, signature, this.getPubKey());
    }

    public boolean verify(Sha256Hash sigHash, ECDSASignature signature) {
        return ECKey.verify(sigHash.getBytes(), signature, this.getPubKey());
    }

    public void verifyOrThrow(byte[] hash, byte[] signature) throws SignatureDecodeException, SignatureException {
        if (!this.verify(hash, signature)) {
            throw new SignatureException();
        }
    }

    public void verifyOrThrow(Sha256Hash sigHash, ECDSASignature signature) throws SignatureException {
        if (!ECKey.verify(sigHash.getBytes(), signature, this.getPubKey())) {
            throw new SignatureException();
        }
    }

    public static boolean isPubKeyCanonical(byte[] pubkey) {
        if (pubkey.length < 33) {
            return false;
        }
        if (pubkey[0] == 4) {
            if (pubkey.length != 65) {
                return false;
            }
        } else if (pubkey[0] == 2 || pubkey[0] == 3) {
            if (pubkey.length != 33) {
                return false;
            }
        } else {
            return false;
        }
        return true;
    }

    public static boolean isPubKeyCompressed(byte[] encoded) {
        if (encoded.length == 33 && (encoded[0] == 2 || encoded[0] == 3)) {
            return true;
        }
        if (encoded.length == 65 && encoded[0] == 4) {
            return false;
        }
        throw new IllegalArgumentException(Utils.HEX.encode(encoded));
    }

    private static ECKey extractKeyFromASN1(byte[] asn1privkey) {
        try {
            ASN1InputStream decoder = new ASN1InputStream(asn1privkey);
            DLSequence seq = (DLSequence)decoder.readObject();
            Preconditions.checkArgument((decoder.readObject() == null ? 1 : 0) != 0, (Object)"Input contains extra bytes");
            decoder.close();
            Preconditions.checkArgument((seq.size() == 4 ? 1 : 0) != 0, (Object)"Input does not appear to be an ASN.1 OpenSSL EC private key");
            Preconditions.checkArgument((boolean)((ASN1Integer)seq.getObjectAt(0)).getValue().equals(BigInteger.ONE), (Object)"Input is of wrong version");
            byte[] privbits = ((ASN1OctetString)seq.getObjectAt(1)).getOctets();
            BigInteger privkey = new BigInteger(1, privbits);
            ASN1TaggedObject pubkey = (ASN1TaggedObject)seq.getObjectAt(3);
            Preconditions.checkArgument((pubkey.getTagNo() == 1 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with bad tag number");
            byte[] pubbits = ((DERBitString)pubkey.getObject()).getBytes();
            Preconditions.checkArgument((pubbits.length == 33 || pubbits.length == 65 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with invalid length");
            int encoding = pubbits[0] & 0xFF;
            Preconditions.checkArgument((encoding >= 2 && encoding <= 4 ? 1 : 0) != 0, (Object)"Input has 'publicKey' with invalid encoding");
            boolean compressed = ECKey.isPubKeyCompressed(pubbits);
            ECKey key = new ECKey(privkey, (byte[])null, compressed);
            if (!Arrays.equals(key.getPubKey(), pubbits)) {
                throw new IllegalArgumentException("Public key in ASN.1 structure does not match private key.");
            }
            return key;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public String signMessage(String message) throws KeyCrypterException {
        return this.signMessage(message, null);
    }

    public String signMessage(String message, @Nullable KeyParameter aesKey) throws KeyCrypterException {
        byte[] data = ECKey.formatMessageForSigning(message);
        Sha256Hash hash = Sha256Hash.twiceOf(data);
        ECDSASignature sig = this.sign(hash, aesKey);
        byte recId = this.findRecoveryId(hash, sig);
        int headerByte = recId + 27 + (this.isCompressed() ? 4 : 0);
        byte[] sigData = new byte[65];
        sigData[0] = (byte)headerByte;
        System.arraycopy(Utils.bigIntegerToBytes(sig.r, 32), 0, sigData, 1, 32);
        System.arraycopy(Utils.bigIntegerToBytes(sig.s, 32), 0, sigData, 33, 32);
        return new String(Base64.encode((byte[])sigData), StandardCharsets.UTF_8);
    }

    public static ECKey signedMessageToKey(String message, String signatureBase64) throws SignatureException {
        int recId;
        ECKey key;
        byte[] signatureEncoded;
        try {
            signatureEncoded = Base64.decode((String)signatureBase64);
        }
        catch (RuntimeException e) {
            throw new SignatureException("Could not decode base64", e);
        }
        if (signatureEncoded.length < 65) {
            throw new SignatureException("Signature truncated, expected 65 bytes and got " + signatureEncoded.length);
        }
        int header = signatureEncoded[0] & 0xFF;
        if (header < 27 || header > 34) {
            throw new SignatureException("Header byte out of range: " + header);
        }
        BigInteger r = new BigInteger(1, Arrays.copyOfRange(signatureEncoded, 1, 33));
        BigInteger s = new BigInteger(1, Arrays.copyOfRange(signatureEncoded, 33, 65));
        ECDSASignature sig = new ECDSASignature(r, s);
        byte[] messageBytes = ECKey.formatMessageForSigning(message);
        Sha256Hash messageHash = Sha256Hash.twiceOf(messageBytes);
        boolean compressed = false;
        if (header >= 31) {
            compressed = true;
            header -= 4;
        }
        if ((key = ECKey.recoverFromSignature(recId = header - 27, sig, messageHash, compressed)) == null) {
            throw new SignatureException("Could not recover public key from signature");
        }
        return key;
    }

    public void verifyMessage(String message, String signatureBase64) throws SignatureException {
        ECKey key = ECKey.signedMessageToKey(message, signatureBase64);
        if (!key.pub.equals(this.pub)) {
            throw new SignatureException("Signature did not match for message");
        }
    }

    public byte findRecoveryId(Sha256Hash hash, ECDSASignature sig) {
        int recId = -1;
        for (int i = 0; i < 4; i = (int)((byte)(i + 1))) {
            ECKey k = ECKey.recoverFromSignature(i, sig, hash, this.isCompressed());
            if (k == null || !k.pub.equals(this.pub)) continue;
            recId = i;
            break;
        }
        if (recId == -1) {
            throw new RuntimeException("Could not construct a recoverable key. This should never happen.");
        }
        return (byte)recId;
    }

    @Nullable
    public static ECKey recoverFromSignature(int recId, ECDSASignature sig, Sha256Hash message, boolean compressed) {
        Preconditions.checkArgument((recId >= 0 ? 1 : 0) != 0, (Object)"recId must be positive");
        Preconditions.checkArgument((sig.r.signum() >= 0 ? 1 : 0) != 0, (Object)"r must be positive");
        Preconditions.checkArgument((sig.s.signum() >= 0 ? 1 : 0) != 0, (Object)"s must be positive");
        Preconditions.checkNotNull((Object)message);
        BigInteger n = CURVE.getN();
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = sig.r.add(i.multiply(n));
        BigInteger prime = SecP256K1Curve.q;
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = ECKey.decompressKey(x, (recId & 1) == 1);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = message.toBigInteger();
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = sig.r.modInverse(n);
        BigInteger srInv = rInv.multiply(sig.s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)CURVE.getG(), (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        return ECKey.fromPublicOnly(q, compressed);
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit) {
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength(CURVE.getCurve()));
        compEnc[0] = (byte)(yBit ? 3 : 2);
        return CURVE.getCurve().decodePoint(compEnc);
    }

    public byte[] getPrivKeyBytes() {
        return Utils.bigIntegerToBytes(this.getPrivKey(), 32);
    }

    public DumpedPrivateKey getPrivateKeyEncoded(NetworkParameters params) {
        return new DumpedPrivateKey(params, this.getPrivKeyBytes(), this.isCompressed());
    }

    @Override
    public long getCreationTimeSeconds() {
        return this.creationTimeSeconds;
    }

    public void setCreationTimeSeconds(long newCreationTimeSeconds) {
        if (newCreationTimeSeconds < 0L) {
            throw new IllegalArgumentException("Cannot set creation time to negative value: " + newCreationTimeSeconds);
        }
        this.creationTimeSeconds = newCreationTimeSeconds;
    }

    public ECKey encrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        byte[] privKeyBytes = this.getPrivKeyBytes();
        EncryptedData encryptedPrivateKey = keyCrypter.encrypt(privKeyBytes, aesKey);
        ECKey result = ECKey.fromEncrypted(encryptedPrivateKey, keyCrypter, this.getPubKey());
        result.setCreationTimeSeconds(this.creationTimeSeconds);
        return result;
    }

    public ECKey decrypt(KeyCrypter keyCrypter, KeyParameter aesKey) throws KeyCrypterException {
        Preconditions.checkNotNull((Object)keyCrypter);
        if (this.keyCrypter != null && !this.keyCrypter.equals(keyCrypter)) {
            throw new KeyCrypterException("The keyCrypter being used to decrypt the key is different to the one that was used to encrypt it");
        }
        Preconditions.checkState((this.encryptedPrivateKey != null ? 1 : 0) != 0, (Object)"This key is not encrypted");
        byte[] unencryptedPrivateKey = keyCrypter.decrypt(this.encryptedPrivateKey, aesKey);
        if (unencryptedPrivateKey.length != 32) {
            throw new KeyCrypterException.InvalidCipherText("Decrypted key must be 32 bytes long, but is " + unencryptedPrivateKey.length);
        }
        ECKey key = ECKey.fromPrivate(unencryptedPrivateKey, this.isCompressed());
        if (!Arrays.equals(key.getPubKey(), this.getPubKey())) {
            throw new KeyCrypterException("Provided AES key is wrong");
        }
        key.setCreationTimeSeconds(this.creationTimeSeconds);
        return key;
    }

    public ECKey decrypt(KeyParameter aesKey) throws KeyCrypterException {
        KeyCrypter crypter = this.getKeyCrypter();
        if (crypter == null) {
            throw new KeyCrypterException("No key crypter available");
        }
        return this.decrypt(crypter, aesKey);
    }

    public ECKey maybeDecrypt(@Nullable KeyParameter aesKey) throws KeyCrypterException {
        return this.isEncrypted() && aesKey != null ? this.decrypt(aesKey) : this;
    }

    public static boolean encryptionIsReversible(ECKey originalKey, ECKey encryptedKey, KeyCrypter keyCrypter, KeyParameter aesKey) {
        try {
            ECKey rebornUnencryptedKey = encryptedKey.decrypt(keyCrypter, aesKey);
            byte[] originalPrivateKeyBytes = originalKey.getPrivKeyBytes();
            byte[] rebornKeyBytes = rebornUnencryptedKey.getPrivKeyBytes();
            if (!Arrays.equals(originalPrivateKeyBytes, rebornKeyBytes)) {
                log.error("The check that encryption could be reversed failed for {}", (Object)originalKey);
                return false;
            }
            return true;
        }
        catch (KeyCrypterException kce) {
            log.error(kce.getMessage());
            return false;
        }
    }

    @Override
    public boolean isEncrypted() {
        return this.keyCrypter != null && this.encryptedPrivateKey != null && this.encryptedPrivateKey.encryptedBytes.length > 0;
    }

    @Override
    @Nullable
    public Protos.Wallet.EncryptionType getEncryptionType() {
        return this.keyCrypter != null ? this.keyCrypter.getUnderstoodEncryptionType() : Protos.Wallet.EncryptionType.UNENCRYPTED;
    }

    @Override
    @Nullable
    public byte[] getSecretBytes() {
        if (this.hasPrivKey()) {
            return this.getPrivKeyBytes();
        }
        return null;
    }

    @Override
    @Nullable
    public EncryptedData getEncryptedData() {
        return this.getEncryptedPrivateKey();
    }

    @Nullable
    public EncryptedData getEncryptedPrivateKey() {
        return this.encryptedPrivateKey;
    }

    @Nullable
    public KeyCrypter getKeyCrypter() {
        return this.keyCrypter;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || !(o instanceof ECKey)) {
            return false;
        }
        ECKey other = (ECKey)o;
        return Objects.equals(this.priv, other.priv) && Objects.equals(this.pub, other.pub) && Objects.equals(this.creationTimeSeconds, other.creationTimeSeconds) && Objects.equals(this.keyCrypter, other.keyCrypter) && Objects.equals(this.encryptedPrivateKey, other.encryptedPrivateKey);
    }

    public int hashCode() {
        return this.pub.hashCode();
    }

    public String toString() {
        return this.toString(false, null, null);
    }

    public String toStringWithPrivate(@Nullable KeyParameter aesKey, NetworkParameters params) {
        return this.toString(true, aesKey, params);
    }

    public String getPrivateKeyAsHex() {
        return Utils.HEX.encode(this.getPrivKeyBytes());
    }

    public String getPublicKeyAsHex() {
        return Utils.HEX.encode(this.pub.getEncoded());
    }

    public String getPrivateKeyAsWiF(NetworkParameters params) {
        return this.getPrivateKeyEncoded(params).toString();
    }

    private String toString(boolean includePrivate, @Nullable KeyParameter aesKey, @Nullable NetworkParameters params) {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper((Object)this).omitNullValues();
        helper.add("pub HEX", (Object)this.getPublicKeyAsHex());
        if (includePrivate) {
            ECKey decryptedKey = this.isEncrypted() ? this.decrypt((KeyParameter)Preconditions.checkNotNull((Object)aesKey)) : this;
            try {
                helper.add("priv HEX", (Object)decryptedKey.getPrivateKeyAsHex());
                helper.add("priv WIF", (Object)decryptedKey.getPrivateKeyAsWiF(params));
            }
            catch (IllegalStateException illegalStateException) {
            }
            catch (Exception e) {
                String message = e.getMessage();
                helper.add("priv EXCEPTION", (Object)(e.getClass().getName() + (message != null ? ": " + message : "")));
            }
        }
        if (this.creationTimeSeconds > 0L) {
            helper.add("creationTimeSeconds", this.creationTimeSeconds);
        }
        helper.add("keyCrypter", (Object)this.keyCrypter);
        if (includePrivate) {
            helper.add("encryptedPrivateKey", (Object)this.encryptedPrivateKey);
        }
        helper.add("isEncrypted", this.isEncrypted());
        helper.add("isPubKeyOnly", this.isPubKeyOnly());
        return helper.toString();
    }

    public void formatKeyWithAddress(boolean includePrivateKeys, @Nullable KeyParameter aesKey, StringBuilder builder, NetworkParameters params, Script.ScriptType outputScriptType, @Nullable String comment) {
        builder.append("  addr:");
        if (outputScriptType != null) {
            builder.append(Address.fromKey(params, this, outputScriptType));
        } else {
            builder.append(LegacyAddress.fromKey(params, this));
            if (this.isCompressed()) {
                builder.append(',').append(SegwitAddress.fromKey(params, this));
            }
        }
        if (!this.isCompressed()) {
            builder.append("  UNCOMPRESSED");
        }
        builder.append("  hash160:");
        builder.append(Utils.HEX.encode(this.getPubKeyHash()));
        if (this.creationTimeSeconds > 0L) {
            builder.append("  creationTimeSeconds:").append(this.creationTimeSeconds).append(" [").append(Utils.dateTimeFormat(this.creationTimeSeconds * 1000L)).append("]");
        }
        if (comment != null) {
            builder.append("  (").append(comment).append(")");
        }
        builder.append("\n");
        if (includePrivateKeys) {
            builder.append("  ");
            builder.append(this.toStringWithPrivate(aesKey, params));
            builder.append("\n");
        }
    }

    private static byte[] formatMessageForSigning(String message) {
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES.length);
            bos.write(BITCOIN_SIGNED_MESSAGE_HEADER_BYTES);
            byte[] messageBytes = message.getBytes(StandardCharsets.UTF_8);
            VarInt size = new VarInt(messageBytes.length);
            bos.write(size.encode());
            bos.write(messageBytes);
            return bos.toByteArray();
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static {
        if (Utils.isAndroidRuntime()) {
            new LinuxSecureRandom();
        }
        FixedPointUtil.precompute((ECPoint)CURVE_PARAMS.getG());
        CURVE = new ECDomainParameters(CURVE_PARAMS.getCurve(), CURVE_PARAMS.getG(), CURVE_PARAMS.getN(), CURVE_PARAMS.getH());
        HALF_CURVE_ORDER = CURVE_PARAMS.getN().shiftRight(1);
        secureRandom = new SecureRandom();
        FAKE_SIGNATURES = false;
        BITCOIN_SIGNED_MESSAGE_HEADER_BYTES = BITCOIN_SIGNED_MESSAGE_HEADER.getBytes(StandardCharsets.UTF_8);
    }

    public static class KeyIsEncryptedException
    extends MissingPrivateKeyException {
    }

    public static class MissingPrivateKeyException
    extends RuntimeException {
    }

    public static class ECDSASignature {
        public final BigInteger r;
        public final BigInteger s;

        public ECDSASignature(BigInteger r, BigInteger s) {
            this.r = r;
            this.s = s;
        }

        public boolean isCanonical() {
            return this.s.compareTo(HALF_CURVE_ORDER) <= 0;
        }

        public ECDSASignature toCanonicalised() {
            if (!this.isCanonical()) {
                return new ECDSASignature(this.r, CURVE.getN().subtract(this.s));
            }
            return this;
        }

        public byte[] encodeToDER() {
            try {
                return this.derByteStream().toByteArray();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }

        public static ECDSASignature decodeFromDER(byte[] bytes) throws SignatureDecodeException {
            ASN1InputStream decoder = null;
            try {
                ASN1Integer s;
                ASN1Integer r;
                Properties.setThreadOverride((String)"org.bouncycastle.asn1.allow_unsafe_integer", (boolean)true);
                decoder = new ASN1InputStream(bytes);
                ASN1Primitive seqObj = decoder.readObject();
                if (seqObj == null) {
                    throw new SignatureDecodeException("Reached past end of ASN.1 stream.");
                }
                if (!(seqObj instanceof DLSequence)) {
                    throw new SignatureDecodeException("Read unexpected class: " + seqObj.getClass().getName());
                }
                DLSequence seq = (DLSequence)seqObj;
                try {
                    r = (ASN1Integer)seq.getObjectAt(0);
                    s = (ASN1Integer)seq.getObjectAt(1);
                }
                catch (ClassCastException e) {
                    throw new SignatureDecodeException(e);
                }
                ECDSASignature eCDSASignature = new ECDSASignature(r.getPositiveValue(), s.getPositiveValue());
                return eCDSASignature;
            }
            catch (IOException e) {
                throw new SignatureDecodeException(e);
            }
            finally {
                if (decoder != null) {
                    try {
                        decoder.close();
                    }
                    catch (IOException iOException) {}
                }
                Properties.removeThreadOverride((String)"org.bouncycastle.asn1.allow_unsafe_integer");
            }
        }

        protected ByteArrayOutputStream derByteStream() throws IOException {
            ByteArrayOutputStream bos = new ByteArrayOutputStream(72);
            DERSequenceGenerator seq = new DERSequenceGenerator((OutputStream)bos);
            seq.addObject((ASN1Encodable)new ASN1Integer(this.r));
            seq.addObject((ASN1Encodable)new ASN1Integer(this.s));
            seq.close();
            return bos;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ECDSASignature other = (ECDSASignature)o;
            return this.r.equals(other.r) && this.s.equals(other.s);
        }

        public int hashCode() {
            return Objects.hash(this.r, this.s);
        }
    }
}

