/*
 * Decompiled with CFR 0.152.
 */
package org.bitcoinj.wallet;

import com.google.common.base.Preconditions;
import java.util.List;
import javax.annotation.Nullable;
import org.bitcoinj.core.Coin;
import org.bitcoinj.core.ECKey;
import org.bitcoinj.core.Transaction;
import org.bitcoinj.core.TransactionConfidence;
import org.bitcoinj.core.TransactionInput;
import org.bitcoinj.core.TransactionOutput;
import org.bitcoinj.core.Wallet;
import org.bitcoinj.crypto.TransactionSignature;
import org.bitcoinj.script.ScriptChunk;
import org.bitcoinj.wallet.RiskAnalysis;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultRiskAnalysis
implements RiskAnalysis {
    private static final Logger log = LoggerFactory.getLogger(DefaultRiskAnalysis.class);
    public static final Coin MIN_ANALYSIS_NONDUST_OUTPUT = Transaction.MIN_NONDUST_OUTPUT;
    protected final Transaction tx;
    protected final List<Transaction> dependencies;
    @Nullable
    protected final Wallet wallet;
    private Transaction nonStandard;
    protected Transaction nonFinal;
    protected boolean analyzed;
    public static Analyzer FACTORY = new Analyzer();

    private DefaultRiskAnalysis(Wallet wallet, Transaction tx, List<Transaction> dependencies) {
        this.tx = tx;
        this.dependencies = dependencies;
        this.wallet = wallet;
    }

    @Override
    public RiskAnalysis.Result analyze() {
        Preconditions.checkState((!this.analyzed ? 1 : 0) != 0);
        this.analyzed = true;
        RiskAnalysis.Result result = this.analyzeIsFinal();
        if (result != null && result != RiskAnalysis.Result.OK) {
            return result;
        }
        return this.analyzeIsStandard();
    }

    @Nullable
    private RiskAnalysis.Result analyzeIsFinal() {
        long time;
        if (this.tx.getConfidence().getSource() == TransactionConfidence.Source.SELF) {
            return RiskAnalysis.Result.OK;
        }
        if (this.tx.isOptInFullRBF()) {
            this.nonFinal = this.tx;
            return RiskAnalysis.Result.NON_FINAL;
        }
        if (this.wallet == null) {
            return null;
        }
        int height = this.wallet.getLastBlockSeenHeight();
        int adjustedHeight = height + 1;
        if (!this.tx.isFinal(adjustedHeight, time = this.wallet.getLastBlockSeenTimeSecs())) {
            this.nonFinal = this.tx;
            return RiskAnalysis.Result.NON_FINAL;
        }
        for (Transaction dep : this.dependencies) {
            if (dep.isFinal(adjustedHeight, time)) continue;
            this.nonFinal = dep;
            return RiskAnalysis.Result.NON_FINAL;
        }
        return RiskAnalysis.Result.OK;
    }

    public static RuleViolation isStandard(Transaction tx) {
        if (tx.getVersion() > 1L || tx.getVersion() < 1L) {
            log.warn("TX considered non-standard due to unknown version number {}", (Object)tx.getVersion());
            return RuleViolation.VERSION;
        }
        List<TransactionOutput> outputs = tx.getOutputs();
        for (int i = 0; i < outputs.size(); ++i) {
            TransactionOutput output = outputs.get(i);
            RuleViolation violation = DefaultRiskAnalysis.isOutputStandard(output);
            if (violation == RuleViolation.NONE) continue;
            log.warn("TX considered non-standard due to output {} violating rule {}", (Object)i, (Object)violation);
            return violation;
        }
        List<TransactionInput> inputs = tx.getInputs();
        for (int i = 0; i < inputs.size(); ++i) {
            TransactionInput input = inputs.get(i);
            RuleViolation violation = DefaultRiskAnalysis.isInputStandard(input);
            if (violation == RuleViolation.NONE) continue;
            log.warn("TX considered non-standard due to input {} violating rule {}", (Object)i, (Object)violation);
            return violation;
        }
        return RuleViolation.NONE;
    }

    public static RuleViolation isOutputStandard(TransactionOutput output) {
        if (output.getValue().compareTo(MIN_ANALYSIS_NONDUST_OUTPUT) < 0) {
            return RuleViolation.DUST;
        }
        for (ScriptChunk chunk : output.getScriptPubKey().getChunks()) {
            if (!chunk.isPushData() || chunk.isShortestPossiblePushData()) continue;
            return RuleViolation.SHORTEST_POSSIBLE_PUSHDATA;
        }
        return RuleViolation.NONE;
    }

    public static RuleViolation isInputStandard(TransactionInput input) {
        for (ScriptChunk chunk : input.getScriptSig().getChunks()) {
            ECKey.ECDSASignature signature;
            if (chunk.data != null && !chunk.isShortestPossiblePushData()) {
                return RuleViolation.SHORTEST_POSSIBLE_PUSHDATA;
            }
            if (!chunk.isPushData()) continue;
            try {
                signature = ECKey.ECDSASignature.decodeFromDER(chunk.data);
            }
            catch (RuntimeException x) {
                signature = null;
            }
            if (signature == null) continue;
            if (!TransactionSignature.isEncodingCanonical(chunk.data)) {
                return RuleViolation.SIGNATURE_CANONICAL_ENCODING;
            }
            if (signature.isCanonical()) continue;
            return RuleViolation.SIGNATURE_CANONICAL_ENCODING;
        }
        return RuleViolation.NONE;
    }

    private RiskAnalysis.Result analyzeIsStandard() {
        if (this.wallet != null && !this.wallet.getNetworkParameters().getId().equals("org.bitcoin.production")) {
            return RiskAnalysis.Result.OK;
        }
        RuleViolation ruleViolation = DefaultRiskAnalysis.isStandard(this.tx);
        if (ruleViolation != RuleViolation.NONE) {
            this.nonStandard = this.tx;
            return RiskAnalysis.Result.NON_STANDARD;
        }
        for (Transaction dep : this.dependencies) {
            ruleViolation = DefaultRiskAnalysis.isStandard(dep);
            if (ruleViolation == RuleViolation.NONE) continue;
            this.nonStandard = dep;
            return RiskAnalysis.Result.NON_STANDARD;
        }
        return RiskAnalysis.Result.OK;
    }

    @Nullable
    public Transaction getNonStandard() {
        return this.nonStandard;
    }

    @Nullable
    public Transaction getNonFinal() {
        return this.nonFinal;
    }

    public String toString() {
        if (!this.analyzed) {
            return "Pending risk analysis for " + this.tx.getHashAsString();
        }
        if (this.nonFinal != null) {
            return "Risky due to non-finality of " + this.nonFinal.getHashAsString();
        }
        if (this.nonStandard != null) {
            return "Risky due to non-standard tx " + this.nonStandard.getHashAsString();
        }
        return "Non-risky";
    }

    public static class Analyzer
    implements RiskAnalysis.Analyzer {
        @Override
        public DefaultRiskAnalysis create(Wallet wallet, Transaction tx, List<Transaction> dependencies) {
            return new DefaultRiskAnalysis(wallet, tx, dependencies);
        }
    }

    public static enum RuleViolation {
        NONE,
        VERSION,
        DUST,
        SHORTEST_POSSIBLE_PUSHDATA,
        NONEMPTY_STACK,
        SIGNATURE_CANONICAL_ENCODING;

    }
}

