/*
 * Decompiled with CFR 0.152.
 */
package org.atmosphere.session.sqlite;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.atmosphere.session.DurableSession;
import org.atmosphere.session.SessionStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SqliteSessionStore
implements SessionStore {
    private static final Logger logger = LoggerFactory.getLogger(SqliteSessionStore.class);
    private final Connection connection;
    private final ObjectMapper mapper = new ObjectMapper();
    private static final TypeReference<Set<String>> SET_TYPE = new TypeReference<Set<String>>(){};
    private static final TypeReference<Map<String, String>> MAP_TYPE = new TypeReference<Map<String, String>>(){};

    public SqliteSessionStore() {
        this(Path.of("atmosphere-sessions.db", new String[0]));
    }

    public SqliteSessionStore(Path dbPath) {
        this("jdbc:sqlite:" + String.valueOf(dbPath.toAbsolutePath()));
    }

    public static SqliteSessionStore inMemory() {
        return new SqliteSessionStore("jdbc:sqlite::memory:");
    }

    private SqliteSessionStore(String jdbcUrl) {
        try {
            this.connection = DriverManager.getConnection(jdbcUrl);
            try (Statement stmt = this.connection.createStatement();){
                stmt.execute("PRAGMA journal_mode=WAL");
            }
            this.createTable();
            logger.info("SQLite session store initialized: {}", (Object)jdbcUrl);
        }
        catch (SQLException e) {
            throw new IllegalStateException("Failed to initialize SQLite session store", e);
        }
    }

    private void createTable() throws SQLException {
        try (Statement stmt = this.connection.createStatement();){
            stmt.execute("CREATE TABLE IF NOT EXISTS durable_sessions (\n    token        TEXT PRIMARY KEY,\n    resource_id  TEXT NOT NULL,\n    rooms        TEXT NOT NULL DEFAULT '',\n    broadcasters TEXT NOT NULL DEFAULT '',\n    metadata     TEXT NOT NULL DEFAULT '',\n    created_at   INTEGER NOT NULL,\n    last_seen    INTEGER NOT NULL\n)\n");
            stmt.execute("CREATE INDEX IF NOT EXISTS idx_last_seen\nON durable_sessions(last_seen)\n");
        }
    }

    public synchronized void save(DurableSession session) {
        String sql = "INSERT OR REPLACE INTO durable_sessions\n(token, resource_id, rooms, broadcasters, metadata, created_at, last_seen)\nVALUES (?, ?, ?, ?, ?, ?, ?)\n";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, session.token());
            stmt.setString(2, session.resourceId());
            stmt.setString(3, this.toJson(session.rooms()));
            stmt.setString(4, this.toJson(session.broadcasters()));
            stmt.setString(5, this.toJson(session.metadata()));
            stmt.setLong(6, session.createdAt().toEpochMilli());
            stmt.setLong(7, session.lastSeen().toEpochMilli());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            logger.error("Failed to save session {}", (Object)session.token(), (Object)e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public synchronized Optional<DurableSession> restore(String token) {
        String sql = "SELECT * FROM durable_sessions WHERE token = ?";
        try (PreparedStatement stmt = this.connection.prepareStatement(sql);){
            stmt.setString(1, token);
            try (ResultSet rs = stmt.executeQuery();){
                if (!rs.next()) return Optional.empty();
                Optional<DurableSession> optional = Optional.of(new DurableSession(rs.getString("token"), rs.getString("resource_id"), this.fromJsonSet(rs.getString("rooms")), this.fromJsonSet(rs.getString("broadcasters")), this.fromJsonMap(rs.getString("metadata")), Instant.ofEpochMilli(rs.getLong("created_at")), Instant.ofEpochMilli(rs.getLong("last_seen"))));
                return optional;
            }
        }
        catch (SQLException e) {
            logger.error("Failed to restore session {}", (Object)token, (Object)e);
        }
        return Optional.empty();
    }

    public synchronized void remove(String token) {
        try (PreparedStatement stmt = this.connection.prepareStatement("DELETE FROM durable_sessions WHERE token = ?");){
            stmt.setString(1, token);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            logger.error("Failed to remove session {}", (Object)token, (Object)e);
        }
    }

    public synchronized void touch(String token) {
        try (PreparedStatement stmt = this.connection.prepareStatement("UPDATE durable_sessions SET last_seen = ? WHERE token = ?");){
            stmt.setLong(1, Instant.now().toEpochMilli());
            stmt.setString(2, token);
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            logger.error("Failed to touch session {}", (Object)token, (Object)e);
        }
    }

    public synchronized List<DurableSession> removeExpired(Duration ttl) {
        long cutoff = Instant.now().minus(ttl).toEpochMilli();
        ArrayList<DurableSession> expired = new ArrayList<DurableSession>();
        try (PreparedStatement select = this.connection.prepareStatement("SELECT * FROM durable_sessions WHERE last_seen < ?");){
            select.setLong(1, cutoff);
            try (ResultSet rs = select.executeQuery();){
                while (rs.next()) {
                    expired.add(new DurableSession(rs.getString("token"), rs.getString("resource_id"), this.fromJsonSet(rs.getString("rooms")), this.fromJsonSet(rs.getString("broadcasters")), this.fromJsonMap(rs.getString("metadata")), Instant.ofEpochMilli(rs.getLong("created_at")), Instant.ofEpochMilli(rs.getLong("last_seen"))));
                }
            }
        }
        catch (SQLException e) {
            logger.error("Failed to query expired sessions", (Throwable)e);
        }
        if (!expired.isEmpty()) {
            try (PreparedStatement delete = this.connection.prepareStatement("DELETE FROM durable_sessions WHERE last_seen < ?");){
                delete.setLong(1, cutoff);
                delete.executeUpdate();
            }
            catch (SQLException e) {
                logger.error("Failed to delete expired sessions", (Throwable)e);
            }
        }
        return expired;
    }

    public synchronized void close() {
        try {
            if (this.connection != null && !this.connection.isClosed()) {
                this.connection.close();
                logger.info("SQLite session store closed");
            }
        }
        catch (SQLException e) {
            logger.warn("Error closing SQLite connection", (Throwable)e);
        }
    }

    private String toJson(Object value) {
        try {
            return this.mapper.writeValueAsString(value);
        }
        catch (JsonProcessingException e) {
            logger.error("Failed to serialize value", (Throwable)e);
            return "[]";
        }
    }

    private Set<String> fromJsonSet(String json) {
        if (json == null || json.isEmpty()) {
            return Set.of();
        }
        try {
            return (Set)this.mapper.readValue(json, SET_TYPE);
        }
        catch (JsonProcessingException e) {
            logger.error("Failed to deserialize set", (Throwable)e);
            return Set.of();
        }
    }

    private Map<String, String> fromJsonMap(String json) {
        if (json == null || json.isEmpty()) {
            return Map.of();
        }
        try {
            return (Map)this.mapper.readValue(json, MAP_TYPE);
        }
        catch (JsonProcessingException e) {
            logger.error("Failed to deserialize map", (Throwable)e);
            return Map.of();
        }
    }
}

