/*
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
 * an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations under the License.
 *
 * Copyright 2012-2018 the original author or authors.
 */
package org.assertj.swing.driver;

import static java.util.Collections.enumeration;
import static org.assertj.core.util.Preconditions.checkNotNull;
import static org.assertj.swing.edt.GuiActionRunner.execute;

import java.applet.Applet;
import java.applet.AppletContext;
import java.applet.AppletStub;
import java.net.URL;
import java.util.Collections;
import java.util.Enumeration;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.swing.JApplet;

import org.assertj.swing.annotation.RunsInEDT;
import org.assertj.swing.core.Robot;
import org.assertj.swing.internal.annotation.InternalApi;

/**
 * <p>
 * Supports functional testing of {@code JApplet}s.
 * </p>
 *
 * <p>
 * <b>Note:</b> This class is intended for internal use only. Please use the classes in the package
 * {@link org.assertj.swing.fixture} in your tests.
 * </p>
 *
 * @author Mel Llaguno
 * @author Alex Ruiz
 */
@InternalApi
public class JAppletDriver extends ComponentDriver implements AppletStub {
  private JApplet applet;

  /**
   * Creates a new {@link JAppletDriver}.
   *
   * @param robot the robot used to simulate user input.
   */
  public JAppletDriver(@Nonnull Robot robot) {
    super(robot);
  }

  /**
   * Creates a new {@link JAppletDriver}.
   *
   * @param robot the robot used to simulate user input.
   * @param newApplet the applet to simulate user input against.
   */
  public JAppletDriver(@Nonnull Robot robot, @Nonnull JApplet newApplet) {
    this(robot);
    applet = newApplet;
  }

  @RunsInEDT
  private static void doResize(final @Nonnull JApplet applet, final int width, final int height) {
    execute(() -> applet.resize(width, height));
  }

  @RunsInEDT
  @Nullable private static URL codeBase(final @Nonnull JApplet applet) {
    return execute(() -> applet.getCodeBase());
  }

  @RunsInEDT
  @Nullable private static URL documentBase(final @Nonnull JApplet applet) {
    return execute(() -> applet.getDocumentBase());
  }

  @RunsInEDT
  @Nullable private static String parameter(final @Nonnull JApplet applet, final @Nullable String parameterName) {
    return execute(() -> applet.getParameter(parameterName));
  }

  @RunsInEDT
  private static boolean active(final @Nonnull JApplet applet) {
    Boolean result = execute(() -> applet.isActive());
    return checkNotNull(result);
  }

  /**
   * Requests the default {@code JApplet} to be resized.
   *
   * @param width the new width.
   * @param height the new height.
   */
  @RunsInEDT
  @Override
  public void appletResize(int width, int height) {
    appletResize(applet, width, height);
  }

  /**
   * Requests the given {@code JApplet} to be resized.
   *
   * @param applet the given {@code JApplet}.
   * @param width the new width.
   * @param height the new height.
   */
  @RunsInEDT
  public void appletResize(@Nonnull JApplet applet, int width, int height) {
    doResize(applet, width, height);
  }

  /**
   * @return the {@code AppletContext} of the default {@code JApplet}.
   */
  @RunsInEDT
  @Override
  public AppletContext getAppletContext() {
    return getAppletContext(applet);
  }

  /**
   * Returns the {@link AppletContext} of the given {@code JApplet}.
   *
   * @param applet the given {@code JApplet}.
   * @return the {@code AppletContext} of the given {@code JApplet}.
   */
  @RunsInEDT
  public AppletContext getAppletContext(JApplet applet) {
    return appletContext(applet);
  }

  @RunsInEDT
  @Nullable private static AppletContext appletContext(final @Nonnull JApplet applet) {
    return execute(() -> applet.getAppletContext());
  }

  /**
   * @return the URL of the directory that contains the default {@code JApplet}.
   */
  @RunsInEDT
  @Override
  @Nullable public URL getCodeBase() {
    return getCodeBase(applet);
  }

  /**
   * Returns the URL of the directory that contains the given {@code JApplet}.
   *
   * @param applet the given {@code JApplet}.
   * @return the URL of the directory that contains the given {@code JApplet}.
   */
  @RunsInEDT
  @Nullable public URL getCodeBase(@Nonnull JApplet applet) {
    return codeBase(applet);
  }

  /**
   * Returns the URL of the document the default {@code JApplet} is embedded.
   *
   * @return the URL of the document the given {@code JApplet} is embedded.
   */
  @RunsInEDT
  @Override
  @Nullable public URL getDocumentBase() {
    return getDocumentBase(applet);
  }

  /**
   * Returns the URL of the document the given {@code JApplet} is embedded.
   *
   * @param applet the given {@code JApplet}.
   * @return the URL of the document the given {@code JApplet} is embedded.
   */
  @RunsInEDT
  @Nullable public URL getDocumentBase(@Nonnull JApplet applet) {
    return documentBase(applet);
  }

  /**
   * Returns the value of the named parameter in the default {@code JApplet} in the HTML tag, or {@code null} if not
   * set.
   *
   * @param name a parameter name.
   * @return the value of the named parameter in the default {code JApplet} in the HTML tag, or {@code null} if not set.
   */
  @RunsInEDT
  @Override
  @Nullable public String getParameter(@Nullable String name) {
    return getParameter(applet, name);
  }

  /**
   * Returns the value of the named parameter in the given {@code JApplet} in the HTML tag, or {@code null} if not set.
   *
   * @param applet the given {@code JApplet}.
   * @param name a parameter name.
   * @return the value of the named parameter in the given {code JApplet} in the HTML tag, or {@code null} if not set.
   */
  @RunsInEDT
  public String getParameter(@Nonnull JApplet applet, @Nullable String name) {
    return parameter(applet, name);
  }

  /**
   * Indicates whether the default {@code JApplet} is active or not.
   *
   * @return {@code true} if the default {@code JApplet} is active; {@code false} otherwise.
   */
  @RunsInEDT
  @Override
  public boolean isActive() {
    return isActive(applet);
  }

  /**
   * Indicates whether the given {@code JApplet} is active or not.
   *
   * @param applet the given {@code JApplet}.
   * @return {@code true} if the given {@code JApplet} is active; {@code false} otherwise.
   */
  @RunsInEDT
  public boolean isActive(@Nonnull JApplet applet) {
    return active(applet);
  }

  /**
   * Returns the {@code JApplet} of the given its name in the {@code AppletContext}.
   *
   * @param name the name of the {@code JApplet}.
   * @return the {@code Applet} with the given name.
   */
  @RunsInEDT
  public Applet getApplet(@Nonnull String name) {
    return applet.getAppletContext().getApplet(name);
  }

  /**
   * @return the collection of {@code Applet}s within the {@code AppletContext}.
   */
  @RunsInEDT
  @Nonnull public Enumeration<Applet> getApplets() {
    Enumeration<Applet> applets = applet.getAppletContext().getApplets();
    return applets != null ? applets : enumeration(Collections.<Applet> emptyList());
  }
}
