/*
 *
 * Artifactory is a binaries repository manager.
 * Copyright (C) 2019 JFrog Ltd.
 *
 * Artifactory is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 * Artifactory is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with Artifactory.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.artifactory.log.logback;

import com.jfrog.sysconf.SysLayout;
import org.apache.commons.lang3.StringUtils;
import org.artifactory.api.context.ArtifactoryContext;
import org.artifactory.common.home.ArtifactoryHome;
import org.artifactory.common.property.ArtifactorySystemProperties;
import org.jfrog.common.logging.logback.LogbackContextConfigurator;
import org.jfrog.common.logging.logback.servlet.LoggerConfigInfo;

import java.io.File;
import java.util.Arrays;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.trimToEmpty;

/**
 * @author Yinon Avraham.
 */
public class ArtifactoryLoggerConfig extends LoggerConfigInfo {

    public static final String LOG_DIR_PROP = "log.dir";
    private static final String LOG_DIR_ARCHIVED_PROP = "log.dir.archived";

    private final File logDir;

    public ArtifactoryLoggerConfig(String contextId, File logDir, File logbackConfigFile) {
        super(contextId, logbackConfigFile);
        this.logDir = logDir;
    }

    @Override
    protected void configure(LogbackContextConfigurator configurator) {
        super.configure(configurator);
        configurator
                .property(ArtifactoryContext.CONTEXT_ID_PROP, normalizedContextId())
                .property(LOG_DIR_PROP, logDir.getAbsolutePath())
                .property(LOG_DIR_ARCHIVED_PROP, new File(logDir.getAbsolutePath(), "archived").getAbsolutePath());
        // add custom logback properties to be used by logback (pattern/file names) if defined for testing
        String customProps = getTestEnvCustomProp();
        if (StringUtils.isNotBlank(customProps)) {
            String[] customParams = customProps.split(",");
            Arrays.stream(customParams).forEach(
                    p -> configurator.property(p.split("::")[0], p.split("::")[1])
            );
        }
    }

    private String normalizedContextId() {
        String contextId = trimToEmpty(getContextId());
        contextId = "artifactory".equalsIgnoreCase(contextId) ? "" : contextId + " ";
        return isBlank(contextId) ? "" : contextId.toLowerCase();
    }

    private String getTestEnvCustomProp() {
        // If this is a test env then system.properties is already in place and we can parse it.
        // Another safe assumption at this point is that homeDir is the parent of the parent of logDir (since
        //var/log/artifactory)
        File homeDir = logDir.getParentFile().getParentFile();
        if (homeDir == null || !homeDir.exists()) {
            //homeDir not present for some reason, but no need to dally over it.
            return "";
        }
        SysLayout layout = new SysLayout(homeDir, ArtifactoryHome.ARTIFACTORY_SERVICE_NAME);
        File systemProps = new File(layout.getServiceEtc(), ArtifactoryHome.ARTIFACTORY_SYSTEM_PROPERTIES_FILE);
        if (!systemProps.exists()) {
            // artifactory.system.properties file doesn't exist, so this is not a test env and we don't care
            return "";
        }
        //Attempt loading system props here, but don't crash over it.
        try {
            ArtifactorySystemProperties properties = new ArtifactorySystemProperties();
            properties.loadArtifactorySystemProperties(systemProps, null);
            return properties.getProperty("artifactory.test.logback.custom", "");
        } catch (Exception e) {
            //     ¯\_(:|)_/¯
        }
        return "";
    }
}
