/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.common.cloud;

import java.io.File;
import java.io.IOException;
import java.lang.invoke.MethodHandles;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Locale;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.cloud.ZkConfigManager;
import org.apache.solr.common.cloud.ZkCopier;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZkMaintenanceUtils {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final String ZKNODE_DATA_FILE = "zknode.data";

    private ZkMaintenanceUtils() {
    }

    public static String listZnode(SolrZkClient zkClient, String path, Boolean recurse) throws KeeperException, InterruptedException, SolrServerException {
        String root = path;
        if (path.toLowerCase(Locale.ROOT).startsWith("zk:")) {
            root = path.substring(3);
        }
        if (!root.equals("/") && root.endsWith("/")) {
            root = root.substring(0, root.length() - 1);
        }
        StringBuilder sb = new StringBuilder();
        if (!recurse.booleanValue()) {
            for (String node : zkClient.getChildren(root, null, true)) {
                if (node.equals("zookeeper")) continue;
                sb.append(node).append(System.lineSeparator());
            }
            return sb.toString();
        }
        ZkMaintenanceUtils.traverseZkTree(zkClient, root, VISIT_ORDER.VISIT_PRE, znode -> {
            if (znode.startsWith("/zookeeper")) {
                return;
            }
            int iPos = znode.lastIndexOf("/");
            if (iPos > 0) {
                for (int idx = 0; idx < iPos; ++idx) {
                    sb.append(" ");
                }
                sb.append(znode.substring(iPos + 1)).append(System.lineSeparator());
            } else {
                sb.append(znode).append(System.lineSeparator());
            }
        });
        return sb.toString();
    }

    public static void zkTransfer(SolrZkClient zkClient, String src, Boolean srcIsZk, String dst, Boolean dstIsZk, Boolean recurse) throws SolrServerException, KeeperException, InterruptedException, IOException {
        if (!srcIsZk.booleanValue() && !dstIsZk.booleanValue()) {
            throw new SolrServerException("One or both of source or destination must specify ZK nodes.");
        }
        if (!recurse.booleanValue()) {
            if (srcIsZk.booleanValue()) {
                if (zkClient.getChildren(src, null, true).size() != 0) {
                    throw new SolrServerException("Zookeeper node " + src + " has children and recurse is false");
                }
            } else if (Files.isDirectory(Paths.get(src, new String[0]), new LinkOption[0])) {
                throw new SolrServerException("Local path " + Paths.get(src, new String[0]).toAbsolutePath() + " is a directory and recurse is false");
            }
        }
        if (dstIsZk.booleanValue() && dst.length() == 0) {
            dst = "/";
        }
        dst = ZkMaintenanceUtils.normalizeDest(src, dst, srcIsZk, dstIsZk);
        if (srcIsZk.booleanValue() && dstIsZk.booleanValue()) {
            ZkMaintenanceUtils.traverseZkTree(zkClient, src, VISIT_ORDER.VISIT_PRE, new ZkCopier(zkClient, src, dst));
            return;
        }
        if (dstIsZk.booleanValue()) {
            ZkMaintenanceUtils.uploadToZK(zkClient, Paths.get(src, new String[0]), dst, null);
            return;
        }
        if (zkClient.getChildren(src, null, true).size() > 0) {
            ZkMaintenanceUtils.downloadFromZK(zkClient, src, Paths.get(dst, new String[0]));
            return;
        }
        if (Files.isDirectory(Paths.get(dst, new String[0]), new LinkOption[0])) {
            if (!dst.endsWith(File.separator)) {
                dst = dst + File.separator;
            }
            dst = ZkMaintenanceUtils.normalizeDest(src, dst, srcIsZk, dstIsZk);
        }
        byte[] data = zkClient.getData(src, null, null, true);
        Path filename = Paths.get(dst, new String[0]);
        Files.createDirectories(filename.getParent(), new FileAttribute[0]);
        log.info("Writing file {}", (Object)filename);
        Files.write(filename, data, new OpenOption[0]);
    }

    private static String normalizeDest(String srcName, String dstName, boolean srcIsZk, boolean dstIsZk) {
        String srcSeparator;
        if (dstName.equals(".")) {
            return Paths.get(".", new String[0]).normalize().toAbsolutePath().toString();
        }
        String dstSeparator = dstIsZk ? "/" : File.separator;
        String string = srcSeparator = srcIsZk ? "/" : File.separator;
        if (dstName.endsWith(dstSeparator)) {
            int pos = srcName.lastIndexOf(srcSeparator);
            dstName = pos < 0 ? dstName + srcName : dstName + srcName.substring(pos + 1);
        }
        log.info("copying from '{}' to '{}'", (Object)srcName, (Object)dstName);
        return dstName;
    }

    public static void moveZnode(SolrZkClient zkClient, String src, String dst) throws SolrServerException, KeeperException, InterruptedException {
        String destName = ZkMaintenanceUtils.normalizeDest(src, dst, true, true);
        if (zkClient.getChildren(src, null, true).size() == 0) {
            zkClient.makePath(destName, false, true);
            zkClient.setData(destName, zkClient.getData(src, null, null, true), true);
        } else {
            ZkMaintenanceUtils.traverseZkTree(zkClient, src, VISIT_ORDER.VISIT_PRE, new ZkCopier(zkClient, src, destName));
        }
        ZkMaintenanceUtils.checkAllZnodesThere(zkClient, src, destName);
        ZkMaintenanceUtils.clean(zkClient, src);
    }

    private static void checkAllZnodesThere(SolrZkClient zkClient, String src, String dst) throws KeeperException, InterruptedException, SolrServerException {
        for (String node : zkClient.getChildren(src, null, true)) {
            if (!zkClient.exists(dst + "/" + node, true).booleanValue()) {
                throw new SolrServerException("mv command did not move node " + dst + "/" + node + " source left intact");
            }
            ZkMaintenanceUtils.checkAllZnodesThere(zkClient, src + "/" + node, dst + "/" + node);
        }
    }

    public static void downConfig(SolrZkClient zkClient, String confName, Path confPath) throws IOException {
        ZkConfigManager manager = new ZkConfigManager(zkClient);
        manager.downloadConfigDir(confName, confPath);
    }

    public static void upConfig(SolrZkClient zkClient, Path confPath, String confName) throws IOException {
        ZkConfigManager manager = new ZkConfigManager(zkClient);
        manager.uploadConfigDir(confPath, confName);
    }

    public static void clean(SolrZkClient zkClient, String path) throws InterruptedException, KeeperException {
        ZkMaintenanceUtils.traverseZkTree(zkClient, path, VISIT_ORDER.VISIT_POST, znode -> {
            try {
                if (!znode.equals("/")) {
                    try {
                        zkClient.delete(znode, -1, true);
                    }
                    catch (KeeperException.NotEmptyException e) {
                        ZkMaintenanceUtils.clean(zkClient, znode);
                    }
                }
            }
            catch (KeeperException.NoNodeException r) {
                return;
            }
        });
    }

    public static void clean(SolrZkClient zkClient, String path, Predicate<String> filter) throws InterruptedException, KeeperException {
        if (filter == null) {
            ZkMaintenanceUtils.clean(zkClient, path);
            return;
        }
        ArrayList<String> paths = new ArrayList<String>();
        ZkMaintenanceUtils.traverseZkTree(zkClient, path, VISIT_ORDER.VISIT_POST, znode -> {
            if (!znode.equals("/") && filter.test(znode)) {
                paths.add(znode);
            }
        });
        paths.sort(Comparator.comparingInt(String::length).reversed());
        for (String subpath : paths) {
            if (subpath.equals("/")) continue;
            try {
                zkClient.delete(subpath, -1, true);
            }
            catch (KeeperException.NoNodeException | KeeperException.NotEmptyException keeperException) {}
        }
    }

    public static void uploadToZK(final SolrZkClient zkClient, Path fromPath, final String zkPath, final Pattern filenameExclusions) throws IOException {
        Path rootPath;
        String path = fromPath.toString();
        if (path.endsWith("*")) {
            path = path.substring(0, path.length() - 1);
        }
        if (!Files.exists(rootPath = Paths.get(path, new String[0]), new LinkOption[0])) {
            throw new IOException("Path " + rootPath + " does not exist");
        }
        Files.walkFileTree(rootPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String filename = file.getFileName().toString();
                if (filenameExclusions != null && filenameExclusions.matcher(filename).matches()) {
                    log.info("uploadToZK skipping '{}' due to filenameExclusions '{}'", (Object)filename, (Object)filenameExclusions);
                    return FileVisitResult.CONTINUE;
                }
                String zkNode = ZkMaintenanceUtils.createZkNodeName(zkPath, rootPath, file);
                try {
                    if (file.toFile().getName().equals(ZkMaintenanceUtils.ZKNODE_DATA_FILE) && zkClient.exists(zkNode, true).booleanValue()) {
                        zkClient.setData(zkNode, file.toFile(), true);
                    } else {
                        zkClient.makePath(zkNode, file.toFile(), false, true);
                    }
                }
                catch (InterruptedException | KeeperException e) {
                    throw new IOException("Error uploading file " + file.toString() + " to zookeeper path " + zkNode, SolrZkClient.checkInterrupted(e));
                }
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (dir.getFileName().toString().startsWith(".")) {
                    return FileVisitResult.SKIP_SUBTREE;
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private static boolean isEphemeral(SolrZkClient zkClient, String zkPath) throws KeeperException, InterruptedException {
        Stat znodeStat = zkClient.exists(zkPath, null, true);
        return znodeStat.getEphemeralOwner() != 0L;
    }

    private static int copyDataDown(SolrZkClient zkClient, String zkPath, File file) throws IOException, KeeperException, InterruptedException {
        byte[] data = zkClient.getData(zkPath, null, null, true);
        if (data != null && data.length > 0) {
            log.info("Writing file {}", (Object)file);
            Files.write(file.toPath(), data, new OpenOption[0]);
            return data.length;
        }
        return 0;
    }

    public static void downloadFromZK(SolrZkClient zkClient, String zkPath, Path file) throws IOException {
        try {
            List<String> children = zkClient.getChildren(zkPath, null, true);
            if (children.size() == 0) {
                if (ZkMaintenanceUtils.copyDataDown(zkClient, zkPath, file.toFile()) == 0) {
                    Files.createFile(file, new FileAttribute[0]);
                }
            } else {
                Files.createDirectories(file, new FileAttribute[0]);
                ZkMaintenanceUtils.copyDataDown(zkClient, zkPath, new File(file.toFile(), ZKNODE_DATA_FILE));
                for (String child : children) {
                    String zkChild = zkPath;
                    if (!zkChild.endsWith("/")) {
                        zkChild = zkChild + "/";
                    }
                    if (ZkMaintenanceUtils.isEphemeral(zkClient, zkChild = zkChild + child)) continue;
                    ZkMaintenanceUtils.downloadFromZK(zkClient, zkChild, file.resolve(child));
                }
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error downloading files from zookeeper path " + zkPath + " to " + file.toString(), SolrZkClient.checkInterrupted(e));
        }
    }

    public static void traverseZkTree(SolrZkClient zkClient, String path, VISIT_ORDER visitOrder, ZkVisitor visitor) throws InterruptedException, KeeperException {
        List<String> children;
        if (visitOrder == VISIT_ORDER.VISIT_PRE) {
            visitor.visit(path);
        }
        try {
            children = zkClient.getChildren(path, null, true);
        }
        catch (KeeperException.NoNodeException r) {
            return;
        }
        for (String string : children) {
            if (path.equals("/") && string.equals("zookeeper") || path.startsWith("/zookeeper")) continue;
            if (path.equals("/")) {
                ZkMaintenanceUtils.traverseZkTree(zkClient, path + string, visitOrder, visitor);
                continue;
            }
            ZkMaintenanceUtils.traverseZkTree(zkClient, path + "/" + string, visitOrder, visitor);
        }
        if (visitOrder == VISIT_ORDER.VISIT_POST) {
            visitor.visit(path);
        }
    }

    public static String getZkParent(String path) {
        if (StringUtils.endsWith(path, "/")) {
            path = StringUtils.substringBeforeLast(path, "/");
        }
        if (!StringUtils.contains(path, "/")) {
            return "";
        }
        return StringUtils.substringBeforeLast(path, "/");
    }

    public static String createZkNodeName(String zkRoot, Path root, Path file) {
        String relativePath = root.relativize(file).toString();
        if ("\\".equals(File.separator)) {
            relativePath = relativePath.replaceAll("\\\\", "/");
        }
        boolean isNonLeafData = file.toFile().getName().equals(ZKNODE_DATA_FILE);
        if (relativePath.length() == 0 && !isNonLeafData) {
            return zkRoot;
        }
        if (!zkRoot.endsWith("/")) {
            zkRoot = zkRoot + "/";
        }
        String ret = zkRoot + relativePath;
        if (isNonLeafData && (ret = ret.substring(0, ret.indexOf(ZKNODE_DATA_FILE))).endsWith("/")) {
            ret = ret.substring(0, ret.length() - 1);
        }
        return ret;
    }

    public static enum VISIT_ORDER {
        VISIT_PRE,
        VISIT_POST;

    }

    @FunctionalInterface
    public static interface ZkVisitor {
        public void visit(String var1) throws InterruptedException, KeeperException;
    }
}

