/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tez.client;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.ServiceException;
import java.io.IOException;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.Credentials;
import org.apache.hadoop.yarn.api.ApplicationConstants;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.ApplicationReport;
import org.apache.hadoop.yarn.api.records.ApplicationSubmissionContext;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.util.Apps;
import org.apache.tez.client.PreWarmContext;
import org.apache.tez.client.TezClientUtils;
import org.apache.tez.client.TezSessionConfiguration;
import org.apache.tez.client.TezSessionStatus;
import org.apache.tez.dag.api.DAG;
import org.apache.tez.dag.api.DAGSubmissionTimedOut;
import org.apache.tez.dag.api.DagTypeConverters;
import org.apache.tez.dag.api.SessionNotRunning;
import org.apache.tez.dag.api.TezException;
import org.apache.tez.dag.api.Vertex;
import org.apache.tez.dag.api.client.DAGClient;
import org.apache.tez.dag.api.client.rpc.DAGClientAMProtocolBlockingPB;
import org.apache.tez.dag.api.client.rpc.DAGClientAMProtocolRPC;
import org.apache.tez.dag.api.client.rpc.DAGClientRPCImpl;
import org.apache.tez.dag.api.records.DAGProtos;

public class TezSession {
    private static final Log LOG = LogFactory.getLog(TezSession.class);
    private final String sessionName;
    private ApplicationId applicationId;
    private final TezSessionConfiguration sessionConfig;
    private YarnClient yarnClient;
    private boolean sessionStarted = false;
    private boolean sessionStopped = false;
    private Credentials sessionCredentials = new Credentials();
    private long clientTimeout;

    public TezSession(String sessionName, ApplicationId applicationId, TezSessionConfiguration sessionConfig) {
        this.sessionName = sessionName;
        this.sessionConfig = sessionConfig;
        this.applicationId = applicationId;
    }

    public TezSession(String sessionName, TezSessionConfiguration sessionConfig) {
        this(sessionName, null, sessionConfig);
    }

    public synchronized void start() throws TezException, IOException {
        this.yarnClient = YarnClient.createYarnClient();
        this.yarnClient.init((Configuration)this.sessionConfig.getYarnConfiguration());
        this.yarnClient.start();
        Map<String, LocalResource> tezJarResources = TezClientUtils.setupTezJarsLocalResources(this.sessionConfig.getTezConfiguration(), this.sessionCredentials);
        this.clientTimeout = this.sessionConfig.getTezConfiguration().getInt("tez.session.client.timeout.secs", 120);
        if (this.sessionConfig.getSessionResources() != null && !this.sessionConfig.getSessionResources().isEmpty()) {
            tezJarResources.putAll(this.sessionConfig.getSessionResources());
        }
        try {
            if (this.applicationId == null) {
                this.applicationId = this.yarnClient.createApplication().getNewApplicationResponse().getApplicationId();
            }
            ApplicationSubmissionContext appContext = TezClientUtils.createApplicationSubmissionContext(this.sessionConfig.getTezConfiguration(), this.applicationId, null, this.sessionName, this.sessionConfig.getAMConfiguration(), tezJarResources, this.sessionCredentials);
            appContext.setMaxAppAttempts(1);
            this.yarnClient.submitApplication(appContext);
        }
        catch (YarnException e) {
            throw new TezException(e);
        }
        this.sessionStarted = true;
    }

    public synchronized DAGClient submitDAG(DAG dag) throws TezException, IOException, InterruptedException {
        String dagId;
        this.verifySessionStateForSubmission();
        LOG.info((Object)("Submitting dag to TezSession, sessionName=" + this.sessionName + ", applicationId=" + this.applicationId));
        TezClientUtils.setupDAGCredentials(dag, this.sessionCredentials, this.sessionConfig.getTezConfiguration());
        String classpath = TezClientUtils.getFrameworkClasspath((Configuration)this.sessionConfig.getYarnConfiguration());
        for (Vertex v : dag.getVertices()) {
            Map<String, String> taskEnv = v.getTaskEnvironment();
            Apps.addToEnvironment(taskEnv, (String)ApplicationConstants.Environment.CLASSPATH.name(), (String)classpath);
        }
        DAGProtos.DAGPlan dagPlan = dag.createDag(this.sessionConfig.getTezConfiguration());
        DAGClientAMProtocolRPC.SubmitDAGRequestProto requestProto = DAGClientAMProtocolRPC.SubmitDAGRequestProto.newBuilder().setDAGPlan(dagPlan).build();
        DAGClientAMProtocolBlockingPB proxy = this.waitForProxy();
        if (proxy == null) {
            try {
                LOG.warn((Object)"DAG submission to session timed out, stopping session");
                this.stop();
            }
            catch (Throwable t) {
                LOG.info((Object)"Got an exception when trying to stop session", t);
            }
            throw new DAGSubmissionTimedOut("Could not submit DAG to Tez Session, timed out after " + this.clientTimeout + " seconds");
        }
        try {
            dagId = proxy.submitDAG(null, requestProto).getDagId();
        }
        catch (ServiceException e) {
            throw new TezException(e);
        }
        LOG.info((Object)("Submitted dag to TezSession, sessionName=" + this.sessionName + ", applicationId=" + this.applicationId + ", dagId=" + dagId));
        return new DAGClientRPCImpl(this.applicationId, dagId, this.sessionConfig.getTezConfiguration());
    }

    public synchronized void stop() throws TezException, IOException {
        if (!this.sessionStarted) {
            LOG.info((Object)"Session not started. Ignoring stop command");
            return;
        }
        LOG.info((Object)("Shutting down Tez Session, sessionName=" + this.sessionName + ", applicationId=" + this.applicationId));
        this.sessionStopped = true;
        try {
            DAGClientAMProtocolBlockingPB proxy = TezClientUtils.getSessionAMProxy(this.yarnClient, (Configuration)this.sessionConfig.getYarnConfiguration(), this.applicationId);
            if (proxy != null) {
                DAGClientAMProtocolRPC.ShutdownSessionRequestProto request = DAGClientAMProtocolRPC.ShutdownSessionRequestProto.newBuilder().build();
                proxy.shutdownSession(null, request);
                return;
            }
        }
        catch (TezException e) {
            LOG.info((Object)"Failed to shutdown Tez Session via proxy", (Throwable)e);
        }
        catch (ServiceException e) {
            LOG.info((Object)"Failed to shutdown Tez Session via proxy", (Throwable)e);
        }
        LOG.info((Object)("Could not connect to AM, killing session via YARN, sessionName=" + this.sessionName + ", applicationId=" + this.applicationId));
        try {
            this.yarnClient.killApplication(this.applicationId);
        }
        catch (YarnException e) {
            throw new TezException(e);
        }
    }

    public String getSessionName() {
        return this.sessionName;
    }

    @InterfaceAudience.Private
    @VisibleForTesting
    public synchronized ApplicationId getApplicationId() {
        return this.applicationId;
    }

    public TezSessionStatus getSessionStatus() throws TezException, IOException {
        try {
            ApplicationReport appReport = this.yarnClient.getApplicationReport(this.applicationId);
            switch (appReport.getYarnApplicationState()) {
                case NEW: 
                case NEW_SAVING: 
                case ACCEPTED: 
                case SUBMITTED: {
                    return TezSessionStatus.INITIALIZING;
                }
                case FINISHED: 
                case FAILED: 
                case KILLED: {
                    return TezSessionStatus.SHUTDOWN;
                }
                case RUNNING: {
                    try {
                        DAGClientAMProtocolBlockingPB proxy = TezClientUtils.getSessionAMProxy(this.yarnClient, (Configuration)this.sessionConfig.getYarnConfiguration(), this.applicationId);
                        if (proxy == null) {
                            return TezSessionStatus.INITIALIZING;
                        }
                        DAGClientAMProtocolRPC.GetAMStatusResponseProto response = proxy.getAMStatus(null, DAGClientAMProtocolRPC.GetAMStatusRequestProto.newBuilder().build());
                        return DagTypeConverters.convertTezSessionStatusFromProto(response.getStatus());
                    }
                    catch (TezException e) {
                        LOG.info((Object)"Failed to retrieve AM Status via proxy", (Throwable)e);
                        break;
                    }
                    catch (ServiceException e) {
                        LOG.info((Object)"Failed to retrieve AM Status via proxy", (Throwable)e);
                    }
                }
            }
        }
        catch (YarnException e) {
            throw new TezException(e);
        }
        return TezSessionStatus.INITIALIZING;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    @InterfaceAudience.Private
    @InterfaceStability.Unstable
    public void preWarm(PreWarmContext context) throws IOException, TezException, InterruptedException {
        this.verifySessionStateForSubmission();
        try {
            DAGClientAMProtocolBlockingPB proxy = this.waitForProxy();
            if (proxy == null) {
                throw new SessionNotRunning("Could not connect to Session within client timeout interval, timeoutSecs=" + this.clientTimeout);
            }
            String classpath = TezClientUtils.getFrameworkClasspath((Configuration)this.sessionConfig.getYarnConfiguration());
            Map<String, String> contextEnv = context.getEnvironment();
            Apps.addToEnvironment(contextEnv, (String)ApplicationConstants.Environment.CLASSPATH.name(), (String)classpath);
            DAGClientAMProtocolRPC.PreWarmRequestProto.Builder preWarmReqProtoBuilder = DAGClientAMProtocolRPC.PreWarmRequestProto.newBuilder();
            preWarmReqProtoBuilder.setPreWarmContext(DagTypeConverters.convertPreWarmContextToProto(context));
            proxy.preWarm(null, preWarmReqProtoBuilder.build());
            try {
                TezSessionStatus status;
                do {
                    Thread.sleep(1000L);
                    status = this.getSessionStatus();
                    if (status.equals((Object)TezSessionStatus.READY)) return;
                } while (!status.equals((Object)TezSessionStatus.SHUTDOWN));
                throw new SessionNotRunning("Could not connect to Session");
            }
            catch (InterruptedException e) {
                return;
            }
        }
        catch (ServiceException e) {
            throw new TezException(e);
        }
    }

    private DAGClientAMProtocolBlockingPB waitForProxy() throws IOException, TezException, InterruptedException {
        long startTime = System.currentTimeMillis();
        long endTime = startTime + this.clientTimeout * 1000L;
        DAGClientAMProtocolBlockingPB proxy = null;
        while ((proxy = TezClientUtils.getSessionAMProxy(this.yarnClient, (Configuration)this.sessionConfig.getYarnConfiguration(), this.applicationId)) == null) {
            Thread.sleep(100L);
            if (this.clientTimeout == -1L || System.currentTimeMillis() <= endTime) continue;
            break;
        }
        return proxy;
    }

    private void verifySessionStateForSubmission() throws SessionNotRunning {
        if (!this.sessionStarted) {
            throw new SessionNotRunning("Session not started");
        }
        if (this.sessionStopped) {
            throw new SessionNotRunning("Session stopped");
        }
    }
}

