/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.solr.client.solrj.request;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.solr.client.solrj.InputStreamResponse;
import org.apache.solr.client.solrj.JacksonParsingResponse;
import org.apache.solr.client.solrj.ResponseParser;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.impl.InputStreamResponseParser;
import org.apache.solr.common.params.ModifiableSolrParams;
import org.apache.solr.common.params.SolrParams;

// WARNING: This class is generated from a Mustache template; any intended
// changes should be made to the underlying template and not this file directly.

/**
 * Experimental SolrRequest's and SolrResponse's for fileStore, generated from an OAS.
 *
 * <p>See individual request and response classes for more detailed and relevant information.
 *
 * <p>All SolrRequest implementations rely on v2 APIs which may require a SolrClient configured to
 * use the '/api' path prefix, instead of '/solr'.
 *
 * @lucene.experimental
 */
public class FileStoreApi {

  public static class DeleteFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.SolrJerseyResponse> {
    public DeleteFileResponse() {
      super(org.apache.solr.client.api.model.SolrJerseyResponse.class);
    }
  }

  public static class DeleteFile extends SolrRequest<DeleteFileResponse> {
    private final String path;
    private Boolean localDelete;

    /**
     * Create a DeleteFile request object.
     *
     * @param path Path param - Path to a file or directory within the filestore
     */
    public DeleteFile(String path) {
      super(
          SolrRequest.METHOD.valueOf("DELETE"),
          "/cluster/filestore/files{path}".replace("{" + "path" + "}", path));

      this.path = path;
    }

    /**
     * @param localDelete Indicates whether the deletion should only be done on the receiving node.
     *     For internal use only
     */
    public void setLocalDelete(Boolean localDelete) {
      this.localDelete = localDelete;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      if (localDelete != null) {

        params.add("localDelete", localDelete.toString());
      }
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      queryParams.add("localDelete");
      return queryParams;
    }

    @Override
    protected DeleteFileResponse createResponse(SolrClient client) {
      return new DeleteFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class FetchFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.SolrJerseyResponse> {
    public FetchFileResponse() {
      super(org.apache.solr.client.api.model.SolrJerseyResponse.class);
    }
  }

  public static class FetchFile extends SolrRequest<FetchFileResponse> {
    private final String path;
    private String getFrom;

    /**
     * Create a FetchFile request object.
     *
     * @param path Path param - Path to a file or directory within the filestore
     */
    public FetchFile(String path) {
      super(
          SolrRequest.METHOD.valueOf("POST"),
          "/cluster/filestore/commands/fetch{path}".replace("{" + "path" + "}", path));

      this.path = path;
    }

    /**
     * @param getFrom An optional Solr node name to fetch the file from
     */
    public void setGetFrom(String getFrom) {
      this.getFrom = getFrom;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      if (getFrom != null) {

        params.add("getFrom", getFrom);
      }
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      queryParams.add("getFrom");
      return queryParams;
    }

    @Override
    protected FetchFileResponse createResponse(SolrClient client) {
      return new FetchFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class GetFileResponse extends InputStreamResponse {}

  public static class GetFile extends SolrRequest<GetFileResponse> {
    private final String filePath;

    /**
     * Create a GetFile request object.
     *
     * @param filePath Path param - Path to a file or directory within the filestore
     */
    public GetFile(String filePath) {
      super(
          SolrRequest.METHOD.valueOf("GET"),
          "/cluster/filestore/files{filePath}".replace("{" + "filePath" + "}", filePath));

      this.filePath = filePath;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      return queryParams;
    }

    @Override
    protected GetFileResponse createResponse(SolrClient client) {
      return new GetFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class GetMetadataResponse
      extends JacksonParsingResponse<
          org.apache.solr.client.api.model.FileStoreDirectoryListingResponse> {
    public GetMetadataResponse() {
      super(org.apache.solr.client.api.model.FileStoreDirectoryListingResponse.class);
    }
  }

  public static class GetMetadata extends SolrRequest<GetMetadataResponse> {
    private final String path;

    /**
     * Create a GetMetadata request object.
     *
     * @param path Path param - Path to a file or directory within the filestore
     */
    public GetMetadata(String path) {
      super(
          SolrRequest.METHOD.valueOf("GET"),
          "/cluster/filestore/metadata{path}".replace("{" + "path" + "}", path));

      this.path = path;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      return queryParams;
    }

    @Override
    protected GetMetadataResponse createResponse(SolrClient client) {
      return new GetMetadataResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class SyncFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.SolrJerseyResponse> {
    public SyncFileResponse() {
      super(org.apache.solr.client.api.model.SolrJerseyResponse.class);
    }
  }

  public static class SyncFile extends SolrRequest<SyncFileResponse> {
    private final String path;

    /**
     * Create a SyncFile request object.
     *
     * @param path Path param - Path to a file or directory within the filestore
     */
    public SyncFile(String path) {
      super(
          SolrRequest.METHOD.valueOf("POST"),
          "/cluster/filestore/commands/sync{path}".replace("{" + "path" + "}", path));

      this.path = path;
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      return queryParams;
    }

    @Override
    protected SyncFileResponse createResponse(SolrClient client) {
      return new SyncFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }

  public static class UploadFileResponse
      extends JacksonParsingResponse<org.apache.solr.client.api.model.UploadToFileStoreResponse> {
    public UploadFileResponse() {
      super(org.apache.solr.client.api.model.UploadToFileStoreResponse.class);
    }
  }

  public static class UploadFile extends SolrRequest<UploadFileResponse> {
    private final Object requestBody;
    private final String filePath;
    private List<String> sig;

    /**
     * Create a UploadFile request object.
     *
     * @param filePath Path param - File store path
     */
    public UploadFile(String filePath, Object requestBody) {
      super(
          SolrRequest.METHOD.valueOf("PUT"),
          "/cluster/filestore/files{filePath}".replace("{" + "filePath" + "}", filePath));

      this.filePath = filePath;
      this.requestBody = requestBody;
    }

    /**
     * @param sig Signature(s) for the file being uploaded
     */
    public void setSig(List<String> sig) {
      this.sig = sig;
    }

    @Override
    @SuppressWarnings("unchecked")
    public RequestWriter.ContentWriter getContentWriter(String _expectedTypeIGNORE) {
      // v2 only supports JSON request bodies, so we ignore this type coming from the RequestWriter
      return new RequestWriter.ContentWriter() {
        @Override
        public void write(OutputStream os) throws IOException {
          ((InputStream) requestBody).transferTo(os);
        }

        @Override
        public String getContentType() {
          return "application/octet-stream";
        }
      };
    }

    // TODO Hardcode this for now, but in reality we'll want to parse this out of the Operation data
    // somehow
    @Override
    public String getRequestType() {
      return SolrRequestType.ADMIN.toString();
    }

    @Override
    public ApiVersion getApiVersion() {
      return ApiVersion.V2;
    }

    @Override
    public SolrParams getParams() {
      final ModifiableSolrParams params = new ModifiableSolrParams();
      if (sig != null) {
        sig.forEach(v -> params.add("sig", v));
      }
      return params;
    }

    @Override
    public Set<String> getQueryParams() {
      final var queryParams = new HashSet<String>();
      queryParams.add("sig");
      return queryParams;
    }

    @Override
    protected UploadFileResponse createResponse(SolrClient client) {
      return new UploadFileResponse();
    }

    @Override
    public ResponseParser getResponseParser() {
      return new InputStreamResponseParser("json");
    }
  }
}
