/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.shindig.gadgets.rewrite;

import com.google.inject.Inject;
import org.apache.shindig.common.uri.Uri;
import org.apache.shindig.gadgets.Gadget;
import org.apache.shindig.gadgets.uri.ProxyUriManager;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import java.util.List;

/**
 * Visitor that replaces urls (@import + background) in
 * &lt;style&gt; ... &lt;/style&gt; with their proxied versions.
 *
 * @since 2.0.0
 */
public class StyleTagProxyEmbeddedUrlsVisitor implements DomWalker.Visitor {
  protected final ContentRewriterFeature.Config config;
  protected final ProxyUriManager proxyUriManager;
  protected final CssResponseRewriter cssRewriter;

  @Inject
  public StyleTagProxyEmbeddedUrlsVisitor(ContentRewriterFeature.Config config,
                                          ProxyUriManager proxyUriManager,
                                          CssResponseRewriter cssRewriter) {
    this.config = config;
    this.proxyUriManager = proxyUriManager;
    this.cssRewriter = cssRewriter;
  }

  public VisitStatus visit(Gadget gadget, Node node) throws RewritingException {
    // Only process <style> elements.
    if (node.getNodeType() != Node.ELEMENT_NODE ||
        !node.getNodeName().equalsIgnoreCase("style")) {
      return VisitStatus.BYPASS;
    }

    return VisitStatus.RESERVE_NODE;
  }

  public boolean revisit(Gadget gadget, List<Node> nodes) throws RewritingException {
    Uri contentBase = gadget.getSpec().getUrl();

    for (Node node: nodes) {
      Element elem = (Element) node;
      cssRewriter.rewrite(
          elem, contentBase,
          CssResponseRewriter.uriMaker(proxyUriManager, config), false, gadget.getContext());
    }
    return !nodes.isEmpty();
  }
}
