/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.controller.util;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Preconditions;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.helix.PropertyPathBuilder;
import org.apache.helix.manager.zk.ZKHelixAdmin;
import org.apache.helix.store.zk.ZkHelixPropertyStore;
import org.apache.helix.zookeeper.datamodel.ZNRecord;
import org.apache.helix.zookeeper.datamodel.serializer.ZNRecordSerializer;
import org.apache.helix.zookeeper.zkclient.serialize.ZkSerializer;
import org.apache.pinot.common.metadata.ZKMetadataProvider;
import org.apache.pinot.spi.config.table.IndexingConfig;
import org.apache.pinot.spi.config.table.TableConfig;
import org.apache.pinot.spi.data.DateTimeFieldSpec;
import org.apache.pinot.spi.data.FieldSpec;
import org.apache.pinot.spi.data.Schema;
import org.apache.pinot.spi.utils.JsonUtils;
import org.apache.pinot.spi.utils.builder.ControllerRequestURLBuilder;
import org.apache.pinot.spi.utils.builder.TableNameBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AutoAddInvertedIndex {
    public static final long DEFAULT_TABLE_SIZE_THRESHOLD = 10000000L;
    public static final long DEFAULT_CARDINALITY_THRESHOLD = 100L;
    public static final int DEFAULT_MAX_NUM_INVERTED_INDEX_ADDED = 2;
    private static final Logger LOGGER = LoggerFactory.getLogger(AutoAddInvertedIndex.class);
    private final String _clusterName;
    private final String _controllerAddress;
    private final String _brokerAddress;
    private final ZKHelixAdmin _helixAdmin;
    private final ZkHelixPropertyStore<ZNRecord> _propertyStore;
    private final Strategy _strategy;
    private final Mode _mode;
    private String _tableNamePattern = null;
    private long _tableSizeThreshold = 10000000L;
    private long _cardinalityThreshold = 100L;
    private int _maxNumInvertedIndexAdded = 2;

    public AutoAddInvertedIndex(String zkAddress, String clusterName, String controllerAddress, String brokerAddress, Strategy strategy, Mode mode) {
        this._clusterName = clusterName;
        this._controllerAddress = controllerAddress;
        this._brokerAddress = brokerAddress;
        this._helixAdmin = new ZKHelixAdmin(zkAddress);
        this._propertyStore = new ZkHelixPropertyStore(zkAddress, (ZkSerializer)new ZNRecordSerializer(), PropertyPathBuilder.propertyStore((String)clusterName));
        this._strategy = strategy;
        this._mode = mode;
    }

    public void overrideDefaultSettings(String tableNamePattern, long tableSizeThreshold, long cardinalityThreshold, int maxNumInvertedIndex) {
        this._tableNamePattern = tableNamePattern;
        this._tableSizeThreshold = tableSizeThreshold;
        this._cardinalityThreshold = cardinalityThreshold;
        this._maxNumInvertedIndexAdded = maxNumInvertedIndex;
    }

    public void run() throws Exception {
        if (this._strategy != Strategy.QUERY) {
            throw new IllegalStateException("Invalid Strategy: " + this._strategy);
        }
        this.runQueryStrategy();
    }

    private void runQueryStrategy() throws Exception {
        List resourcesInCluster = this._helixAdmin.getResourcesInCluster(this._clusterName);
        for (String tableNameWithType : resourcesInCluster) {
            Schema tableSchema;
            if (!TableNameBuilder.isTableResource((String)tableNameWithType) || this._tableNamePattern != null && !tableNameWithType.matches(this._tableNamePattern)) continue;
            LOGGER.info("Table: {} matches the table name pattern: {}", (Object)tableNameWithType, (Object)this._tableNamePattern);
            TableConfig tableConfig = ZKMetadataProvider.getTableConfig(this._propertyStore, (String)tableNameWithType);
            Preconditions.checkNotNull((Object)tableConfig);
            IndexingConfig indexingConfig = tableConfig.getIndexingConfig();
            ArrayList<String> invertedIndexColumns = indexingConfig.getInvertedIndexColumns();
            boolean autoGeneratedInvertedIndex = indexingConfig.isAutoGeneratedInvertedIndex();
            if (autoGeneratedInvertedIndex) {
                Preconditions.checkState((!invertedIndexColumns.isEmpty() ? 1 : 0) != 0, (Object)"Auto-generated inverted index list is empty");
                if (this._mode == Mode.NEW) {
                    LOGGER.info("Table: {}, skip adding inverted index because it has auto-generated inverted index and under NEW mode", (Object)tableNameWithType);
                    continue;
                }
                if (this._mode == Mode.REMOVE) {
                    invertedIndexColumns.clear();
                    indexingConfig.setAutoGeneratedInvertedIndex(false);
                    if (this.updateIndexConfig(tableNameWithType, tableConfig)) {
                        LOGGER.info("Table: {}, removed auto-generated inverted index", (Object)tableNameWithType);
                        continue;
                    }
                    LOGGER.error("Table: {}, failed to remove auto-generated inverted index", (Object)tableNameWithType);
                    continue;
                }
                if (this._mode == Mode.REFRESH) {
                    invertedIndexColumns.clear();
                }
            } else {
                int emptyStringIndex;
                if (invertedIndexColumns == null) {
                    invertedIndexColumns = new ArrayList<String>();
                    indexingConfig.setInvertedIndexColumns(invertedIndexColumns);
                }
                while ((emptyStringIndex = invertedIndexColumns.indexOf("")) != -1) {
                    invertedIndexColumns.remove(emptyStringIndex);
                }
                if (!invertedIndexColumns.isEmpty()) {
                    LOGGER.info("Table: {}, skip adding inverted index because it has non-auto-generated inverted index", (Object)tableNameWithType);
                    continue;
                }
            }
            if ((tableSchema = ZKMetadataProvider.getTableSchema(this._propertyStore, (String)tableNameWithType)) == null) {
                LOGGER.info("Table: {}, skip adding inverted index because it does not have a schema", (Object)tableNameWithType);
                continue;
            }
            List dimensionNames = tableSchema.getDimensionNames();
            if (dimensionNames.isEmpty()) {
                LOGGER.info("Table: {}, skip adding inverted index because it does not have any dimension column", (Object)tableNameWithType);
                continue;
            }
            String timeColumnName = tableConfig.getValidationConfig().getTimeColumnName();
            if (timeColumnName == null) {
                LOGGER.info("Table: {}, skip adding inverted index because it does not have a time column specified in the table config", (Object)tableNameWithType);
                continue;
            }
            DateTimeFieldSpec dateTimeSpec = tableSchema.getSpecForTimeColumn(timeColumnName);
            if (dateTimeSpec == null || dateTimeSpec.getDataType() == FieldSpec.DataType.STRING) {
                LOGGER.info("Table: {}, skip adding inverted index because it does not have a numeric time column", (Object)tableNameWithType);
                continue;
            }
            TimeUnit timeUnit = dateTimeSpec.getFormatSpec().getColumnUnit();
            if (timeUnit != TimeUnit.DAYS) {
                LOGGER.warn("Table: {}, time column {] has non-DAYS time unit: {}", (Object)timeColumnName, (Object)timeUnit);
            }
            JsonNode queryResponse = this.sendQuery("SELECT COUNT(*) FROM " + tableNameWithType);
            long numTotalDocs = queryResponse.get("totalDocs").asLong();
            LOGGER.info("Table: {}, number of total documents: {}", (Object)tableNameWithType, (Object)numTotalDocs);
            if (numTotalDocs <= this._tableSizeThreshold) {
                LOGGER.info("Table: {}, skip adding inverted index because the table is too small", (Object)tableNameWithType);
                continue;
            }
            queryResponse = this.sendQuery("SELECT Max(" + timeColumnName + ") FROM " + tableNameWithType);
            long maxTimeStamp = queryResponse.get("aggregationResults").get(0).get("value").asLong();
            LOGGER.info("Table: {}, max time column {}: {}", new Object[]{tableNameWithType, timeColumnName, maxTimeStamp});
            ArrayList<ResultPair> resultPairs = new ArrayList<ResultPair>();
            for (String dimensionName : dimensionNames) {
                String query = "SELECT DISTINCTCOUNT(" + dimensionName + ") FROM " + tableNameWithType + " WHERE " + timeColumnName + " = " + maxTimeStamp;
                queryResponse = this.sendQuery(query);
                JsonNode result = queryResponse.get("aggregationResults").get(0);
                resultPairs.add(new ResultPair(result.get("function").asText().substring("distinctCount_".length()), result.get("value").asLong()));
            }
            Collections.sort(resultPairs);
            int numInvertedIndex = Math.min(this._maxNumInvertedIndexAdded, resultPairs.size());
            for (int i = 0; i < numInvertedIndex; ++i) {
                ResultPair resultPair = (ResultPair)resultPairs.get(i);
                String columnName = resultPair._key;
                long cardinality = resultPair._value;
                if (cardinality > this._cardinalityThreshold) {
                    if (!invertedIndexColumns.contains(columnName)) {
                        invertedIndexColumns.add(columnName);
                    }
                } else {
                    LOGGER.info("Table: {}, skip adding inverted index to column {} with cardinality: {}", new Object[]{tableNameWithType, columnName, cardinality});
                    break;
                }
                LOGGER.info("Table: {}, add inverted index to column {} with cardinality: {}", new Object[]{tableNameWithType, columnName, cardinality});
            }
            if (!invertedIndexColumns.isEmpty()) {
                indexingConfig.setAutoGeneratedInvertedIndex(true);
                if (this.updateIndexConfig(tableNameWithType, tableConfig)) {
                    LOGGER.info("Table: {}, added inverted index to columns: {}", (Object)tableNameWithType, invertedIndexColumns);
                    continue;
                }
                LOGGER.error("Table: {}, failed to add inverted index to columns: {}", (Object)tableNameWithType, invertedIndexColumns);
                continue;
            }
            if (!autoGeneratedInvertedIndex) continue;
            Preconditions.checkState((this._mode == Mode.REFRESH ? 1 : 0) != 0);
            indexingConfig.setAutoGeneratedInvertedIndex(false);
            if (this.updateIndexConfig(tableNameWithType, tableConfig)) {
                LOGGER.info("Table: {}, removed auto-generated inverted index", (Object)tableNameWithType);
                continue;
            }
            LOGGER.error("Table: {}, failed to remove auto-generated inverted index", (Object)tableNameWithType);
        }
    }

    private JsonNode sendQuery(String query) throws Exception {
        URLConnection urlConnection = new URL("http://" + this._brokerAddress + "/query").openConnection();
        urlConnection.setDoOutput(true);
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(urlConnection.getOutputStream(), "UTF-8"));
        writer.write(JsonUtils.newObjectNode().put("sql", query).toString());
        writer.flush();
        BufferedReader reader = new BufferedReader(new InputStreamReader(urlConnection.getInputStream(), "UTF-8"));
        return JsonUtils.stringToJsonNode((String)reader.readLine());
    }

    private boolean updateIndexConfig(String tableName, TableConfig tableConfig) throws Exception {
        String request = ControllerRequestURLBuilder.baseUrl((String)("http://" + this._controllerAddress)).forTableUpdateIndexingConfigs(tableName);
        HttpURLConnection httpURLConnection = (HttpURLConnection)new URL(request).openConnection();
        httpURLConnection.setDoOutput(true);
        httpURLConnection.setRequestMethod("PUT");
        BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(httpURLConnection.getOutputStream(), "UTF-8"));
        writer.write(tableConfig.toJsonString());
        writer.flush();
        BufferedReader reader = new BufferedReader(new InputStreamReader(httpURLConnection.getInputStream(), "UTF-8"));
        return reader.readLine().equals("done");
    }

    private static class ResultPair
    implements Comparable<ResultPair> {
        private final String _key;
        private final long _value;

        public ResultPair(String key, long value) {
            this._key = key;
            this._value = value;
        }

        @Override
        public int compareTo(ResultPair o) {
            return Long.compare(o._value, this._value);
        }

        public String toString() {
            return this._key + ": " + this._value;
        }
    }

    public static enum Mode {
        NEW,
        REMOVE,
        REFRESH,
        APPEND;

    }

    public static enum Strategy {
        QUERY;

    }
}

