/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.controller.helix.core.minion;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.Uninterruptibles;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import org.apache.helix.task.JobConfig;
import org.apache.helix.task.JobContext;
import org.apache.helix.task.JobQueue;
import org.apache.helix.task.TaskConfig;
import org.apache.helix.task.TaskDriver;
import org.apache.helix.task.TaskPartitionState;
import org.apache.helix.task.TaskState;
import org.apache.helix.task.WorkflowConfig;
import org.apache.helix.task.WorkflowContext;
import org.apache.pinot.common.utils.DateTimeUtils;
import org.apache.pinot.core.minion.PinotTaskConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PinotHelixTaskResourceManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(PinotHelixTaskResourceManager.class);
    public static final String TASK_NAME_SEPARATOR = "_";
    private static final String TASK_QUEUE_PREFIX = "TaskQueue_";
    private static final String TASK_PREFIX = "Task_";
    private final TaskDriver _taskDriver;

    public PinotHelixTaskResourceManager(TaskDriver taskDriver) {
        this._taskDriver = taskDriver;
    }

    public synchronized Set<String> getTaskTypes() {
        Set helixJobQueues = this._taskDriver.getWorkflows().keySet();
        HashSet<String> taskTypes = new HashSet<String>(helixJobQueues.size());
        for (String helixJobQueue : helixJobQueues) {
            taskTypes.add(PinotHelixTaskResourceManager.getTaskType(helixJobQueue));
        }
        return taskTypes;
    }

    public void ensureTaskQueueExists(String taskType) {
        String helixJobQueueName = PinotHelixTaskResourceManager.getHelixJobQueueName(taskType);
        WorkflowConfig workflowConfig = this._taskDriver.getWorkflowConfig(helixJobQueueName);
        if (workflowConfig == null) {
            LOGGER.info("Creating task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
            JobQueue jobQueue = new JobQueue.Builder(helixJobQueueName).setWorkflowConfig(new WorkflowConfig.Builder().setParallelJobs(Integer.MAX_VALUE).build()).build();
            this._taskDriver.createQueue(jobQueue);
        }
        while (this._taskDriver.getWorkflowContext(helixJobQueueName) == null) {
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
        }
    }

    public synchronized void cleanUpTaskQueue(String taskType) {
        String helixJobQueueName = PinotHelixTaskResourceManager.getHelixJobQueueName(taskType);
        LOGGER.info("Cleaning up task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
        this._taskDriver.cleanupQueue(helixJobQueueName);
    }

    public synchronized void stopTaskQueue(String taskType) {
        String helixJobQueueName = PinotHelixTaskResourceManager.getHelixJobQueueName(taskType);
        LOGGER.info("Stopping task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
        this._taskDriver.stop(helixJobQueueName);
    }

    public synchronized void resumeTaskQueue(String taskType) {
        String helixJobQueueName = PinotHelixTaskResourceManager.getHelixJobQueueName(taskType);
        LOGGER.info("Resuming task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
        this._taskDriver.resume(helixJobQueueName);
    }

    public synchronized void deleteTaskQueue(String taskType, boolean forceDelete) {
        String helixJobQueueName = PinotHelixTaskResourceManager.getHelixJobQueueName(taskType);
        if (forceDelete) {
            LOGGER.warn("Force deleting task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
        } else {
            LOGGER.info("Deleting task queue: {} for task type: {}", (Object)helixJobQueueName, (Object)taskType);
        }
        this._taskDriver.delete(helixJobQueueName, forceDelete);
    }

    public synchronized Set<String> getTaskQueues() {
        return this._taskDriver.getWorkflows().keySet();
    }

    public synchronized TaskState getTaskQueueState(String taskType) {
        return this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType)).getWorkflowState();
    }

    public synchronized String submitTask(List<PinotTaskConfig> pinotTaskConfigs, long taskTimeoutMs, int numConcurrentTasksPerInstance) {
        return this.submitTask(pinotTaskConfigs, "minion_untagged", taskTimeoutMs, numConcurrentTasksPerInstance);
    }

    public synchronized String submitTask(List<PinotTaskConfig> pinotTaskConfigs, String minionInstanceTag, long taskTimeoutMs, int numConcurrentTasksPerInstance) {
        int numChildTasks = pinotTaskConfigs.size();
        Preconditions.checkState((numChildTasks > 0 ? 1 : 0) != 0);
        Preconditions.checkState((numConcurrentTasksPerInstance > 0 ? 1 : 0) != 0);
        String taskType = pinotTaskConfigs.get(0).getTaskType();
        String parentTaskName = TASK_PREFIX + taskType + TASK_NAME_SEPARATOR + System.currentTimeMillis();
        LOGGER.info("Submitting parent task: {} of type: {} with {} child task configs: {} to Minion instances with tag: {}", new Object[]{parentTaskName, taskType, numChildTasks, pinotTaskConfigs, minionInstanceTag});
        ArrayList<TaskConfig> helixTaskConfigs = new ArrayList<TaskConfig>(numChildTasks);
        for (int i = 0; i < numChildTasks; ++i) {
            PinotTaskConfig pinotTaskConfig = pinotTaskConfigs.get(i);
            Preconditions.checkState((boolean)pinotTaskConfig.getTaskType().equals(taskType));
            helixTaskConfigs.add(pinotTaskConfig.toHelixTaskConfig(parentTaskName + TASK_NAME_SEPARATOR + i));
        }
        JobConfig.Builder jobBuilder = new JobConfig.Builder().addTaskConfigs(helixTaskConfigs).setInstanceGroupTag(minionInstanceTag).setTimeoutPerTask(taskTimeoutMs).setNumConcurrentTasksPerInstance(numConcurrentTasksPerInstance).setIgnoreDependentJobFailure(true).setMaxAttemptsPerTask(1).setFailureThreshold(Integer.MAX_VALUE);
        this._taskDriver.enqueueJob(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType), parentTaskName, jobBuilder);
        while (this.getTaskState(parentTaskName) == null) {
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
        }
        return parentTaskName;
    }

    public synchronized Set<String> getTasks(String taskType) {
        Set helixJobs = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType)).getJobStates().keySet();
        HashSet<String> tasks = new HashSet<String>(helixJobs.size());
        for (String helixJobName : helixJobs) {
            tasks.add(PinotHelixTaskResourceManager.getPinotTaskName(helixJobName));
        }
        return tasks;
    }

    public synchronized Map<String, TaskState> getTaskStates(String taskType) {
        Map<String, Object> helixJobStates = new HashMap<String, TaskState>();
        WorkflowContext workflowContext = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType));
        if (workflowContext == null) {
            return helixJobStates;
        }
        helixJobStates = workflowContext.getJobStates();
        HashMap<String, TaskState> taskStates = new HashMap<String, TaskState>(helixJobStates.size());
        for (Map.Entry<String, Object> entry : helixJobStates.entrySet()) {
            taskStates.put(PinotHelixTaskResourceManager.getPinotTaskName(entry.getKey()), (TaskState)entry.getValue());
        }
        return taskStates;
    }

    public synchronized TaskCount getTaskCount(String parentTaskName) {
        TaskCount taskCount = new TaskCount();
        JobContext jobContext = this._taskDriver.getJobContext(PinotHelixTaskResourceManager.getHelixJobName(parentTaskName));
        if (jobContext == null) {
            return taskCount;
        }
        Set partitionSet = jobContext.getPartitionSet();
        Iterator iterator = partitionSet.iterator();
        while (iterator.hasNext()) {
            int partition = (Integer)iterator.next();
            TaskPartitionState state = jobContext.getPartitionState(partition);
            taskCount.addTaskState(state);
        }
        return taskCount;
    }

    public synchronized Set<String> getTasksInProgress(String taskType) {
        HashSet<String> tasksInProgress = new HashSet<String>();
        WorkflowContext workflowContext = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType));
        if (workflowContext == null) {
            return tasksInProgress;
        }
        Map helixJobStates = workflowContext.getJobStates();
        for (Map.Entry entry : helixJobStates.entrySet()) {
            if (!((TaskState)entry.getValue()).equals((Object)TaskState.NOT_STARTED) && !((TaskState)entry.getValue()).equals((Object)TaskState.IN_PROGRESS)) continue;
            tasksInProgress.add(PinotHelixTaskResourceManager.getPinotTaskName((String)entry.getKey()));
        }
        return tasksInProgress;
    }

    public synchronized TaskState getTaskState(String taskName) {
        String taskType = PinotHelixTaskResourceManager.getTaskType(taskName);
        return this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType)).getJobState(PinotHelixTaskResourceManager.getHelixJobName(taskName));
    }

    public synchronized List<PinotTaskConfig> getTaskConfigs(String taskName) {
        Collection helixTaskConfigs = this._taskDriver.getJobConfig(PinotHelixTaskResourceManager.getHelixJobName(taskName)).getTaskConfigMap().values();
        ArrayList<PinotTaskConfig> taskConfigs = new ArrayList<PinotTaskConfig>(helixTaskConfigs.size());
        for (TaskConfig helixTaskConfig : helixTaskConfigs) {
            taskConfigs.add(PinotTaskConfig.fromHelixTaskConfig((TaskConfig)helixTaskConfig));
        }
        return taskConfigs;
    }

    public synchronized Map<String, TaskState> getTaskStatesByTable(String taskType, String tableNameWithType) {
        HashMap<String, TaskState> filteredTaskStateMap = new HashMap<String, TaskState>();
        Map<String, TaskState> taskStateMap = this.getTaskStates(taskType);
        block0: for (Map.Entry<String, TaskState> taskState : taskStateMap.entrySet()) {
            String taskName = taskState.getKey();
            for (PinotTaskConfig taskConfig : this.getTaskConfigs(taskName)) {
                String tableNameConfig;
                Map pinotConfigs = taskConfig.getConfigs();
                if (pinotConfigs == null || (tableNameConfig = (String)pinotConfigs.get("tableName")) == null || !tableNameConfig.equals(tableNameWithType)) continue;
                filteredTaskStateMap.put(taskName, taskStateMap.get(taskName));
                continue block0;
            }
        }
        return filteredTaskStateMap;
    }

    public synchronized Map<String, TaskCount> getTaskCounts(String taskType) {
        TreeMap<String, TaskCount> taskCounts = new TreeMap<String, TaskCount>();
        WorkflowContext workflowContext = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType));
        if (workflowContext == null) {
            return taskCounts;
        }
        Map helixJobStates = workflowContext.getJobStates();
        for (String helixJobName : helixJobStates.keySet()) {
            String pinotTaskName = PinotHelixTaskResourceManager.getPinotTaskName(helixJobName);
            taskCounts.put(pinotTaskName, this.getTaskCount(pinotTaskName));
        }
        return taskCounts;
    }

    public synchronized Map<String, TaskDebugInfo> getTasksDebugInfo(String taskType, int verbosity) {
        TreeMap<String, TaskDebugInfo> taskDebugInfos = new TreeMap<String, TaskDebugInfo>();
        WorkflowContext workflowContext = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType));
        if (workflowContext == null) {
            return taskDebugInfos;
        }
        Map helixJobStates = workflowContext.getJobStates();
        for (String helixJobName : helixJobStates.keySet()) {
            taskDebugInfos.put(PinotHelixTaskResourceManager.getPinotTaskName(helixJobName), this.getTaskDebugInfo(workflowContext, helixJobName, verbosity));
        }
        return taskDebugInfos;
    }

    public synchronized TaskDebugInfo getTaskDebugInfo(String taskName, int verbosity) {
        String taskType = PinotHelixTaskResourceManager.getTaskType(taskName);
        WorkflowContext workflowContext = this._taskDriver.getWorkflowContext(PinotHelixTaskResourceManager.getHelixJobQueueName(taskType));
        if (workflowContext == null) {
            return null;
        }
        String helixJobName = PinotHelixTaskResourceManager.getHelixJobName(taskName);
        return this.getTaskDebugInfo(workflowContext, helixJobName, verbosity);
    }

    private synchronized TaskDebugInfo getTaskDebugInfo(WorkflowContext workflowContext, String helixJobName, int verbosity) {
        JobContext jobContext;
        boolean showCompleted = verbosity > 0;
        TaskDebugInfo taskDebugInfo = new TaskDebugInfo();
        taskDebugInfo.setTaskState(workflowContext.getJobState(helixJobName));
        long jobStartTimeMs = workflowContext.getJobStartTime(helixJobName);
        if (jobStartTimeMs > 0L) {
            taskDebugInfo.setStartTime(DateTimeUtils.epochToDefaultDateFormat((long)jobStartTimeMs));
        }
        if ((jobContext = this._taskDriver.getJobContext(helixJobName)) != null) {
            JobConfig jobConfig = this._taskDriver.getJobConfig(helixJobName);
            long jobExecutionStartTimeMs = jobContext.getExecutionStartTime();
            if (jobExecutionStartTimeMs > 0L) {
                taskDebugInfo.setExecutionStartTime(DateTimeUtils.epochToDefaultDateFormat((long)jobExecutionStartTimeMs));
            }
            Set partitionSet = jobContext.getPartitionSet();
            TaskCount subtaskCount = new TaskCount();
            Iterator iterator = partitionSet.iterator();
            while (iterator.hasNext()) {
                long subtaskFinishTimeMs;
                int partition = (Integer)iterator.next();
                TaskPartitionState partitionState = jobContext.getPartitionState(partition);
                subtaskCount.addTaskState(partitionState);
                if (!showCompleted && partitionState == TaskPartitionState.COMPLETED) continue;
                SubtaskDebugInfo subtaskDebugInfo = new SubtaskDebugInfo();
                String taskIdForPartition = jobContext.getTaskIdForPartition(partition);
                subtaskDebugInfo.setTaskId(taskIdForPartition);
                subtaskDebugInfo.setState(partitionState);
                long subtaskStartTimeMs = jobContext.getPartitionStartTime(partition);
                if (subtaskStartTimeMs > 0L) {
                    subtaskDebugInfo.setStartTime(DateTimeUtils.epochToDefaultDateFormat((long)subtaskStartTimeMs));
                }
                if ((subtaskFinishTimeMs = jobContext.getPartitionFinishTime(partition)) > 0L) {
                    subtaskDebugInfo.setFinishTime(DateTimeUtils.epochToDefaultDateFormat((long)subtaskFinishTimeMs));
                }
                subtaskDebugInfo.setParticipant(jobContext.getAssignedParticipant(partition));
                subtaskDebugInfo.setInfo(jobContext.getPartitionInfo(partition));
                TaskConfig helixTaskConfig = jobConfig.getTaskConfig(taskIdForPartition);
                if (helixTaskConfig != null) {
                    PinotTaskConfig pinotTaskConfig = PinotTaskConfig.fromHelixTaskConfig((TaskConfig)helixTaskConfig);
                    subtaskDebugInfo.setTaskConfig(pinotTaskConfig);
                }
                taskDebugInfo.addSubtaskInfo(subtaskDebugInfo);
            }
            taskDebugInfo.setSubtaskCount(subtaskCount);
        }
        return taskDebugInfo;
    }

    public static String getHelixJobQueueName(String taskType) {
        return TASK_QUEUE_PREFIX + taskType;
    }

    private static String getHelixJobName(String pinotTaskName) {
        return PinotHelixTaskResourceManager.getHelixJobQueueName(PinotHelixTaskResourceManager.getTaskType(pinotTaskName)) + TASK_NAME_SEPARATOR + pinotTaskName;
    }

    private static String getPinotTaskName(String helixJobName) {
        return helixJobName.substring(TASK_QUEUE_PREFIX.length() + PinotHelixTaskResourceManager.getTaskType(helixJobName).length() + 1);
    }

    private static String getTaskType(String name) {
        return name.split(TASK_NAME_SEPARATOR)[1];
    }

    @JsonPropertyOrder(value={"total", "completed", "running", "waiting", "error", "unknown"})
    public static class TaskCount {
        private int _waiting;
        private int _error;
        private int _running;
        private int _completed;
        private int _unknown;
        private int _total;

        public void addTaskState(TaskPartitionState state) {
            ++this._total;
            if (state == null) {
                ++this._waiting;
            } else {
                switch (state) {
                    case INIT: 
                    case RUNNING: {
                        ++this._running;
                        break;
                    }
                    case TASK_ERROR: {
                        ++this._error;
                        break;
                    }
                    case COMPLETED: {
                        ++this._completed;
                        break;
                    }
                    default: {
                        ++this._unknown;
                    }
                }
            }
        }

        public int getWaiting() {
            return this._waiting;
        }

        public int getRunning() {
            return this._running;
        }

        public int getTotal() {
            return this._total;
        }

        public int getError() {
            return this._error;
        }

        public int getCompleted() {
            return this._completed;
        }

        public int getUnknown() {
            return this._unknown;
        }

        public void accumulate(TaskCount other) {
            this._waiting += other.getWaiting();
            this._running += other.getRunning();
            this._error += other.getError();
            this._completed += other.getCompleted();
            this._unknown += other.getUnknown();
            this._total += other.getTotal();
        }
    }

    @JsonPropertyOrder(value={"taskId", "state", "startTime", "finishTime", "participant", "info", "taskConfig"})
    @JsonInclude(value=JsonInclude.Include.NON_NULL)
    public static class SubtaskDebugInfo {
        private String _taskId;
        private TaskPartitionState _state;
        private String _startTime;
        private String _finishTime;
        private String _participant;
        private String _info;
        private PinotTaskConfig _taskConfig;

        public void setTaskId(String taskId) {
            this._taskId = taskId;
        }

        public void setState(TaskPartitionState state) {
            this._state = state;
        }

        public void setStartTime(String startTime) {
            this._startTime = startTime;
        }

        public void setFinishTime(String finishTime) {
            this._finishTime = finishTime;
        }

        public void setParticipant(String participant) {
            this._participant = participant;
        }

        public void setInfo(String info) {
            this._info = info;
        }

        public void setTaskConfig(PinotTaskConfig taskConfig) {
            this._taskConfig = taskConfig;
        }

        public String getTaskId() {
            return this._taskId;
        }

        public TaskPartitionState getState() {
            return this._state;
        }

        public String getStartTime() {
            return this._startTime;
        }

        public String getFinishTime() {
            return this._finishTime;
        }

        public String getParticipant() {
            return this._participant;
        }

        public String getInfo() {
            return this._info;
        }

        public PinotTaskConfig getTaskConfig() {
            return this._taskConfig;
        }
    }

    @JsonPropertyOrder(value={"taskState", "subtaskCount", "startTime", "executionStartTime", "subtaskInfos"})
    @JsonInclude(value=JsonInclude.Include.NON_NULL)
    public static class TaskDebugInfo {
        private String _startTime;
        private String _executionStartTime;
        private TaskState _taskState;
        private TaskCount _subtaskCount;
        private List<SubtaskDebugInfo> _subtaskInfos;

        public void setStartTime(String startTime) {
            this._startTime = startTime;
        }

        public void setExecutionStartTime(String executionStartTime) {
            this._executionStartTime = executionStartTime;
        }

        public void setTaskState(TaskState taskState) {
            this._taskState = taskState;
        }

        public void setSubtaskCount(TaskCount subtaskCount) {
            this._subtaskCount = subtaskCount;
        }

        public void addSubtaskInfo(SubtaskDebugInfo subtaskInfo) {
            if (this._subtaskInfos == null) {
                this._subtaskInfos = new ArrayList<SubtaskDebugInfo>();
            }
            this._subtaskInfos.add(subtaskInfo);
        }

        public String getStartTime() {
            return this._startTime;
        }

        public String getExecutionStartTime() {
            return this._executionStartTime;
        }

        public TaskState getTaskState() {
            return this._taskState;
        }

        public TaskCount getSubtaskCount() {
            return this._subtaskCount;
        }

        public List<SubtaskDebugInfo> getSubtaskInfos() {
            return this._subtaskInfos;
        }
    }
}

