/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.privilege;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.data.BinaryString;
import org.apache.paimon.data.GenericRow;
import org.apache.paimon.data.InternalRow;
import org.apache.paimon.fs.FileIO;
import org.apache.paimon.fs.Path;
import org.apache.paimon.options.Options;
import org.apache.paimon.predicate.Predicate;
import org.apache.paimon.predicate.PredicateBuilder;
import org.apache.paimon.privilege.AllGrantedPrivilegeChecker;
import org.apache.paimon.privilege.EntityType;
import org.apache.paimon.privilege.PrivilegeChecker;
import org.apache.paimon.privilege.PrivilegeCheckerImpl;
import org.apache.paimon.privilege.PrivilegeManager;
import org.apache.paimon.privilege.PrivilegeType;
import org.apache.paimon.reader.RecordReaderIterator;
import org.apache.paimon.schema.Schema;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.table.FileStoreTableFactory;
import org.apache.paimon.table.Table;
import org.apache.paimon.table.sink.BatchTableCommit;
import org.apache.paimon.table.sink.BatchTableWrite;
import org.apache.paimon.table.sink.BatchWriteBuilder;
import org.apache.paimon.table.source.ReadBuilder;
import org.apache.paimon.table.source.TableScan;
import org.apache.paimon.types.DataType;
import org.apache.paimon.types.DataTypes;
import org.apache.paimon.types.RowKind;
import org.apache.paimon.types.RowType;
import org.apache.paimon.utils.CloseableIterator;
import org.apache.paimon.utils.Preconditions;

public class FileBasedPrivilegeManager
implements PrivilegeManager {
    private static final String USER_TABLE_DIR = "user.sys";
    private static final RowType USER_TABLE_TYPE = RowType.of(new DataType[]{DataTypes.STRING(), DataTypes.BYTES()}, new String[]{"user", "sha256"});
    private static final String PRIVILEGE_TABLE_DIR = "privilege.sys";
    private static final RowType PRIVILEGE_TABLE_TYPE = RowType.of(new DataType[]{DataTypes.STRING(), DataTypes.STRING(), DataTypes.STRING(), DataTypes.STRING()}, new String[]{"name", "entity_type", "identifier", "privilege"});
    private final String warehouse;
    private final FileIO fileIO;
    private final String user;
    private final byte[] sha256;
    private Table userTable;
    private Table privilegeTable;

    public FileBasedPrivilegeManager(String warehouse, FileIO fileIO, String user, String password) {
        this.warehouse = warehouse;
        this.fileIO = fileIO;
        this.user = user;
        this.sha256 = this.getSha256(password);
    }

    @Override
    public boolean privilegeEnabled() {
        return this.getUserTable(false) != null && this.getPrivilegeTable(false) != null;
    }

    @Override
    public void initializePrivilege(String rootPassword) {
        if (this.privilegeEnabled()) {
            throw new IllegalStateException("Privilege system is already enabled in warehouse " + this.warehouse);
        }
        this.createUserTable();
        this.createUserImpl("root", rootPassword);
        this.createUserImpl("anonymous", "anonymous");
        this.createPrivilegeTable();
    }

    @Override
    public void createUser(String user, String password) {
        this.getPrivilegeChecker().assertCanCreateUser();
        if (this.userExists(user)) {
            throw new IllegalArgumentException("User " + user + " already exists");
        }
        this.createUserImpl(user, password);
    }

    @Override
    public void dropUser(String user) {
        this.getPrivilegeChecker().assertCanDropUser();
        Preconditions.checkArgument(!"root".equals(user), "root cannot be dropped");
        Preconditions.checkArgument(!"anonymous".equals(user), "anonymous cannot be dropped");
        this.dropUserImpl(user);
    }

    @Override
    public void grant(String user, String identifier, PrivilegeType privilege) {
        this.getPrivilegeChecker().assertCanGrant(identifier, privilege);
        Preconditions.checkArgument(!"root".equals(user), "Cannot change privilege for user root");
        if (!this.userExists(user)) {
            throw new IllegalArgumentException("User " + user + " does not exist");
        }
        this.grantImpl(Collections.singletonList(new PrivilegeEntry(user, EntityType.USER, identifier, privilege)));
    }

    @Override
    public int revoke(String user, String identifier, PrivilegeType privilege) {
        this.getPrivilegeChecker().assertCanRevoke();
        Preconditions.checkArgument(!"root".equals(user), "Cannot change privilege for user root");
        if (!this.userExists(user)) {
            throw new IllegalArgumentException("User " + user + " does not exist");
        }
        int count = this.revokeImpl(user, identifier, privilege);
        Preconditions.checkArgument(count > 0, String.format("User %s does not have privilege %s on %s. It's possible that the user has such privilege on a higher level. Please check the privilege table.", new Object[]{user, privilege, identifier}));
        return count;
    }

    @Override
    public void objectRenamed(String oldName, String newName) {
        Table privilegeTable = this.getPrivilegeTable(true);
        PredicateBuilder predicateBuilder = new PredicateBuilder(PRIVILEGE_TABLE_TYPE);
        Predicate predicate = predicateBuilder.equal(2, BinaryString.fromString(oldName));
        BatchWriteBuilder writeBuilder = privilegeTable.newBatchWriteBuilder();
        try (CloseableIterator<InternalRow> it = this.read(privilegeTable, predicate);
             BatchTableWrite write = writeBuilder.newWrite();
             BatchTableCommit commit = writeBuilder.newCommit();){
            while (it.hasNext()) {
                InternalRow row = (InternalRow)it.next();
                GenericRow replaced = GenericRow.of(row.getString(0), row.getString(1), BinaryString.fromString(newName), row.getString(3));
                write.write(replaced);
            }
            commit.commit(write.prepareCommit());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void objectDropped(String identifier) {
        Table privilegeTable = this.getPrivilegeTable(true);
        PredicateBuilder predicateBuilder = new PredicateBuilder(PRIVILEGE_TABLE_TYPE);
        Predicate predicate = predicateBuilder.startsWith(2, BinaryString.fromString(identifier));
        this.deleteAll(privilegeTable, predicate);
    }

    @Override
    public PrivilegeChecker getPrivilegeChecker() {
        this.assertUserPassword();
        if ("root".equals(this.user)) {
            return new AllGrantedPrivilegeChecker();
        }
        Table privilegeTable = this.getPrivilegeTable(true);
        PredicateBuilder predicateBuilder = new PredicateBuilder(PRIVILEGE_TABLE_TYPE);
        Predicate predicate = PredicateBuilder.and(predicateBuilder.equal(0, BinaryString.fromString(this.user)), predicateBuilder.equal(1, BinaryString.fromString(EntityType.USER.name())));
        HashMap<String, Set<PrivilegeType>> privileges = new HashMap<String, Set<PrivilegeType>>();
        try (CloseableIterator<InternalRow> it = this.read(privilegeTable, predicate);){
            while (it.hasNext()) {
                InternalRow row = (InternalRow)it.next();
                privileges.computeIfAbsent(row.getString(2).toString(), ignore -> new HashSet()).add(PrivilegeType.valueOf(row.getString(3).toString()));
            }
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        return new PrivilegeCheckerImpl(this.user, privileges);
    }

    private void createUserImpl(String user, String password) {
        byte[] sha256 = this.getSha256(password);
        BatchWriteBuilder writeBuilder = this.getUserTable(true).newBatchWriteBuilder();
        try (BatchTableWrite write = writeBuilder.newWrite();
             BatchTableCommit commit = writeBuilder.newCommit();){
            write.write(GenericRow.of(BinaryString.fromString(user), sha256));
            commit.commit(write.prepareCommit());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void dropUserImpl(String user) {
        BatchWriteBuilder writeBuilder = this.getUserTable(true).newBatchWriteBuilder();
        try (BatchTableWrite write = writeBuilder.newWrite();
             BatchTableCommit commit = writeBuilder.newCommit();){
            write.write(GenericRow.ofKind(RowKind.DELETE, BinaryString.fromString(user), new byte[0]));
            commit.commit(write.prepareCommit());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        PredicateBuilder predicateBuilder = new PredicateBuilder(PRIVILEGE_TABLE_TYPE);
        Predicate predicate = PredicateBuilder.and(predicateBuilder.equal(0, BinaryString.fromString(user)), predicateBuilder.equal(1, BinaryString.fromString(EntityType.USER.name())));
        this.deleteAll(this.getPrivilegeTable(true), predicate);
    }

    private void grantImpl(List<PrivilegeEntry> entries) {
        BatchWriteBuilder writeBuilder = this.getPrivilegeTable(true).newBatchWriteBuilder();
        try (BatchTableWrite write = writeBuilder.newWrite();
             BatchTableCommit commit = writeBuilder.newCommit();){
            for (PrivilegeEntry entry : entries) {
                write.write(GenericRow.of(BinaryString.fromString(entry.name), BinaryString.fromString(entry.entityType.name()), BinaryString.fromString(entry.identifier), BinaryString.fromString(entry.privilege.name())));
            }
            commit.commit(write.prepareCommit());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private int revokeImpl(String user, String identifier, PrivilegeType privilege) {
        PredicateBuilder predicateBuilder = new PredicateBuilder(PRIVILEGE_TABLE_TYPE);
        Predicate predicate = PredicateBuilder.and(predicateBuilder.equal(0, BinaryString.fromString(user)), predicateBuilder.equal(1, BinaryString.fromString(EntityType.USER.name())), predicateBuilder.startsWith(2, BinaryString.fromString(identifier)), predicateBuilder.equal(3, BinaryString.fromString(privilege.name())));
        return this.deleteAll(this.getPrivilegeTable(true), predicate);
    }

    private boolean userExists(String user) {
        boolean bl;
        block8: {
            Table userTable = this.getUserTable(true);
            Predicate predicate = new PredicateBuilder(USER_TABLE_TYPE).equal(0, BinaryString.fromString(user));
            CloseableIterator<InternalRow> it = this.read(userTable, predicate);
            try {
                bl = it.hasNext();
                if (it == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (it != null) {
                        try {
                            it.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
            it.close();
        }
        return bl;
    }

    private void assertUserPassword() {
        Table userTable = this.getUserTable(true);
        PredicateBuilder predicateBuilder = new PredicateBuilder(USER_TABLE_TYPE);
        Predicate predicate = PredicateBuilder.and(predicateBuilder.equal(0, BinaryString.fromString(this.user)), predicateBuilder.equal(1, this.sha256));
        try (CloseableIterator<InternalRow> it = this.read(userTable, predicate);){
            if (it.hasNext()) {
                return;
            }
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        throw new IllegalArgumentException("User " + this.user + " not found, or password incorrect.");
    }

    private Table getUserTable(boolean assertExists) {
        this.userTable = this.getTable(this.userTable, USER_TABLE_DIR, assertExists);
        return this.userTable;
    }

    private Table getPrivilegeTable(boolean assertExists) {
        this.privilegeTable = this.getTable(this.privilegeTable, PRIVILEGE_TABLE_DIR, assertExists);
        return this.privilegeTable;
    }

    private Table getTable(Table lazy, String dir, boolean assertExists) {
        boolean tableExists;
        if (lazy != null) {
            return lazy;
        }
        Path tableRoot = new Path(this.warehouse, dir);
        try {
            tableExists = this.fileIO.exists(tableRoot);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        if (tableExists) {
            return FileStoreTableFactory.create(this.fileIO, tableRoot);
        }
        if (assertExists) {
            throw new RuntimeException("Privilege system is not enabled in warehouse " + this.warehouse + ".");
        }
        return null;
    }

    private void createUserTable() {
        Options options = new Options();
        options.set(CoreOptions.BUCKET, 1);
        options.set(CoreOptions.FILE_FORMAT, "avro");
        Path tableRoot = new Path(this.warehouse, USER_TABLE_DIR);
        SchemaManager schemaManager = new SchemaManager(this.fileIO, tableRoot);
        try {
            schemaManager.createTable(new Schema(USER_TABLE_TYPE.getFields(), Collections.emptyList(), Collections.singletonList("user"), options.toMap(), ""));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void createPrivilegeTable() {
        Options options = new Options();
        options.set(CoreOptions.BUCKET, 1);
        Path tableRoot = new Path(this.warehouse, PRIVILEGE_TABLE_DIR);
        SchemaManager schemaManager = new SchemaManager(this.fileIO, tableRoot);
        try {
            schemaManager.createTable(new Schema(PRIVILEGE_TABLE_TYPE.getFields(), Collections.emptyList(), Arrays.asList("name", "entity_type", "privilege", "identifier"), options.toMap(), ""));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private CloseableIterator<InternalRow> read(Table table, Predicate predicate) {
        ReadBuilder readBuilder = table.newReadBuilder().withFilter(predicate);
        TableScan.Plan plan = readBuilder.newScan().plan();
        try {
            return new RecordReaderIterator<InternalRow>(readBuilder.newRead().executeFilter().createReader(plan));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private int deleteAll(Table table, Predicate predicate) {
        int count = 0;
        BatchWriteBuilder writeBuilder = table.newBatchWriteBuilder();
        try (CloseableIterator<InternalRow> it = this.read(table, predicate);
             BatchTableWrite write = writeBuilder.newWrite();
             BatchTableCommit commit = writeBuilder.newCommit();){
            while (it.hasNext()) {
                InternalRow row = (InternalRow)it.next();
                row.setRowKind(RowKind.DELETE);
                write.write(row);
                ++count;
            }
            commit.commit(write.prepareCommit());
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        return count;
    }

    private byte[] getSha256(String s) {
        try {
            return MessageDigest.getInstance("SHA-256").digest(s.getBytes(StandardCharsets.UTF_8));
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    private static class PrivilegeEntry {
        String name;
        EntityType entityType;
        String identifier;
        PrivilegeType privilege;

        private PrivilegeEntry(String name, EntityType entityType, String identifier, PrivilegeType privilege) {
            this.name = name;
            this.entityType = entityType;
            this.identifier = identifier;
            this.privilege = privilege;
        }
    }
}

