/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.service;

import com.google.common.base.Preconditions;
import com.google.protobuf.ServiceException;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.TreeSet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.hadoop.hdds.utils.BackgroundService;
import org.apache.hadoop.hdds.utils.BackgroundTask;
import org.apache.hadoop.hdds.utils.BackgroundTaskQueue;
import org.apache.hadoop.hdds.utils.BackgroundTaskResult;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.om.OMMetadataManager;
import org.apache.hadoop.ozone.om.OMMultiTenantManager;
import org.apache.hadoop.ozone.om.OMMultiTenantManagerImpl;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.exceptions.OMNotLeaderException;
import org.apache.hadoop.ozone.om.helpers.OmDBAccessIdInfo;
import org.apache.hadoop.ozone.om.helpers.OmDBTenantState;
import org.apache.hadoop.ozone.om.multitenant.AuthorizerLock;
import org.apache.hadoop.ozone.om.multitenant.InMemoryMultiTenantAccessController;
import org.apache.hadoop.ozone.om.multitenant.MultiTenantAccessController;
import org.apache.hadoop.ozone.om.ratis.utils.OzoneManagerRatisUtils;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.ratis.protocol.ClientId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OMRangerBGSyncService
extends BackgroundService {
    public static final Logger LOG = LoggerFactory.getLogger(OMRangerBGSyncService.class);
    private static final ClientId CLIENT_ID = ClientId.randomId();
    private final OzoneManager ozoneManager;
    private final OMMetadataManager metadataManager;
    private final OMMultiTenantManager multiTenantManager;
    private final MultiTenantAccessController accessController;
    private final AuthorizerLock authorizerLock;
    private static final int MAX_ATTEMPT = 2;
    private final AtomicLong runCount = new AtomicLong(0L);
    private volatile boolean isServiceStarted = false;
    private final HashMap<String, PolicyInfo> mtRangerPoliciesToBeCreated = new HashMap();
    private final HashMap<String, String> mtRangerPoliciesToBeDeleted = new HashMap();
    private final HashMap<String, BGRole> mtRangerRoles = new HashMap();
    private final HashMap<String, HashSet<String>> mtOMDBRoles = new HashMap();

    public OMRangerBGSyncService(OzoneManager ozoneManager, OMMultiTenantManager omMultiTenantManager, MultiTenantAccessController accessController, long interval, TimeUnit unit, long serviceTimeout) {
        super("OMRangerBGSyncService", interval, unit, 1, serviceTimeout, ozoneManager.getThreadNamePrefix());
        this.ozoneManager = ozoneManager;
        this.metadataManager = ozoneManager.getMetadataManager();
        this.multiTenantManager = omMultiTenantManager;
        this.authorizerLock = omMultiTenantManager.getAuthorizerLock();
        if (accessController != null) {
            this.accessController = accessController;
        } else {
            LOG.warn("MultiTenantAccessController not set. Using in-memory controller.");
            this.accessController = new InMemoryMultiTenantAccessController();
        }
    }

    public BackgroundTaskQueue getTasks() {
        BackgroundTaskQueue queue = new BackgroundTaskQueue();
        queue.add((BackgroundTask)new RangerBGSyncTask());
        return queue;
    }

    public void start() {
        if (this.accessController == null) {
            LOG.error("Failed to start the background sync service: null authorizer. Please check OM configuration. Aborting");
            return;
        }
        this.isServiceStarted = true;
        super.start();
    }

    public void shutdown() {
        this.isServiceStarted = false;
        super.shutdown();
    }

    private boolean shouldRun() {
        if (this.ozoneManager == null) {
            return true;
        }
        if (this.ozoneManager.getOmRatisServer() == null) {
            LOG.warn("OzoneManagerRatisServer is not initialized yet");
            return false;
        }
        return this.isServiceStarted && this.ozoneManager.isLeaderReady();
    }

    public synchronized boolean triggerRangerSyncOnce() {
        int attempt = 0;
        try {
            try {
                long dbOzoneServiceVersion = this.getOMDBRangerServiceVersion();
                long rangerOzoneServiceVersion = this.getRangerOzoneServicePolicyVersion();
                while (dbOzoneServiceVersion != rangerOzoneServiceVersion) {
                    if (++attempt > 2) {
                        LOG.warn("Reached maximum number of attempts ({}). Abort", (Object)2);
                        break;
                    }
                    LOG.info("Executing Multi-Tenancy Ranger Sync: run # {}, attempt # {}. Ranger service version: {}, DB service version: {}", new Object[]{this.runCount.get(), attempt, rangerOzoneServiceVersion, dbOzoneServiceVersion});
                    this.executeOMDBToRangerSync(dbOzoneServiceVersion);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Setting DB Ranger service version to {} (was {})", (Object)rangerOzoneServiceVersion, (Object)dbOzoneServiceVersion);
                    }
                    this.setOMDBRangerServiceVersion(rangerOzoneServiceVersion);
                    dbOzoneServiceVersion = rangerOzoneServiceVersion;
                    rangerOzoneServiceVersion = this.getRangerOzoneServicePolicyVersion();
                }
            }
            catch (ServiceException | IOException e) {
                LOG.warn("Exception during Ranger Sync", e);
                if (attempt > 0) {
                    LOG.info("Finished executing Multi-Tenancy Ranger Sync run # {} after{} attempts.", (Object)this.runCount.get(), (Object)attempt);
                }
                return false;
            }
        }
        finally {
            if (attempt > 0) {
                LOG.info("Finished executing Multi-Tenancy Ranger Sync run # {} after{} attempts.", (Object)this.runCount.get(), (Object)attempt);
            }
        }
        return true;
    }

    long getRangerOzoneServicePolicyVersion() throws IOException {
        long policyVersion = this.accessController.getRangerServicePolicyVersion();
        if (policyVersion < 0L) {
            LOG.warn("Unable to get valid policyVersion for Ranger background sync to function properly. Please check if the Kerberos principal as configured in ozone.om.kerberos.principal ({}) has admin privilege in Ranger.", (Object)this.ozoneManager.getConfiguration().get("ozone.om.kerberos.principal"));
        }
        return policyVersion;
    }

    public void setOMDBRangerServiceVersion(long version) throws ServiceException {
        OzoneManagerProtocolProtos.SetRangerServiceVersionRequest.Builder versionSyncRequest = OzoneManagerProtocolProtos.SetRangerServiceVersionRequest.newBuilder().setRangerServiceVersion(version);
        OzoneManagerProtocolProtos.OMRequest omRequest = OzoneManagerProtocolProtos.OMRequest.newBuilder().setCmdType(OzoneManagerProtocolProtos.Type.SetRangerServiceVersion).setSetRangerServiceVersionRequest(versionSyncRequest).setClientId(CLIENT_ID.toString()).build();
        try {
            OzoneManagerRatisUtils.submitRequest(this.ozoneManager, omRequest, CLIENT_ID, this.runCount.get());
        }
        catch (ServiceException e) {
            LOG.error("SetRangerServiceVersion request failed. Will retry at next run.", (Throwable)e);
            throw e;
        }
    }

    long getOMDBRangerServiceVersion() throws IOException {
        String dbValue = (String)this.ozoneManager.getMetadataManager().getMetaTable().get((Object)"#RANGEROZONESERVICEVERSION");
        if (dbValue == null) {
            return -1L;
        }
        return Long.parseLong(dbValue);
    }

    private void executeOMDBToRangerSync(long baseVersion) throws IOException {
        this.clearPolicyAndRoleMaps();
        this.withOptimisticRead(() -> {
            try {
                this.loadAllPoliciesAndRoleNamesFromRanger(baseVersion);
                this.loadAllRolesFromRanger();
                this.loadAllRolesFromOM();
            }
            catch (IOException e) {
                LOG.error("Failed to load policies or roles from Ranger or DB", (Throwable)e);
                throw new RuntimeException(e);
            }
        });
        this.processAllPoliciesFromOMDB();
        this.processAllRolesFromOMDB();
    }

    private void clearPolicyAndRoleMaps() {
        this.mtRangerPoliciesToBeCreated.clear();
        this.mtRangerPoliciesToBeDeleted.clear();
        this.mtRangerRoles.clear();
        this.mtOMDBRoles.clear();
    }

    private List<MultiTenantAccessController.Policy> getAllMultiTenantPolicies() throws IOException {
        return this.accessController.getLabeledPolicies("OzoneTenant");
    }

    private void loadAllPoliciesAndRoleNamesFromRanger(long baseVersion) throws IOException {
        if (LOG.isDebugEnabled()) {
            LOG.debug("baseVersion is {}", (Object)baseVersion);
        }
        List<MultiTenantAccessController.Policy> allPolicies = this.getAllMultiTenantPolicies();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received policies with {} label: {}", (Object)"OzoneTenant", allPolicies);
        }
        if (allPolicies.isEmpty()) {
            LOG.info("No Ranger policy with label {} received.", (Object)"OzoneTenant");
            return;
        }
        for (MultiTenantAccessController.Policy policy : allPolicies) {
            if (!policy.getLabels().contains("OzoneTenant")) {
                LOG.warn("Ignoring Ranger policy without the {} label: {}", (Object)"OzoneTenant", (Object)policy);
                continue;
            }
            this.mtRangerPoliciesToBeDeleted.put(policy.getName(), String.valueOf(policy.getId()));
            policy.getRoleAcls().keySet().forEach(roleName -> {
                if (!this.mtRangerRoles.containsKey(roleName)) {
                    this.mtRangerRoles.put((String)roleName, new BGRole((String)roleName));
                }
            });
        }
    }

    private void checkLeader() throws IOException {
        if (this.ozoneManager.getConfiguration().getBoolean("ozone.om.tenant.dev.skip.ranger", false)) {
            return;
        }
        if (!this.ozoneManager.isLeaderReady()) {
            throw new OMNotLeaderException(this.ozoneManager.getOmRatisServer().getRaftPeerId());
        }
    }

    private void loadAllRolesFromRanger() throws IOException {
        for (Map.Entry<String, BGRole> entry : this.mtRangerRoles.entrySet()) {
            String roleName = entry.getKey();
            this.checkLeader();
            MultiTenantAccessController.Role role = this.accessController.getRole(roleName);
            BGRole bgRole = entry.getValue();
            bgRole.setId(role.getName());
            for (String username : role.getUsersMap().keySet()) {
                bgRole.addUserPrincipal(username);
            }
        }
    }

    private void mtRangerPoliciesOpHelper(String policyName, PolicyInfo policyInfo) {
        if (this.mtRangerPoliciesToBeDeleted.containsKey(policyName)) {
            this.mtRangerPoliciesToBeDeleted.remove(policyName);
        } else {
            this.mtRangerPoliciesToBeCreated.put(policyName, policyInfo);
        }
    }

    private void withOptimisticRead(Runnable block) throws IOException {
        int attempt = 0;
        boolean readSuccess = false;
        while (!readSuccess && attempt < 2) {
            long stamp = this.authorizerLock.tryOptimisticReadThrowOnTimeout();
            block.run();
            readSuccess = this.authorizerLock.validateOptimisticRead(stamp);
            ++attempt;
        }
        if (!readSuccess) {
            throw new IOException("Failed to read state for Ranger background sync without an interrupting write operation after " + attempt + " attempts.");
        }
    }

    private void withWriteLock(Runnable block) throws IOException {
        long stamp = this.authorizerLock.tryWriteLockThrowOnTimeout();
        try {
            block.run();
        }
        finally {
            this.authorizerLock.unlockWrite(stamp);
        }
    }

    /*
     * WARNING - void declaration
     */
    private void processAllPoliciesFromOMDB() throws IOException {
        String policyName;
        Throwable throwable = null;
        Iterator<Map.Entry<String, Object>> iterator = null;
        try (TableIterator tenantStateTableIt = this.metadataManager.getTenantStateTable().iterator();){
            while (tenantStateTableIt.hasNext()) {
                Table.KeyValue tableKeyValue = (Table.KeyValue)tenantStateTableIt.next();
                OmDBTenantState dbTenantState = (OmDBTenantState)tableKeyValue.getValue();
                String tenantId = dbTenantState.getTenantId();
                String volumeName = dbTenantState.getBucketNamespaceName();
                Preconditions.checkNotNull((Object)volumeName);
                this.mtRangerPoliciesOpHelper(dbTenantState.getBucketNamespacePolicyName(), new PolicyInfo(tenantId, PolicyType.BUCKET_NAMESPACE_POLICY));
                this.mtRangerPoliciesOpHelper(dbTenantState.getBucketPolicyName(), new PolicyInfo(tenantId, PolicyType.BUCKET_POLICY));
            }
        }
        catch (Throwable throwable2) {
            void var1_4;
            if (throwable == null) {
                Throwable throwable3 = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw var1_4;
        }
        for (Map.Entry<String, PolicyInfo> entry : this.mtRangerPoliciesToBeCreated.entrySet()) {
            policyName = entry.getKey();
            LOG.warn("Expected policy not found in Ranger: {}", (Object)policyName);
            this.checkLeader();
            this.attemptToCreateDefaultPolicy(entry.getValue());
        }
        for (Map.Entry<String, Object> entry : this.mtRangerPoliciesToBeDeleted.entrySet()) {
            policyName = entry.getKey();
            LOG.info("Deleting policy from Ranger: {}", (Object)policyName);
            this.checkLeader();
            this.withWriteLock(() -> {
                try {
                    this.accessController.deletePolicy(policyName);
                }
                catch (IOException e) {
                    LOG.error("Failed to delete policy: {}", (Object)policyName, (Object)e);
                }
            });
        }
    }

    private void attemptToCreateDefaultPolicy(PolicyInfo policyInfo) throws IOException {
        MultiTenantAccessController.Policy accessPolicy;
        String tenantId = policyInfo.getTenantId();
        String volumeName = this.multiTenantManager.getTenantVolumeName(tenantId);
        String userRoleName = this.multiTenantManager.getTenantUserRoleName(tenantId);
        switch (policyInfo.getPolicyType()) {
            case BUCKET_NAMESPACE_POLICY: {
                LOG.info("Recovering VolumeAccess policy for tenant: {}", (Object)tenantId);
                String adminRoleName = this.multiTenantManager.getTenantAdminRoleName(tenantId);
                accessPolicy = OMMultiTenantManager.getDefaultVolumeAccessPolicy(tenantId, volumeName, userRoleName, adminRoleName);
                break;
            }
            case BUCKET_POLICY: {
                LOG.info("Recovering BucketAccess policy for tenant: {}", (Object)tenantId);
                accessPolicy = OMMultiTenantManager.getDefaultBucketAccessPolicy(tenantId, volumeName, userRoleName);
                break;
            }
            default: {
                throw new OMException("Unknown policy type in " + policyInfo, OMException.ResultCodes.INTERNAL_ERROR);
            }
        }
        this.withWriteLock(() -> {
            try {
                MultiTenantAccessController.Policy policy2 = this.accessController.createPolicy(accessPolicy);
                LOG.info("Created policy: {}", (Object)policy2);
            }
            catch (IOException e) {
                LOG.error("Failed to create policy: {}", (Object)accessPolicy, (Object)e);
            }
        });
    }

    private void loadAllRolesFromOM() throws IOException {
        if (this.multiTenantManager instanceof OMMultiTenantManagerImpl) {
            this.loadAllRolesFromCache();
        } else {
            LOG.warn("Cache is not supported for {}. Loading roles directly from DB", (Object)this.multiTenantManager.getClass().getSimpleName());
            this.loadAllRolesFromDB();
        }
    }

    private void loadAllRolesFromCache() {
        OMMultiTenantManagerImpl impl = (OMMultiTenantManagerImpl)this.multiTenantManager;
        this.mtOMDBRoles.putAll(impl.getAllRolesFromCache());
    }

    private void loadAllRolesFromDB() throws IOException {
        Table tenantStateTable = this.metadataManager.getTenantStateTable();
        Throwable throwable = null;
        Object var3_4 = null;
        try (TableIterator tenantAccessIdTableIter = this.metadataManager.getTenantAccessIdTable().iterator();){
            while (tenantAccessIdTableIter.hasNext()) {
                Table.KeyValue tableKeyValue = (Table.KeyValue)tenantAccessIdTableIter.next();
                OmDBAccessIdInfo dbAccessIdInfo = (OmDBAccessIdInfo)tableKeyValue.getValue();
                String tenantId = dbAccessIdInfo.getTenantId();
                String userPrincipal = dbAccessIdInfo.getUserPrincipal();
                OmDBTenantState dbTenantState = (OmDBTenantState)tenantStateTable.get((Object)tenantId);
                if (dbTenantState == null) {
                    LOG.warn("OmDBTenantState for tenant '{}' doesn't exist!", (Object)tenantId);
                    continue;
                }
                String userRoleName = dbTenantState.getUserRoleName();
                this.mtOMDBRoles.computeIfAbsent(userRoleName, any -> new HashSet());
                String adminRoleName = dbTenantState.getAdminRoleName();
                this.mtOMDBRoles.computeIfAbsent(adminRoleName, any -> new HashSet());
                this.addUserToMtOMDBRoles(userRoleName, userPrincipal);
                if (!dbAccessIdInfo.getIsAdmin()) continue;
                this.addUserToMtOMDBRoles(adminRoleName, userPrincipal);
            }
        }
        catch (Throwable throwable2) {
            if (throwable == null) {
                throwable = throwable2;
            } else if (throwable != throwable2) {
                throwable.addSuppressed(throwable2);
            }
            throw throwable;
        }
    }

    private void addUserToMtOMDBRoles(String roleName, String userPrincipal) {
        if (!this.mtOMDBRoles.containsKey(roleName)) {
            this.mtOMDBRoles.put(roleName, new HashSet<String>(Collections.singletonList(userPrincipal)));
        } else {
            HashSet<String> usersInTheRole = this.mtOMDBRoles.get(roleName);
            usersInTheRole.add(userPrincipal);
        }
    }

    private void processAllRolesFromOMDB() throws IOException {
        for (Map.Entry<String, HashSet<String>> entry : this.mtOMDBRoles.entrySet()) {
            String roleName = entry.getKey();
            boolean pushRoleToRanger = false;
            if (this.mtRangerRoles.containsKey(roleName)) {
                HashSet<String> rangerUserList = this.mtRangerRoles.get(roleName).getUserSet();
                HashSet<String> userSet = entry.getValue();
                for (String userPrincipal : userSet) {
                    if (rangerUserList.contains(userPrincipal)) {
                        rangerUserList.remove(userPrincipal);
                        continue;
                    }
                    pushRoleToRanger = true;
                    break;
                }
                if (!rangerUserList.isEmpty()) {
                    pushRoleToRanger = true;
                }
            } else {
                this.checkLeader();
                this.withWriteLock(() -> {
                    try {
                        MultiTenantAccessController.Role role = new MultiTenantAccessController.Role.Builder().setName(roleName).setDescription("Managed by Ozone. WARNING: Changes will be overridden. Use Ozone tenant CLI to manage users in this tenant role instead.").build();
                        this.accessController.createRole(role);
                    }
                    catch (IOException e) {
                        LOG.error("Failed to create role: {}", (Object)roleName, (Object)e);
                    }
                });
                pushRoleToRanger = true;
            }
            if (pushRoleToRanger) {
                LOG.info("Updating role in Ranger: {}", (Object)roleName);
                this.checkLeader();
                this.pushOMDBRoleToRanger(roleName);
            }
            this.mtRangerRoles.remove(roleName);
        }
        TreeSet rolesToDelete = new TreeSet(Collections.reverseOrder());
        rolesToDelete.addAll(this.mtRangerRoles.keySet());
        for (String roleName : rolesToDelete) {
            LOG.warn("Deleting role from Ranger: {}", (Object)roleName);
            this.checkLeader();
            this.withWriteLock(() -> {
                try {
                    this.accessController.deleteRole(roleName);
                }
                catch (IOException iOException) {
                    LOG.error("Failed to delete role: {}", (Object)roleName);
                }
            });
        }
    }

    private void pushOMDBRoleToRanger(String roleName) throws IOException {
        HashSet<String> omDBUserList = this.mtOMDBRoles.get(roleName);
        this.withWriteLock(() -> {
            try {
                MultiTenantAccessController.Role existingRole = this.accessController.getRole(roleName);
                if (!existingRole.getId().isPresent()) {
                    LOG.error("Role doesn't have ID: {}", (Object)existingRole);
                    return;
                }
                long roleId = existingRole.getId().get();
                MultiTenantAccessController.Role newRole = new MultiTenantAccessController.Role.Builder(existingRole).clearUsers().addUsers(omDBUserList).build();
                this.accessController.updateRole(roleId, newRole);
            }
            catch (IOException e) {
                LOG.error("Failed to update role: {}, target user list: {}", new Object[]{roleName, omDBUserList, e});
            }
        });
    }

    public long getRangerSyncRunCount() {
        return this.runCount.get();
    }

    static class BGRole {
        private final String name;
        private String id;
        private final HashSet<String> userSet;

        BGRole(String n) {
            this.name = n;
            this.userSet = new HashSet();
        }

        public void setId(String id) {
            this.id = id;
        }

        public String getId() {
            return this.id;
        }

        public void addUserPrincipal(String userPrincipal) {
            this.userSet.add(userPrincipal);
        }

        public HashSet<String> getUserSet() {
            return this.userSet;
        }

        public int hashCode() {
            return Objects.hash(this.name, this.id, this.userSet);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            BGRole bgRole = (BGRole)o;
            return this.name.equals(bgRole.name) && this.id.equals(bgRole.id) && this.userSet.equals(bgRole.userSet);
        }
    }

    static class PolicyInfo {
        private final String tenantId;
        private final PolicyType policyType;

        PolicyInfo(String tenantId, PolicyType policyType) {
            this.tenantId = tenantId;
            this.policyType = policyType;
        }

        public String getTenantId() {
            return this.tenantId;
        }

        public PolicyType getPolicyType() {
            return this.policyType;
        }

        public String toString() {
            return "PolicyInfo{tenantId='" + this.tenantId + '\'' + ", policyType=" + (Object)((Object)this.policyType) + '}';
        }
    }

    static enum PolicyType {
        BUCKET_NAMESPACE_POLICY,
        BUCKET_POLICY;

    }

    private class RangerBGSyncTask
    implements BackgroundTask {
        private RangerBGSyncTask() {
        }

        public int getPriority() {
            return 0;
        }

        public BackgroundTaskResult call() {
            if (OMRangerBGSyncService.this.shouldRun()) {
                long count = OMRangerBGSyncService.this.runCount.incrementAndGet();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Initiating Multi-Tenancy Ranger Sync: run # {}", (Object)count);
                }
                OMRangerBGSyncService.this.triggerRangerSyncOnce();
            }
            return BackgroundTaskResult.EmptyTaskResult.newResult();
        }
    }
}

