/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.client.io;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.hadoop.fs.Syncable;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.ContainerClientMetrics;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.StreamBufferArgs;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.client.HddsClientUtils;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.common.helpers.ExcludeList;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.ozone.OzoneManagerVersion;
import org.apache.hadoop.ozone.client.io.BlockOutputStreamEntry;
import org.apache.hadoop.ozone.client.io.BlockOutputStreamEntryPool;
import org.apache.hadoop.ozone.client.io.KeyMetadataAware;
import org.apache.hadoop.ozone.client.io.KeyOutputStreamSemaphore;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfo;
import org.apache.hadoop.ozone.om.helpers.OmKeyLocationInfoGroup;
import org.apache.hadoop.ozone.om.helpers.OmMultipartCommitUploadPartInfo;
import org.apache.hadoop.ozone.om.helpers.OpenKeySession;
import org.apache.hadoop.ozone.om.protocol.OzoneManagerProtocol;
import org.apache.hadoop.ozone.util.MetricUtil;
import org.apache.ratis.protocol.exceptions.AlreadyClosedException;
import org.apache.ratis.protocol.exceptions.RaftRetryFailureException;
import org.apache.ratis.util.function.CheckedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyOutputStream
extends OutputStream
implements Syncable,
KeyMetadataAware {
    private final ReplicationConfig replication;
    public static final Logger LOG = LoggerFactory.getLogger(KeyOutputStream.class);
    private boolean closed;
    private final Map<Class<? extends Throwable>, RetryPolicy> retryPolicyMap;
    private int retryCount;
    private long offset;
    private volatile long writeOffset;
    private boolean isException;
    private final BlockOutputStreamEntryPool blockOutputStreamEntryPool;
    private long clientID;
    private StreamBufferArgs streamBufferArgs;
    private boolean atomicKeyCreation;
    private ContainerClientMetrics clientMetrics;
    private OzoneManagerVersion ozoneManagerVersion;
    private final Lock writeLock = new ReentrantLock();
    private final Condition retryHandlingCondition = this.writeLock.newCondition();
    private final int maxConcurrentWritePerKey;
    private final KeyOutputStreamSemaphore keyOutputStreamSemaphore;

    @VisibleForTesting
    KeyOutputStreamSemaphore getRequestSemaphore() {
        return this.keyOutputStreamSemaphore;
    }

    @VisibleForTesting
    KeyOutputStream() {
        this.maxConcurrentWritePerKey = 0;
        this.keyOutputStreamSemaphore = null;
        this.blockOutputStreamEntryPool = null;
        this.retryPolicyMap = null;
        this.replication = null;
    }

    public KeyOutputStream(ReplicationConfig replicationConfig, BlockOutputStreamEntryPool blockOutputStreamEntryPool) {
        this.replication = replicationConfig;
        this.closed = false;
        this.retryPolicyMap = HddsClientUtils.getExceptionList().stream().collect(Collectors.toMap(Function.identity(), e -> RetryPolicies.TRY_ONCE_THEN_FAIL));
        this.retryCount = 0;
        this.offset = 0L;
        this.blockOutputStreamEntryPool = blockOutputStreamEntryPool;
        this.maxConcurrentWritePerKey = 1;
        this.keyOutputStreamSemaphore = new KeyOutputStreamSemaphore(this.maxConcurrentWritePerKey);
    }

    protected BlockOutputStreamEntryPool getBlockOutputStreamEntryPool() {
        return this.blockOutputStreamEntryPool;
    }

    @VisibleForTesting
    public List<BlockOutputStreamEntry> getStreamEntries() {
        return this.blockOutputStreamEntryPool.getStreamEntries();
    }

    @VisibleForTesting
    public List<OmKeyLocationInfo> getLocationInfoList() {
        return this.blockOutputStreamEntryPool.getLocationInfoList();
    }

    @VisibleForTesting
    public int getRetryCount() {
        return this.retryCount;
    }

    @VisibleForTesting
    public long getClientID() {
        return this.clientID;
    }

    public KeyOutputStream(Builder b) {
        this.replication = b.replicationConfig;
        this.blockOutputStreamEntryPool = new BlockOutputStreamEntryPool(b);
        OzoneClientConfig config = b.getClientConfig();
        this.maxConcurrentWritePerKey = config.getMaxConcurrentWritePerKey();
        this.keyOutputStreamSemaphore = new KeyOutputStreamSemaphore(this.maxConcurrentWritePerKey);
        this.retryPolicyMap = HddsClientUtils.getRetryPolicyByException((int)config.getMaxRetryCount(), (long)config.getRetryInterval());
        this.retryCount = 0;
        this.isException = false;
        this.writeOffset = 0L;
        this.clientID = b.getOpenHandler().getId();
        this.atomicKeyCreation = b.getAtomicKeyCreation();
        this.streamBufferArgs = b.getStreamBufferArgs();
        this.clientMetrics = b.getClientMetrics();
        this.ozoneManagerVersion = b.ozoneManagerVersion;
    }

    public void addPreallocateBlocks(OmKeyLocationInfoGroup version, long openVersion) {
        this.blockOutputStreamEntryPool.addPreallocateBlocks(version, openVersion);
    }

    @Override
    public void write(int b) throws IOException {
        byte[] buf = new byte[]{(byte)b};
        this.write(buf, 0, 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        try {
            this.getRequestSemaphore().acquire();
            this.checkNotClosed();
            if (b == null) {
                throw new NullPointerException();
            }
            if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
                throw new IndexOutOfBoundsException();
            }
            if (len == 0) {
                return;
            }
            this.doInWriteLock(() -> {
                this.handleWrite(b, off, len, false);
                this.writeOffset += (long)len;
            });
        }
        finally {
            this.getRequestSemaphore().release();
        }
    }

    private <E extends Throwable> void doInWriteLock(CheckedRunnable<E> block) throws E {
        this.writeLock.lock();
        try {
            block.run();
        }
        finally {
            this.writeLock.unlock();
        }
    }

    @VisibleForTesting
    void handleWrite(byte[] b, int off, long len, boolean retry) throws IOException {
        while (len > 0L) {
            try {
                BlockOutputStreamEntry current = this.blockOutputStreamEntryPool.allocateBlockIfNeeded(retry);
                int expectedWriteLen = Math.min((int)len, (int)current.getRemaining());
                long currentPos = current.getWrittenDataLength();
                int writtenLength = this.writeToOutputStream(current, retry, len, b, expectedWriteLen, off, currentPos);
                if (current.getRemaining() <= 0L) {
                    this.handleFlushOrClose(StreamAction.FULL);
                }
                len -= (long)writtenLength;
                off += writtenLength;
            }
            catch (Exception e) {
                this.markStreamClosed();
                throw new IOException(e);
            }
        }
    }

    private int writeToOutputStream(BlockOutputStreamEntry current, boolean retry, long len, byte[] b, int writeLen, int off, long currentPos) throws IOException {
        try {
            current.registerCallReceived();
            if (retry) {
                current.writeOnRetry(len);
            } else {
                current.waitForRetryHandling(this.retryHandlingCondition);
                current.write(b, off, writeLen);
                this.offset += (long)writeLen;
            }
            current.registerCallFinished();
        }
        catch (InterruptedException e) {
            current.registerCallFinished();
            throw new InterruptedIOException();
        }
        catch (IOException ioe) {
            Preconditions.checkState((!retry || len <= this.streamBufferArgs.getStreamBufferMaxSize() ? 1 : 0) != 0);
            int dataWritten = (int)(current.getWrittenDataLength() - currentPos);
            int n = writeLen = retry ? (int)len : dataWritten;
            if (!retry) {
                this.offset += (long)writeLen;
            }
            LOG.debug("writeLen {}, total len {}", (Object)writeLen, (Object)len);
            this.handleException(current, ioe, retry);
        }
        return writeLen;
    }

    private void handleException(BlockOutputStreamEntry entry, IOException exception, boolean fromRetry) throws IOException {
        this.doInWriteLock(() -> {
            this.handleExceptionInternal(entry, exception);
            BlockOutputStreamEntry current = this.blockOutputStreamEntryPool.getCurrentStreamEntry();
            if (!fromRetry && entry.registerCallFinished()) {
                current.finishRetryHandling(this.retryHandlingCondition);
            }
        });
    }

    private void handleExceptionInternal(BlockOutputStreamEntry streamEntry, IOException exception) throws IOException {
        try {
            streamEntry.waitForAllPendingFlushes();
        }
        catch (IOException iOException) {
            // empty catch block
        }
        Throwable t = HddsClientUtils.checkForException((Exception)exception);
        Preconditions.checkNotNull((Object)t);
        boolean retryFailure = this.checkForRetryFailure(t);
        boolean containerExclusionException = false;
        if (!retryFailure) {
            containerExclusionException = this.checkIfContainerToExclude(t);
        }
        Pipeline pipeline = streamEntry.getPipeline();
        PipelineID pipelineId = pipeline.getId();
        long totalSuccessfulFlushedData = streamEntry.getTotalAckDataLength();
        streamEntry.resetToAckedPosition();
        long bufferedDataLen = this.blockOutputStreamEntryPool.computeBufferData();
        if (containerExclusionException) {
            LOG.debug("Encountered exception {}. The last committed block length is {}, uncommitted data length is {} retry count {}", new Object[]{exception, totalSuccessfulFlushedData, bufferedDataLen, this.retryCount});
        } else {
            LOG.warn("Encountered exception {} on the pipeline {}. The last committed block length is {}, uncommitted data length is {} retry count {}", new Object[]{exception, pipeline, totalSuccessfulFlushedData, bufferedDataLen, this.retryCount});
        }
        Preconditions.checkArgument((bufferedDataLen <= this.streamBufferArgs.getStreamBufferMaxSize() ? 1 : 0) != 0);
        long containerId = streamEntry.getBlockID().getContainerID();
        Collection<DatanodeDetails> failedServers = streamEntry.getFailedServers();
        Preconditions.checkNotNull(failedServers);
        ExcludeList excludeList = this.blockOutputStreamEntryPool.getExcludeList();
        if (!failedServers.isEmpty()) {
            excludeList.addDatanodes(failedServers);
        }
        if (containerExclusionException) {
            excludeList.addConatinerId(ContainerID.valueOf((long)containerId));
        } else {
            excludeList.addPipeline(pipelineId);
        }
        streamEntry.cleanup(retryFailure);
        if (containerExclusionException) {
            this.blockOutputStreamEntryPool.discardPreallocatedBlocks(streamEntry.getBlockID().getContainerID(), null);
        } else {
            this.blockOutputStreamEntryPool.discardPreallocatedBlocks(-1L, pipelineId);
        }
        if (bufferedDataLen > 0L) {
            this.handleRetry(exception, bufferedDataLen);
            this.retryCount = 0;
        }
    }

    private synchronized void markStreamClosed() {
        this.blockOutputStreamEntryPool.cleanup();
        this.closed = true;
    }

    private void handleRetry(IOException exception, long len) throws IOException {
        RetryPolicy retryPolicy = this.retryPolicyMap.get(HddsClientUtils.checkForException((Exception)exception).getClass());
        if (retryPolicy == null) {
            retryPolicy = this.retryPolicyMap.get(Exception.class);
        }
        RetryPolicy.RetryAction action = null;
        try {
            action = retryPolicy.shouldRetry((Exception)exception, this.retryCount, 0, true);
        }
        catch (Exception e) {
            this.setExceptionAndThrow(new IOException(e));
        }
        if (action.action == RetryPolicy.RetryAction.RetryDecision.FAIL) {
            String msg = "";
            if (action.reason != null) {
                msg = "Retry request failed. " + action.reason;
                LOG.error(msg, (Throwable)exception);
            }
            this.setExceptionAndThrow(new IOException(msg, exception));
        }
        if (Thread.currentThread().isInterrupted()) {
            LOG.warn("Interrupted while trying for retry");
            this.setExceptionAndThrow(exception);
        }
        Preconditions.checkArgument((action.action == RetryPolicy.RetryAction.RetryDecision.RETRY ? 1 : 0) != 0);
        if (action.delayMillis > 0L) {
            try {
                Thread.sleep(action.delayMillis);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                IOException ioe = (IOException)new InterruptedIOException("Interrupted: action=" + action + ", retry policy=" + retryPolicy).initCause(e);
                this.setExceptionAndThrow(ioe);
            }
        }
        ++this.retryCount;
        if (LOG.isTraceEnabled()) {
            LOG.trace("Retrying Write request. Already tried {} time(s); retry policy is {} ", (Object)this.retryCount, (Object)retryPolicy);
        }
        this.handleWrite(null, 0, len, true);
    }

    private void setExceptionAndThrow(IOException ioe) throws IOException {
        this.isException = true;
        throw ioe;
    }

    private boolean checkForRetryFailure(Throwable t) {
        return t instanceof RaftRetryFailureException || t instanceof AlreadyClosedException;
    }

    protected boolean checkIfContainerToExclude(Throwable t) {
        return t instanceof StorageContainerException;
    }

    @Override
    public void flush() throws IOException {
        try {
            this.getRequestSemaphore().acquire();
            this.checkNotClosed();
            this.handleFlushOrClose(StreamAction.FLUSH);
        }
        finally {
            this.getRequestSemaphore().release();
        }
    }

    public void hflush() throws IOException {
        this.hsync();
    }

    public void hsync() throws IOException {
        try {
            this.getRequestSemaphore().acquire();
            if (this.replication.getReplicationType() != HddsProtos.ReplicationType.RATIS) {
                throw new UnsupportedOperationException("Replication type is not " + HddsProtos.ReplicationType.RATIS);
            }
            if (this.replication.getRequiredNodes() <= 1) {
                throw new UnsupportedOperationException("The replication factor = " + this.replication.getRequiredNodes() + " <= 1");
            }
            if (this.ozoneManagerVersion.compareTo((Enum)OzoneManagerVersion.HBASE_SUPPORT) < 0) {
                throw new UnsupportedOperationException("Hsync API requires OM version " + OzoneManagerVersion.HBASE_SUPPORT + " or later. Current OM version " + this.ozoneManagerVersion);
            }
            this.checkNotClosed();
            long hsyncPos = this.writeOffset;
            this.handleFlushOrClose(StreamAction.HSYNC);
            this.doInWriteLock(() -> {
                Preconditions.checkState((this.offset >= hsyncPos ? 1 : 0) != 0, (String)"offset = %s < hsyncPos = %s", (long)this.offset, (long)hsyncPos);
                MetricUtil.captureLatencyNs(arg_0 -> ((ContainerClientMetrics)this.clientMetrics).addHsyncLatency(arg_0), () -> this.blockOutputStreamEntryPool.hsyncKey(hsyncPos));
            });
        }
        finally {
            this.getRequestSemaphore().release();
        }
    }

    private void handleFlushOrClose(StreamAction op) throws IOException {
        if (!this.blockOutputStreamEntryPool.isEmpty()) {
            try {
                BlockOutputStreamEntry entry;
                while ((entry = this.blockOutputStreamEntryPool.getCurrentStreamEntry()) != null) {
                    this.doInWriteLock(() -> entry.waitForRetryHandling(this.retryHandlingCondition));
                    entry.registerCallReceived();
                    try {
                        this.handleStreamAction(entry, op);
                        entry.registerCallFinished();
                        break;
                    }
                    catch (IOException ioe) {
                        this.handleException(entry, ioe, false);
                    }
                    catch (Exception e) {
                        entry.registerCallFinished();
                        throw e;
                    }
                }
                return;
            }
            catch (InterruptedException e) {
                throw new InterruptedIOException();
            }
            catch (Exception e) {
                this.markStreamClosed();
                throw e;
            }
        }
    }

    private void handleStreamAction(BlockOutputStreamEntry entry, StreamAction op) throws IOException {
        Collection<DatanodeDetails> failedServers = entry.getFailedServers();
        if (!failedServers.isEmpty()) {
            this.blockOutputStreamEntryPool.getExcludeList().addDatanodes(failedServers);
        }
        switch (op) {
            case CLOSE: {
                entry.close();
                break;
            }
            case FULL: {
                if (entry.getRemaining() != 0L) break;
                entry.close();
                break;
            }
            case FLUSH: {
                entry.flush();
                break;
            }
            case HSYNC: {
                entry.hsync();
                break;
            }
            default: {
                throw new IOException("Invalid Operation");
            }
        }
    }

    @Override
    public void close() throws IOException {
        this.doInWriteLock(this::closeInternal);
    }

    private void closeInternal() throws IOException {
        if (this.closed) {
            return;
        }
        this.closed = true;
        try {
            this.handleFlushOrClose(StreamAction.CLOSE);
            if (!this.isException) {
                Preconditions.checkArgument((this.writeOffset == this.offset ? 1 : 0) != 0);
            }
            if (this.atomicKeyCreation) {
                long expectedSize = this.blockOutputStreamEntryPool.getDataSize();
                Preconditions.checkState((expectedSize == this.offset ? 1 : 0) != 0, (Object)String.format("Expected: %d and actual %d write sizes do not match", expectedSize, this.offset));
            }
            this.blockOutputStreamEntryPool.commitKey(this.offset);
        }
        finally {
            this.blockOutputStreamEntryPool.cleanup();
        }
    }

    synchronized OmMultipartCommitUploadPartInfo getCommitUploadPartInfo() {
        return this.blockOutputStreamEntryPool.getCommitUploadPartInfo();
    }

    @VisibleForTesting
    public ExcludeList getExcludeList() {
        return this.blockOutputStreamEntryPool.getExcludeList();
    }

    @Override
    public Map<String, String> getMetadata() {
        return this.blockOutputStreamEntryPool.getMetadata();
    }

    private void checkNotClosed() throws IOException {
        if (this.closed) {
            throw new IOException(": Stream is closed! Key: " + this.blockOutputStreamEntryPool.getKeyName());
        }
    }

    public static class Builder {
        private OpenKeySession openHandler;
        private XceiverClientFactory xceiverManager;
        private OzoneManagerProtocol omClient;
        private final String requestID = UUID.randomUUID().toString();
        private String multipartUploadID;
        private int multipartNumber;
        private boolean isMultipartKey;
        private boolean unsafeByteBufferConversion;
        private OzoneClientConfig clientConfig;
        private ReplicationConfig replicationConfig;
        private ContainerClientMetrics clientMetrics;
        private boolean atomicKeyCreation = false;
        private StreamBufferArgs streamBufferArgs;
        private Supplier<ExecutorService> executorServiceSupplier;
        private OzoneManagerVersion ozoneManagerVersion;

        public String getMultipartUploadID() {
            return this.multipartUploadID;
        }

        public Builder setMultipartUploadID(String uploadID) {
            this.multipartUploadID = uploadID;
            return this;
        }

        public int getMultipartNumber() {
            return this.multipartNumber;
        }

        public Builder setMultipartNumber(int partNumber) {
            this.multipartNumber = partNumber;
            return this;
        }

        public OpenKeySession getOpenHandler() {
            return this.openHandler;
        }

        public Builder setHandler(OpenKeySession handler) {
            this.openHandler = handler;
            return this;
        }

        public XceiverClientFactory getXceiverManager() {
            return this.xceiverManager;
        }

        public Builder setXceiverClientManager(XceiverClientFactory manager) {
            this.xceiverManager = manager;
            return this;
        }

        public OzoneManagerProtocol getOmClient() {
            return this.omClient;
        }

        public Builder setOmClient(OzoneManagerProtocol client) {
            this.omClient = client;
            return this;
        }

        public String getRequestID() {
            return this.requestID;
        }

        public boolean isMultipartKey() {
            return this.isMultipartKey;
        }

        public Builder setIsMultipartKey(boolean isMultipart) {
            this.isMultipartKey = isMultipart;
            return this;
        }

        public OzoneClientConfig getClientConfig() {
            return this.clientConfig;
        }

        public Builder setConfig(OzoneClientConfig config) {
            this.clientConfig = config;
            return this;
        }

        public StreamBufferArgs getStreamBufferArgs() {
            return this.streamBufferArgs;
        }

        public Builder setStreamBufferArgs(StreamBufferArgs streamBufferArgs) {
            this.streamBufferArgs = streamBufferArgs;
            return this;
        }

        public boolean isUnsafeByteBufferConversionEnabled() {
            return this.unsafeByteBufferConversion;
        }

        public Builder enableUnsafeByteBufferConversion(boolean enabled) {
            this.unsafeByteBufferConversion = enabled;
            return this;
        }

        public ReplicationConfig getReplicationConfig() {
            return this.replicationConfig;
        }

        public Builder setReplicationConfig(ReplicationConfig replConfig) {
            this.replicationConfig = replConfig;
            return this;
        }

        public Builder setAtomicKeyCreation(boolean atomicKey) {
            this.atomicKeyCreation = atomicKey;
            return this;
        }

        public Builder setClientMetrics(ContainerClientMetrics clientMetrics) {
            this.clientMetrics = clientMetrics;
            return this;
        }

        public ContainerClientMetrics getClientMetrics() {
            return this.clientMetrics;
        }

        public boolean getAtomicKeyCreation() {
            return this.atomicKeyCreation;
        }

        public Builder setExecutorServiceSupplier(Supplier<ExecutorService> executorServiceSupplier) {
            this.executorServiceSupplier = executorServiceSupplier;
            return this;
        }

        public Supplier<ExecutorService> getExecutorServiceSupplier() {
            return this.executorServiceSupplier;
        }

        public Builder setOmVersion(OzoneManagerVersion omVersion) {
            this.ozoneManagerVersion = omVersion;
            return this;
        }

        public OzoneManagerVersion getOmVersion() {
            return this.ozoneManagerVersion;
        }

        public KeyOutputStream build() {
            return new KeyOutputStream(this);
        }
    }

    static enum StreamAction {
        FLUSH,
        HSYNC,
        CLOSE,
        FULL;

    }
}

