/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.windows.event.log;

import com.sun.jna.platform.win32.Kernel32;
import com.sun.jna.platform.win32.Kernel32Util;
import com.sun.jna.platform.win32.WinNT;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.TriggerSerially;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.processor.AbstractSessionFactoryProcessor;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.ProcessSessionFactory;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.windows.event.log.AlreadySubscribedException;
import org.apache.nifi.processors.windows.event.log.jna.ErrorLookup;
import org.apache.nifi.processors.windows.event.log.jna.EventSubscribeXmlRenderingCallback;
import org.apache.nifi.processors.windows.event.log.jna.WEvtApi;

@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@Tags(value={"ingest", "event", "windows"})
@TriggerSerially
@CapabilityDescription(value="Registers a Windows Event Log Subscribe Callback to receive FlowFiles from Events on Windows.  These can be filtered via channel and XPath.")
@WritesAttributes(value={@WritesAttribute(attribute="mime.type", description="Will set a MIME type value of application/xml.")})
public class ConsumeWindowsEventLog
extends AbstractSessionFactoryProcessor {
    public static final String DEFAULT_CHANNEL = "System";
    public static final String DEFAULT_XPATH = "*";
    public static final int DEFAULT_MAX_BUFFER = 0x100000;
    public static final int DEFAULT_MAX_QUEUE_SIZE = 1024;
    public static final PropertyDescriptor CHANNEL = new PropertyDescriptor.Builder().name("channel").displayName("Channel").required(true).defaultValue("System").description("The Windows Event Log Channel to listen to.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).build();
    public static final PropertyDescriptor QUERY = new PropertyDescriptor.Builder().name("query").displayName("XPath Query").required(true).defaultValue("*").description("XPath Query to filter events. (See https://msdn.microsoft.com/en-us/library/windows/desktop/dd996910(v=vs.85).aspx for examples.)").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).build();
    public static final PropertyDescriptor MAX_BUFFER_SIZE = new PropertyDescriptor.Builder().name("maxBuffer").displayName("Maximum Buffer Size").required(true).defaultValue(Integer.toString(0x100000)).description("The individual Event Log XMLs are rendered to a buffer.  This specifies the maximum size in bytes that the buffer will be allowed to grow to. (Limiting the maximum size of an individual Event XML.)").addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).build();
    public static final PropertyDescriptor MAX_EVENT_QUEUE_SIZE = new PropertyDescriptor.Builder().name("maxQueue").displayName("Maximum queue size").required(true).defaultValue(Integer.toString(1024)).description("Events are received asynchronously and must be output as FlowFiles when the processor is triggered.  This specifies the maximum number of events to queue for transformation into FlowFiles.").addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).build();
    public static final PropertyDescriptor INACTIVE_DURATION_TO_RECONNECT = new PropertyDescriptor.Builder().name("inactiveDurationToReconnect").displayName("Inactive duration to reconnect").description("If no new event logs are processed for the specified time period, this processor will try reconnecting to recover from a state where any further messages cannot be consumed. Such situation can happen if Windows Event Log service is restarted, or ERROR_EVT_QUERY_RESULT_STALE (15011) is returned. Setting no duration, e.g. '0 ms' disables auto-reconnection.").required(true).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).defaultValue("10 mins").addValidator(StandardValidators.createTimePeriodValidator((long)0L, (TimeUnit)TimeUnit.MILLISECONDS, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.MILLISECONDS)).build();
    public static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(CHANNEL, QUERY, MAX_BUFFER_SIZE, MAX_EVENT_QUEUE_SIZE, INACTIVE_DURATION_TO_RECONNECT);
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Relationship for successfully consumed events.").build();
    public static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS);
    public static final String APPLICATION_XML = "application/xml";
    public static final String UNABLE_TO_SUBSCRIBE = "Unable to subscribe with provided parameters, received the following error code: ";
    public static final String PROCESSOR_ALREADY_SUBSCRIBED = "Processor already subscribed to Event Log, expected cleanup to unsubscribe.";
    private final WEvtApi wEvtApi;
    private final Kernel32 kernel32;
    private final ErrorLookup errorLookup;
    private final String name;
    private Throwable wEvtApiError = null;
    private Throwable kernel32Error = null;
    private BlockingQueue<String> renderedXMLs;
    private WEvtApi.EVT_SUBSCRIBE_CALLBACK evtSubscribeCallback;
    private WinNT.HANDLE subscriptionHandle;
    private ProcessSessionFactory sessionFactory;
    private String provenanceUri;
    private long inactiveDurationToReconnect = 0L;
    private long lastActivityTimestamp = 0L;

    public ConsumeWindowsEventLog() {
        this(null, null);
    }

    public ConsumeWindowsEventLog(WEvtApi wEvtApi, Kernel32 kernel32) {
        this.wEvtApi = wEvtApi == null ? this.loadWEvtApi() : wEvtApi;
        this.kernel32 = kernel32 == null ? this.loadKernel32() : kernel32;
        this.errorLookup = new ErrorLookup(this.kernel32);
        this.name = this.kernel32 != null ? Kernel32Util.getComputerName() : null;
    }

    private WEvtApi loadWEvtApi() {
        try {
            return WEvtApi.INSTANCE;
        }
        catch (Throwable e) {
            this.wEvtApiError = e;
            return null;
        }
    }

    private Kernel32 loadKernel32() {
        try {
            return Kernel32.INSTANCE;
        }
        catch (Throwable e) {
            this.kernel32Error = e;
            return null;
        }
    }

    private String subscribe(ProcessContext context) {
        String channel = context.getProperty(CHANNEL).evaluateAttributeExpressions().getValue();
        String query = context.getProperty(QUERY).evaluateAttributeExpressions().getValue();
        this.renderedXMLs = new LinkedBlockingQueue<String>(context.getProperty(MAX_EVENT_QUEUE_SIZE).asInteger());
        try {
            this.provenanceUri = new URI("winlog", this.name, "/" + channel, query, null).toASCIIString();
        }
        catch (URISyntaxException e) {
            this.getLogger().debug("Failed to construct detailed provenanceUri from channel={}, query={}, use simpler one.", new Object[]{channel, query});
            this.provenanceUri = String.format("winlog://%s/%s", this.name, channel);
        }
        this.inactiveDurationToReconnect = context.getProperty(INACTIVE_DURATION_TO_RECONNECT).evaluateAttributeExpressions().asTimePeriod(TimeUnit.MILLISECONDS);
        this.evtSubscribeCallback = new EventSubscribeXmlRenderingCallback(this.getLogger(), s -> {
            try {
                this.renderedXMLs.put((String)s);
            }
            catch (InterruptedException e) {
                throw new IllegalStateException("Got interrupted while waiting to add to queue.", e);
            }
        }, context.getProperty(MAX_BUFFER_SIZE).asInteger(), this.wEvtApi, this.kernel32, this.errorLookup);
        this.subscriptionHandle = this.wEvtApi.EvtSubscribe(null, null, channel, query, null, null, this.evtSubscribeCallback, 65537);
        if (!this.isSubscribed()) {
            return UNABLE_TO_SUBSCRIBE + this.errorLookup.getLastError();
        }
        this.lastActivityTimestamp = System.currentTimeMillis();
        return null;
    }

    private boolean isSubscriptionHandleOpen() {
        return this.subscriptionHandle != null && this.subscriptionHandle.getPointer() != null;
    }

    private boolean isSubscribed() {
        boolean subscribed = this.isSubscriptionHandleOpen();
        boolean subscriptionFailed = this.evtSubscribeCallback != null && ((EventSubscribeXmlRenderingCallback)this.evtSubscribeCallback).isSubscriptionFailed();
        boolean subscribing = subscribed && !subscriptionFailed;
        this.getLogger().debug("subscribing? {}, subscribed={}, subscriptionFailed={}", new Object[]{subscribing, subscribed, subscriptionFailed});
        if (subscriptionFailed) {
            this.getLogger().info("Canceling a failed subscription.");
            this.unsubscribe();
        }
        return subscribing;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) throws AlreadySubscribedException {
        if (this.isSubscribed()) {
            throw new AlreadySubscribedException(PROCESSOR_ALREADY_SUBSCRIBED);
        }
        String errorMessage = this.subscribe(context);
        if (errorMessage != null) {
            this.getLogger().error(errorMessage);
        }
    }

    @OnStopped
    public void stop() {
        this.unsubscribe();
        if (!this.renderedXMLs.isEmpty()) {
            if (this.sessionFactory != null) {
                this.getLogger().info("Finishing processing leftover events");
                ProcessSession session = this.sessionFactory.createSession();
                this.processQueue(session);
            } else {
                throw new ProcessException("Stopping the processor but there is no ProcessSessionFactory stored and there are messages in the internal queue. Removing the processor now will clear the queue but will result in DATA LOSS. This is normally due to starting the processor, receiving events and stopping before the onTrigger happens. The messages in the internal queue cannot finish processing until until the processor is triggered to run.");
            }
        }
        this.sessionFactory = null;
        this.provenanceUri = null;
        this.renderedXMLs = null;
    }

    private void unsubscribe() {
        if (this.isSubscriptionHandleOpen()) {
            this.wEvtApi.EvtClose(this.subscriptionHandle);
        }
        this.subscriptionHandle = null;
        this.evtSubscribeCallback = null;
    }

    public void onTrigger(ProcessContext context, ProcessSessionFactory sessionFactory) throws ProcessException {
        String errorMessage;
        this.sessionFactory = sessionFactory;
        if (!this.isSubscribed() && (errorMessage = this.subscribe(context)) != null) {
            context.yield();
            this.getLogger().error(errorMessage);
            return;
        }
        int flowFileCount = this.processQueue(sessionFactory.createSession());
        long now = System.currentTimeMillis();
        if (flowFileCount > 0) {
            this.lastActivityTimestamp = now;
        } else if (this.inactiveDurationToReconnect > 0L && now - this.lastActivityTimestamp > this.inactiveDurationToReconnect) {
            this.getLogger().info("Exceeds configured 'inactive duration to reconnect' {} ms. Unsubscribe to reconnect..", new Object[]{this.inactiveDurationToReconnect});
            this.unsubscribe();
        }
    }

    private int processQueue(ProcessSession session) {
        ArrayList xmlMessages = new ArrayList();
        this.renderedXMLs.drainTo(xmlMessages);
        try {
            for (String xmlMessage : xmlMessages) {
                FlowFile flowFile = session.create();
                byte[] xmlBytes = xmlMessage.getBytes(StandardCharsets.UTF_8);
                flowFile = session.write(flowFile, out -> out.write(xmlBytes));
                flowFile = session.putAttribute(flowFile, CoreAttributes.MIME_TYPE.key(), APPLICATION_XML);
                session.getProvenanceReporter().receive(flowFile, this.provenanceUri);
                session.transfer(flowFile, REL_SUCCESS);
            }
        }
        catch (Throwable t2) {
            this.getLogger().error("Failed to create FlowFile for XML message", t2);
            this.renderedXMLs.addAll(xmlMessages);
            session.rollback();
            throw t2;
        }
        session.commitAsync(() -> {}, t -> this.renderedXMLs.addAll(xmlMessages));
        return xmlMessages.size();
    }

    protected Collection<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> validationResults = new ArrayList<ValidationResult>(super.customValidate(validationContext));
        if (this.wEvtApiError != null) {
            validationResults.add(new ValidationResult.Builder().valid(false).subject("System Configuration").explanation("NiFi failed to load wevtapi on this system.  This processor utilizes native Windows APIs and will only work on Windows. (" + this.wEvtApiError.getMessage() + ")").build());
        }
        if (this.kernel32Error != null) {
            validationResults.add(new ValidationResult.Builder().valid(false).subject("System Configuration").explanation("NiFi failed to load kernel32 on this system.  This processor utilizes native Windows APIs and will only work on Windows. (" + this.kernel32Error.getMessage() + ")").build());
        }
        return validationResults;
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }
}

