/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.stateless.core;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.SSLContext;
import org.apache.nifi.flow.VersionedFlowCoordinates;
import org.apache.nifi.flow.VersionedProcessGroup;
import org.apache.nifi.registry.flow.VersionedFlow;
import org.apache.nifi.registry.flow.VersionedFlowSnapshot;
import org.apache.nifi.stateless.core.FlowSnapshotProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RegistryFlowSnapshotProvider
implements FlowSnapshotProvider {
    private static final Logger logger = LoggerFactory.getLogger(RegistryFlowSnapshotProvider.class);
    private static final Pattern REGISTRY_URL_PATTERN = Pattern.compile("^(https?://.+?)/?nifi-registry-api.*$");
    private static final Duration TIMEOUT = Duration.ofSeconds(30L);
    private static final String FORWARD_SLASH = "/";
    private static final String BUCKET_FLOW_PATH_FORMAT = "buckets/%s/flows/%s";
    private static final String BUCKET_FLOW_VERSION_PATH_FORMAT = "buckets/%s/flows/%s/versions/%d";
    private static final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private final String registryUrl;
    private final HttpClient httpClient;
    private final SSLContext sslContext;

    public RegistryFlowSnapshotProvider(String registryUrl, SSLContext sslContext) {
        this.registryUrl = registryUrl;
        this.sslContext = sslContext;
        HttpClient.Builder builder = HttpClient.newBuilder();
        builder.connectTimeout(TIMEOUT);
        if (sslContext != null) {
            builder.sslContext(sslContext);
        }
        this.httpClient = builder.build();
    }

    @Override
    public VersionedFlowSnapshot getFlowSnapshot(String bucketID, String flowID, int versionRequested) throws IOException {
        int version = versionRequested == -1 ? this.getLatestVersion(bucketID, flowID) : versionRequested;
        logger.debug("Fetching flow Bucket={}, Flow={}, Version={}, FetchRemoteFlows=true", new Object[]{bucketID, flowID, version});
        long start = System.nanoTime();
        VersionedFlowSnapshot snapshot = this.getFlowContents(bucketID, flowID, version);
        long millis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start);
        logger.info("Fetched Flow [{}] Version [{}] from Bucket [{}] in {} ms", new Object[]{flowID, version, bucketID, millis});
        return snapshot;
    }

    private int getLatestVersion(String bucketId, String flowId) throws IOException {
        String path = BUCKET_FLOW_PATH_FORMAT.formatted(bucketId, flowId);
        URI uri = this.getUri(path);
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri);
        try {
            byte[] responseBody = this.sendRequest(requestBuilder);
            VersionedFlow versionedFlow = (VersionedFlow)objectMapper.readValue(responseBody, VersionedFlow.class);
            long versionCount = versionedFlow.getVersionCount();
            return Math.toIntExact(versionCount);
        }
        catch (Exception e) {
            throw new IOException("Failed to get Latest Version for Bucket [%s] Flow [%s]".formatted(bucketId, flowId));
        }
    }

    private VersionedFlowSnapshot getFlowContents(String bucketId, String flowId, int version) throws IOException {
        String path = BUCKET_FLOW_VERSION_PATH_FORMAT.formatted(bucketId, flowId, version);
        URI uri = this.getUri(path);
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri);
        try {
            byte[] responseBody = this.sendRequest(requestBuilder);
            VersionedFlowSnapshot flowSnapshot = (VersionedFlowSnapshot)objectMapper.readValue(responseBody, VersionedFlowSnapshot.class);
            VersionedProcessGroup contents = flowSnapshot.getFlowContents();
            for (VersionedProcessGroup child : contents.getProcessGroups()) {
                this.populateVersionedContentsRecursively(child);
            }
            return flowSnapshot;
        }
        catch (Exception e) {
            throw new IOException("Failed to get contents for Flow [%s] Version [%d] Bucket [%s]".formatted(flowId, version, bucketId), e);
        }
    }

    protected String getBaseRegistryUrl(String storageLocation) {
        Matcher matcher = REGISTRY_URL_PATTERN.matcher(storageLocation);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return storageLocation;
    }

    private void populateVersionedContentsRecursively(VersionedProcessGroup group) throws IOException {
        if (group == null) {
            return;
        }
        VersionedFlowCoordinates coordinates = group.getVersionedFlowCoordinates();
        if (coordinates != null) {
            String subRegistryUrl = this.getBaseRegistryUrl(coordinates.getStorageLocation());
            String bucketId = coordinates.getBucketId();
            String flowId = coordinates.getFlowId();
            int version = Integer.parseInt(coordinates.getVersion());
            RegistryFlowSnapshotProvider nestedProvider = new RegistryFlowSnapshotProvider(subRegistryUrl, this.sslContext);
            VersionedFlowSnapshot snapshot = nestedProvider.getFlowSnapshot(bucketId, flowId, version);
            VersionedProcessGroup contents = snapshot.getFlowContents();
            group.setComments(contents.getComments());
            group.setConnections(contents.getConnections());
            group.setControllerServices(contents.getControllerServices());
            group.setFunnels(contents.getFunnels());
            group.setInputPorts(contents.getInputPorts());
            group.setLabels(contents.getLabels());
            group.setOutputPorts(contents.getOutputPorts());
            group.setProcessGroups(contents.getProcessGroups());
            group.setProcessors(contents.getProcessors());
            group.setRemoteProcessGroups(contents.getRemoteProcessGroups());
            group.setFlowFileConcurrency(contents.getFlowFileConcurrency());
            group.setFlowFileOutboundPolicy(contents.getFlowFileOutboundPolicy());
            group.setDefaultFlowFileExpiration(contents.getDefaultFlowFileExpiration());
            group.setDefaultBackPressureObjectThreshold(contents.getDefaultBackPressureObjectThreshold());
            group.setDefaultBackPressureDataSizeThreshold(contents.getDefaultBackPressureDataSizeThreshold());
            group.setLogFileSuffix(contents.getLogFileSuffix());
            coordinates.setLatest(Boolean.valueOf(snapshot.isLatest()));
        }
        for (VersionedProcessGroup child : group.getProcessGroups()) {
            this.populateVersionedContentsRecursively(child);
        }
    }

    private byte[] sendRequest(HttpRequest.Builder requestBuilder) throws IOException {
        requestBuilder.timeout(TIMEOUT);
        HttpRequest request = requestBuilder.build();
        URI uri = request.uri();
        try {
            HttpResponse<byte[]> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofByteArray());
            int statusCode = response.statusCode();
            if (200 == statusCode) {
                return response.body();
            }
            throw new IOException("Registry request failed with HTTP %d [%s]".formatted(statusCode, uri));
        }
        catch (IOException e) {
            throw new IOException("Registry request failed [%s]".formatted(uri), e);
        }
        catch (InterruptedException e) {
            throw new IOException("Registry requested interrupted [%s]".formatted(uri), e);
        }
    }

    private URI getUri(String path) {
        StringBuilder builder = new StringBuilder();
        builder.append(this.registryUrl);
        if (!this.registryUrl.endsWith(FORWARD_SLASH)) {
            builder.append(FORWARD_SLASH);
        }
        builder.append(path);
        return URI.create(builder.toString());
    }
}

