/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.controller;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryUsage;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.controller.ConfigurationContext;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.reporting.AbstractReportingTask;
import org.apache.nifi.reporting.InitializationException;
import org.apache.nifi.reporting.ReportingContext;
import org.apache.nifi.util.FormatUtils;

@Tags(value={"monitor", "memory", "heap", "jvm", "gc", "garbage collection", "warning"})
@CapabilityDescription(value="Checks the amount of Java Heap available in the JVM for a particular JVM Memory Pool. If the amount of space used exceeds some configurable threshold, will warn (via a log message and System-Level Bulletin) that the memory pool is exceeding this threshold.")
public class MonitorMemory
extends AbstractReportingTask {
    private static final List<String> GC_OLD_GEN_POOLS = Collections.unmodifiableList(Arrays.asList("Tenured Gen", "PS Old Gen", "G1 Old Gen", "CMS Old Gen", "ZHeap"));
    private static final AllowableValue[] memPoolAllowableValues = (AllowableValue[])ManagementFactory.getMemoryPoolMXBeans().stream().filter(MemoryPoolMXBean::isCollectionUsageThresholdSupported).map(MemoryPoolMXBean::getName).map(AllowableValue::new).toArray(AllowableValue[]::new);
    private static String defaultMemoryPool = Arrays.stream(memPoolAllowableValues).map(AllowableValue::getValue).filter(GC_OLD_GEN_POOLS::contains).findFirst().orElse(null);
    public static final PropertyDescriptor MEMORY_POOL_PROPERTY = new PropertyDescriptor.Builder().name("Memory Pool").description("The name of the JVM Memory Pool to monitor. The allowed values for Memory Pools are platform and JVM dependent and may vary for different versions of Java and from published documentation. This reporting task will become invalidated if configured to use a Memory Pool that is not available on the currently running host platform and JVM").required(true).allowableValues((DescribedValue[])memPoolAllowableValues).defaultValue(defaultMemoryPool).build();
    public static final PropertyDescriptor THRESHOLD_PROPERTY = new PropertyDescriptor.Builder().name("Usage Threshold").description("Indicates the threshold at which warnings should be generated. This can be a percentage or a Data Size").required(true).addValidator((Validator)new ThresholdValidator()).defaultValue("65%").build();
    public static final PropertyDescriptor REPORTING_INTERVAL = new PropertyDescriptor.Builder().name("Reporting Interval").description("Indicates how often this reporting task should report bulletins while the memory utilization exceeds the configured threshold").required(false).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).build();
    public static final Pattern PERCENTAGE_PATTERN = Pattern.compile("\\d{1,2}%");
    public static final Pattern DATA_SIZE_PATTERN = DataUnit.DATA_SIZE_PATTERN;
    public static final Pattern TIME_PERIOD_PATTERN = FormatUtils.TIME_DURATION_PATTERN;
    private volatile MemoryPoolMXBean monitoredBean;
    private volatile String threshold = "65%";
    private volatile long calculatedThreshold;
    private volatile long lastReportTime;
    private volatile long reportingIntervalMillis;
    private volatile boolean lastValueWasExceeded;
    private static final List<PropertyDescriptor> propertyDescriptors;

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return propertyDescriptors;
    }

    @OnScheduled
    public void onConfigured(ConfigurationContext config) throws InitializationException {
        String thresholdValue;
        String desiredMemoryPoolName = config.getProperty(MEMORY_POOL_PROPERTY).getValue();
        this.threshold = thresholdValue = config.getProperty(THRESHOLD_PROPERTY).getValue().trim();
        Long reportingIntervalValue = config.getProperty(REPORTING_INTERVAL).asTimePeriod(TimeUnit.MILLISECONDS);
        this.reportingIntervalMillis = reportingIntervalValue == null ? config.getSchedulingPeriod(TimeUnit.MILLISECONDS).longValue() : reportingIntervalValue.longValue();
        List<MemoryPoolMXBean> memoryPoolBeans = ManagementFactory.getMemoryPoolMXBeans();
        for (int i = 0; i < memoryPoolBeans.size() && this.monitoredBean == null; ++i) {
            MemoryPoolMXBean memoryPoolBean = memoryPoolBeans.get(i);
            String memoryPoolName = memoryPoolBean.getName();
            if (!desiredMemoryPoolName.equals(memoryPoolName)) continue;
            this.monitoredBean = memoryPoolBean;
            if (!memoryPoolBean.isCollectionUsageThresholdSupported()) continue;
            if (DATA_SIZE_PATTERN.matcher(thresholdValue).matches()) {
                this.calculatedThreshold = DataUnit.parseDataSize((String)thresholdValue, (DataUnit)DataUnit.B).longValue();
            } else {
                String percentage = thresholdValue.substring(0, thresholdValue.length() - 1);
                double pct = Double.parseDouble(percentage) / 100.0;
                this.calculatedThreshold = (long)((double)this.monitoredBean.getCollectionUsage().getMax() * pct);
            }
            if (!this.monitoredBean.isCollectionUsageThresholdSupported()) continue;
            this.monitoredBean.setCollectionUsageThreshold(this.calculatedThreshold);
        }
        if (this.monitoredBean == null) {
            throw new InitializationException("Found no JVM Memory Pool with name " + desiredMemoryPoolName + "; will not monitor Memory Pool");
        }
    }

    public void onTrigger(ReportingContext context) {
        MemoryPoolMXBean bean = this.monitoredBean;
        if (bean == null) {
            return;
        }
        MemoryUsage usage = bean.getCollectionUsage();
        if (usage == null) {
            this.getLogger().warn("{} could not determine memory usage for pool with name {}", new Object[]{this, context.getProperty(MEMORY_POOL_PROPERTY)});
            return;
        }
        double percentageUsed = (double)usage.getUsed() / (double)usage.getMax() * 100.0;
        if (bean.isCollectionUsageThresholdSupported() && bean.isCollectionUsageThresholdExceeded() && usage.getUsed() > this.calculatedThreshold) {
            if (System.currentTimeMillis() < this.reportingIntervalMillis + this.lastReportTime && this.lastReportTime > 0L) {
                return;
            }
            this.lastReportTime = System.currentTimeMillis();
            this.lastValueWasExceeded = true;
            String message = String.format("Memory Pool '%1$s' has exceeded the configured Threshold of %2$s, having used %3$s / %4$s (%5$.2f%%)", bean.getName(), this.threshold, FormatUtils.formatDataSize((double)usage.getUsed()), FormatUtils.formatDataSize((double)usage.getMax()), percentageUsed);
            this.getLogger().warn("{}", new Object[]{message});
        } else if (this.lastValueWasExceeded) {
            this.lastValueWasExceeded = false;
            this.lastReportTime = System.currentTimeMillis();
            String message = String.format("Memory Pool '%1$s' is no longer exceeding the configured Threshold of %2$s; currently using %3$s / %4$s (%5$.2f%%)", bean.getName(), this.threshold, FormatUtils.formatDataSize((double)usage.getUsed()), FormatUtils.formatDataSize((double)usage.getMax()), percentageUsed);
            this.getLogger().info("{}", new Object[]{message});
        }
    }

    @OnStopped
    public void onStopped() {
        this.monitoredBean = null;
    }

    static {
        ArrayList<PropertyDescriptor> _propertyDescriptors = new ArrayList<PropertyDescriptor>();
        _propertyDescriptors.add(MEMORY_POOL_PROPERTY);
        _propertyDescriptors.add(THRESHOLD_PROPERTY);
        _propertyDescriptors.add(REPORTING_INTERVAL);
        propertyDescriptors = Collections.unmodifiableList(_propertyDescriptors);
    }

    private static class ThresholdValidator
    implements Validator {
        private ThresholdValidator() {
        }

        public ValidationResult validate(String subject, String input, ValidationContext context) {
            if (!PERCENTAGE_PATTERN.matcher(input).matches() && !DATA_SIZE_PATTERN.matcher(input).matches()) {
                return new ValidationResult.Builder().input(input).subject(subject).valid(false).explanation("Valid value is a number in the range of 0-99 followed by a percent sign (e.g. 65%) or a Data Size (e.g. 100 MB)").build();
            }
            return new ValidationResult.Builder().input(input).subject(subject).valid(true).build();
        }
    }
}

