/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.standard;

import java.io.File;
import java.io.IOException;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileOwnerAttributeView;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermissions;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.BiPredicate;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.Stateful;
import org.apache.nifi.annotation.behavior.TriggerSerially;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.configuration.DefaultSchedule;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.annotation.lifecycle.OnStopped;
import org.apache.nifi.components.AllowableValue;
import org.apache.nifi.components.DescribedValue;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.context.PropertyContext;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.logging.ComponentLog;
import org.apache.nifi.migration.PropertyConfiguration;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessorInitializationContext;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processor.util.file.transfer.FileInfo;
import org.apache.nifi.processor.util.list.AbstractListProcessor;
import org.apache.nifi.processor.util.list.ListedEntityTracker;
import org.apache.nifi.processors.standard.FetchFile;
import org.apache.nifi.processors.standard.GetFile;
import org.apache.nifi.processors.standard.PutFile;
import org.apache.nifi.scheduling.SchedulingStrategy;
import org.apache.nifi.serialization.record.RecordSchema;
import org.apache.nifi.util.Tuple;

@TriggerSerially
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@Tags(value={"file", "get", "list", "ingest", "source", "filesystem"})
@CapabilityDescription(value="Retrieves a listing of files from the input directory. For each file listed, creates a FlowFile that represents the file so that it can be fetched in conjunction with FetchFile. This Processor is designed to run on Primary Node only in a cluster when 'Input Directory Location' is set to 'Remote'. If the primary node changes, the new Primary Node will pick up where the previous node left off without duplicating all the data. When 'Input Directory Location' is 'Local', the 'Execution' mode can be anything, and synchronization won't happen. Unlike GetFile, this Processor does not delete any data from the local filesystem.")
@WritesAttributes(value={@WritesAttribute(attribute="filename", description="The name of the file that was read from filesystem."), @WritesAttribute(attribute="path", description="The path is set to the relative path of the file's directory on filesystem compared to the Input Directory property. For example, if Input Directory is set to /tmp, then files picked up from /tmp will have the path attribute set to \"/\". If the Recurse Subdirectories property is set to true and a file is picked up from /tmp/abc/1/2/3, then the path attribute will be set to \"abc/1/2/3/\"."), @WritesAttribute(attribute="absolute.path", description="The absolute.path is set to the absolute path of the file's directory on filesystem. For example, if the Input Directory property is set to /tmp, then files picked up from /tmp will have the path attribute set to \"/tmp/\". If the Recurse Subdirectories property is set to true and a file is picked up from /tmp/abc/1/2/3, then the path attribute will be set to \"/tmp/abc/1/2/3/\"."), @WritesAttribute(attribute="file.owner", description="The user that owns the file in filesystem"), @WritesAttribute(attribute="file.group", description="The group that owns the file in filesystem"), @WritesAttribute(attribute="file.size", description="The number of bytes in the file in filesystem"), @WritesAttribute(attribute="file.permissions", description="The permissions for the file in filesystem. This is formatted as 3 characters for the owner, 3 for the group, and 3 for other users. For example rw-rw-r--"), @WritesAttribute(attribute="file.lastModifiedTime", description="The timestamp of when the file in filesystem was last modified as 'yyyy-MM-dd'T'HH:mm:ssZ'"), @WritesAttribute(attribute="file.lastAccessTime", description="The timestamp of when the file in filesystem was last accessed as 'yyyy-MM-dd'T'HH:mm:ssZ'"), @WritesAttribute(attribute="file.creationTime", description="The timestamp of when the file in filesystem was created as 'yyyy-MM-dd'T'HH:mm:ssZ'")})
@SeeAlso(value={GetFile.class, PutFile.class, FetchFile.class})
@Stateful(scopes={Scope.LOCAL, Scope.CLUSTER}, description="After performing a listing of files, the timestamp of the newest file is stored. This allows the Processor to list only files that have been added or modified after this date the next time that the Processor is run. Whether the state is stored with a Local or Cluster scope depends on the value of the <Input Directory Location> property.")
@DefaultSchedule(strategy=SchedulingStrategy.TIMER_DRIVEN, period="1 min")
public class ListFile
extends AbstractListProcessor<FileInfo> {
    static final AllowableValue LOCATION_LOCAL = new AllowableValue("Local", "Local", "Input Directory is located on a local disk. State will be stored locally on each node in the cluster.");
    static final AllowableValue LOCATION_REMOTE = new AllowableValue("Remote", "Remote", "Input Directory is located on a remote system. State will be stored across the cluster so that the listing can be performed on Primary Node Only and another node can pick up where the last node left off, if the Primary Node changes");
    public static final PropertyDescriptor DIRECTORY = new PropertyDescriptor.Builder().name("Input Directory").description("The input directory from which files to pull files").required(true).addValidator(StandardValidators.createDirectoryExistsValidator((boolean)true, (boolean)false)).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).build();
    public static final PropertyDescriptor RECURSE = new PropertyDescriptor.Builder().name("Recurse Subdirectories").description("Indicates whether to list files from subdirectories of the directory").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("true").build();
    public static final PropertyDescriptor DIRECTORY_LOCATION = new PropertyDescriptor.Builder().name("Input Directory Location").description("Specifies where the Input Directory is located. This is used to determine whether state should be stored locally or across the cluster.").allowableValues(new DescribedValue[]{LOCATION_LOCAL, LOCATION_REMOTE}).defaultValue(LOCATION_LOCAL.getValue()).required(true).build();
    public static final PropertyDescriptor FILE_FILTER = new PropertyDescriptor.Builder().name("File Filter").description("Only files whose names match the given regular expression will be picked up").required(true).defaultValue("[^\\.].*").addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
    public static final PropertyDescriptor PATH_FILTER = new PropertyDescriptor.Builder().name("Path Filter").description("When " + RECURSE.getName() + " is true, then only subdirectories whose path matches the given regular expression will be scanned").required(false).addValidator(StandardValidators.REGULAR_EXPRESSION_VALIDATOR).build();
    public static final PropertyDescriptor INCLUDE_FILE_ATTRIBUTES = new PropertyDescriptor.Builder().name("Include File Attributes").description("Whether or not to include information such as the file's Last Modified Time and Owner as FlowFile Attributes. Depending on the File System being used, gathering this information can be expensive and as a result should be disabled. This is especially true of remote file shares.").allowableValues(new String[]{"true", "false"}).defaultValue("true").required(true).build();
    public static final PropertyDescriptor MIN_AGE = new PropertyDescriptor.Builder().name("Minimum File Age").description("The minimum age that a file must be in order to be pulled; any file younger than this amount of time (according to last modification date) will be ignored").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).defaultValue("0 sec").build();
    public static final PropertyDescriptor MAX_AGE = new PropertyDescriptor.Builder().name("Maximum File Age").description("The maximum age that a file must be in order to be pulled; any file older than this amount of time (according to last modification date) will be ignored").required(false).addValidator(StandardValidators.createTimePeriodValidator((long)100L, (TimeUnit)TimeUnit.MILLISECONDS, (long)Long.MAX_VALUE, (TimeUnit)TimeUnit.NANOSECONDS)).build();
    public static final PropertyDescriptor MIN_SIZE = new PropertyDescriptor.Builder().name("Minimum File Size").description("The minimum size that a file must be in order to be pulled").required(true).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).defaultValue("0 B").build();
    public static final PropertyDescriptor MAX_SIZE = new PropertyDescriptor.Builder().name("Maximum File Size").description("The maximum size that a file can be in order to be pulled").required(false).addValidator(StandardValidators.DATA_SIZE_VALIDATOR).build();
    public static final PropertyDescriptor IGNORE_HIDDEN_FILES = new PropertyDescriptor.Builder().name("Ignore Hidden Files").description("Indicates whether or not hidden files should be ignored").allowableValues(new String[]{"true", "false"}).defaultValue("true").required(true).build();
    public static final PropertyDescriptor TRACK_PERFORMANCE = new PropertyDescriptor.Builder().name("Track Performance").description("Whether or not the Processor should track the performance of disk access operations. If true, all accesses to disk will be recorded, including the file being accessed, the information being obtained, and how long it takes. This is then logged periodically at a DEBUG level. While the amount of data will be capped, this option may still consume a significant amount of heap (controlled by the 'Maximum Number of Files to Track' property), but it can be very useful for troubleshooting purposes if performance is poor is degraded.").required(true).allowableValues(new String[]{"true", "false"}).defaultValue("false").build();
    public static final PropertyDescriptor MAX_TRACKED_FILES = new PropertyDescriptor.Builder().name("Maximum Number of Files to Track").description("If the 'Track Performance' property is set to 'true', this property indicates the maximum number of files whose performance metrics should be held onto. A smaller value for this property will result in less heap utilization, while a larger value may provide more accurate insights into how the disk access operations are performing").required(true).addValidator(StandardValidators.POSITIVE_INTEGER_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).defaultValue("100000").build();
    public static final PropertyDescriptor MAX_DISK_OPERATION_TIME = new PropertyDescriptor.Builder().name("Max Disk Operation Time").description("The maximum amount of time that any single disk operation is expected to take. If any disk operation takes longer than this amount of time, a warning bulletin will be generated for each operation that exceeds this amount of time.").required(false).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).defaultValue("10 secs").build();
    public static final PropertyDescriptor MAX_LISTING_TIME = new PropertyDescriptor.Builder().name("Max Directory Listing Time").description("The maximum amount of time that listing any single directory is expected to take. If the listing for the directory specified by the 'Input Directory' property, or the listing of any subdirectory (if 'Recurse' is set to true) takes longer than this amount of time, a warning bulletin will be generated for each directory listing that exceeds this amount of time.").required(false).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).defaultValue("3 mins").build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(DIRECTORY, LISTING_STRATEGY, RECURSE, RECORD_WRITER, DIRECTORY_LOCATION, FILE_FILTER, PATH_FILTER, INCLUDE_FILE_ATTRIBUTES, MIN_AGE, MAX_AGE, MIN_SIZE, MAX_SIZE, IGNORE_HIDDEN_FILES, TARGET_SYSTEM_TIMESTAMP_PRECISION, ListedEntityTracker.TRACKING_STATE_CACHE, ListedEntityTracker.TRACKING_TIME_WINDOW, ListedEntityTracker.INITIAL_LISTING_TARGET, ListedEntityTracker.NODE_IDENTIFIER, TRACK_PERFORMANCE, MAX_TRACKED_FILES, MAX_DISK_OPERATION_TIME, MAX_LISTING_TIME);
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS);
    private volatile ScheduledExecutorService monitoringThreadPool;
    private volatile Future<?> monitoringFuture;
    private volatile boolean includeFileAttributes;
    private volatile PerformanceTracker performanceTracker;
    private volatile long performanceLoggingTimestamp = System.currentTimeMillis();
    public static final String FILE_CREATION_TIME_ATTRIBUTE = "file.creationTime";
    public static final String FILE_LAST_MODIFY_TIME_ATTRIBUTE = "file.lastModifiedTime";
    public static final String FILE_LAST_ACCESS_TIME_ATTRIBUTE = "file.lastAccessTime";
    public static final String FILE_SIZE_ATTRIBUTE = "file.size";
    public static final String FILE_OWNER_ATTRIBUTE = "file.owner";
    public static final String FILE_GROUP_ATTRIBUTE = "file.group";
    public static final String FILE_PERMISSIONS_ATTRIBUTE = "file.permissions";
    public static final String FILE_MODIFY_DATE_ATTR_FORMAT = "yyyy-MM-dd'T'HH:mm:ssZ";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ", Locale.US);

    protected void init(ProcessorInitializationContext context) {
        this.monitoringThreadPool = Executors.newScheduledThreadPool(1, r -> {
            Thread t = Executors.defaultThreadFactory().newThread(r);
            t.setName("Monitor ListFile Performance [UUID=" + context.getIdentifier() + "]");
            t.setDaemon(true);
            return t;
        });
    }

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) {
        this.includeFileAttributes = context.getProperty(INCLUDE_FILE_ATTRIBUTES).asBoolean();
        long maxDiskOperationMillis = context.getProperty(MAX_DISK_OPERATION_TIME).evaluateAttributeExpressions().asTimePeriod(TimeUnit.MILLISECONDS);
        long maxListingMillis = context.getProperty(MAX_LISTING_TIME).evaluateAttributeExpressions().asTimePeriod(TimeUnit.MILLISECONDS);
        boolean trackPerformance = context.getProperty(TRACK_PERFORMANCE).asBoolean();
        if (trackPerformance) {
            int maxEntries = context.getProperty(MAX_TRACKED_FILES).evaluateAttributeExpressions().asInteger();
            this.performanceTracker = new RollingMetricPerformanceTracker(this.getLogger(), maxDiskOperationMillis, maxEntries);
        } else {
            this.performanceTracker = new UntrackedPerformanceTracker(this.getLogger(), maxDiskOperationMillis);
        }
        long millisToKeepStats = TimeUnit.MINUTES.toMillis(15L);
        MonitorActiveTasks monitorTask = new MonitorActiveTasks(this.performanceTracker, this.getLogger(), maxDiskOperationMillis, maxListingMillis, millisToKeepStats);
        this.monitoringFuture = this.monitoringThreadPool.scheduleAtFixedRate(monitorTask, 15L, 15L, TimeUnit.SECONDS);
    }

    @OnStopped
    public void onStopped(ProcessContext context) {
        boolean trackPerformance;
        if (this.monitoringFuture != null) {
            this.monitoringFuture.cancel(true);
        }
        if (trackPerformance = context.getProperty(TRACK_PERFORMANCE).asBoolean().booleanValue()) {
            this.logPerformance();
        }
    }

    public void migrateProperties(PropertyConfiguration config) {
        super.migrateProperties(config);
        config.renameProperty("track-performance", TRACK_PERFORMANCE.getName());
        config.renameProperty("max-performance-metrics", MAX_TRACKED_FILES.getName());
        config.renameProperty("max-operation-time", MAX_DISK_OPERATION_TIME.getName());
        config.renameProperty("max-listing-time", MAX_LISTING_TIME.getName());
        config.renameProperty("et-state-cache", ListedEntityTracker.TRACKING_STATE_CACHE.getName());
        config.renameProperty("et-time-window", ListedEntityTracker.TRACKING_TIME_WINDOW.getName());
        config.renameProperty("et-initial-listing-target", ListedEntityTracker.INITIAL_LISTING_TARGET.getName());
        config.renameProperty("et-node-identifier", ListedEntityTracker.NODE_IDENTIFIER.getName());
    }

    protected PerformanceTracker getPerformanceTracker() {
        return this.performanceTracker;
    }

    public void logPerformance() {
        ComponentLog logger = this.getLogger();
        if (!logger.isDebugEnabled()) {
            return;
        }
        long earliestTimestamp = this.performanceTracker.getEarliestTimestamp();
        long millis = System.currentTimeMillis() - earliestTimestamp;
        long seconds = TimeUnit.MILLISECONDS.toSeconds(millis);
        for (DiskOperation operation : DiskOperation.values()) {
            OperationStatistics stats = this.performanceTracker.getOperationStatistics(operation);
            StringBuilder sb = new StringBuilder();
            if (stats.getCount() == 0L) {
                sb.append("Over the past ").append(seconds).append(" seconds, for Operation '").append((Object)operation).append("' there were no operations performed");
            } else {
                sb.append("Over the past ").append(seconds).append(" seconds, For Operation '").append((Object)operation).append("' there were ").append(stats.getCount()).append(" operations performed with an average time of ").append(stats.getAverage()).append(" milliseconds; Standard Deviation = ").append(stats.getStandardDeviation()).append(" millis; Min Time = ").append(stats.getMin()).append(" millis, Max Time = ").append(stats.getMax()).append(" millis");
                if (logger.isDebugEnabled()) {
                    Map<String, Long> outliers = stats.getOutliers();
                    sb.append("; ").append(stats.getOutliers().size()).append(" significant outliers: ");
                    sb.append(outliers);
                }
            }
            logger.debug("{}", new Object[]{sb});
        }
        this.performanceLoggingTimestamp = System.currentTimeMillis();
    }

    protected Map<String, String> createAttributes(FileInfo fileInfo, ProcessContext context) {
        HashMap<String, String> attributes = new HashMap<String, String>();
        String fullPath = fileInfo.getFullPathFileName();
        File file = new File(fullPath);
        Path filePath = file.toPath();
        Path directoryPath = new File(this.getPath(context)).toPath();
        Path relativePath = directoryPath.toAbsolutePath().relativize(filePath.getParent());
        Object relativePathString = relativePath.toString();
        relativePathString = ((String)relativePathString).isEmpty() ? "." + File.separator : (String)relativePathString + File.separator;
        Path absPath = filePath.toAbsolutePath();
        String absPathString = absPath.getParent().toString() + File.separator;
        attributes.put(CoreAttributes.PATH.key(), (String)relativePathString);
        attributes.put(CoreAttributes.FILENAME.key(), fileInfo.getFileName());
        attributes.put(CoreAttributes.ABSOLUTE_PATH.key(), absPathString);
        attributes.put(FILE_SIZE_ATTRIBUTE, Long.toString(fileInfo.getSize()));
        attributes.put(FILE_LAST_MODIFY_TIME_ATTRIBUTE, this.formatDateTime(fileInfo.getLastModifiedTime()));
        if (this.includeFileAttributes) {
            TimingInfo timingInfo = this.performanceTracker.getTimingInfo(relativePath.toString(), file.getName());
            try {
                FileStore store = Files.getFileStore(filePath);
                timingInfo.timeOperation(DiskOperation.RETRIEVE_BASIC_ATTRIBUTES, () -> {
                    if (store.supportsFileAttributeView("basic")) {
                        try {
                            BasicFileAttributeView view = Files.getFileAttributeView(filePath, BasicFileAttributeView.class, new LinkOption[0]);
                            BasicFileAttributes attrs = view.readAttributes();
                            attributes.put(FILE_CREATION_TIME_ATTRIBUTE, this.formatDateTime(attrs.creationTime().toMillis()));
                            attributes.put(FILE_LAST_ACCESS_TIME_ATTRIBUTE, this.formatDateTime(attrs.lastAccessTime().toMillis()));
                        }
                        catch (Exception exception) {
                            // empty catch block
                        }
                    }
                });
                timingInfo.timeOperation(DiskOperation.RETRIEVE_OWNER_ATTRIBUTES, () -> {
                    if (store.supportsFileAttributeView("owner")) {
                        try {
                            FileOwnerAttributeView view = Files.getFileAttributeView(filePath, FileOwnerAttributeView.class, new LinkOption[0]);
                            attributes.put(FILE_OWNER_ATTRIBUTE, view.getOwner().getName());
                        }
                        catch (Exception exception) {
                            // empty catch block
                        }
                    }
                });
                timingInfo.timeOperation(DiskOperation.RETRIEVE_POSIX_ATTRIBUTES, () -> {
                    if (store.supportsFileAttributeView("posix")) {
                        try {
                            PosixFileAttributeView view = Files.getFileAttributeView(filePath, PosixFileAttributeView.class, new LinkOption[0]);
                            attributes.put(FILE_PERMISSIONS_ATTRIBUTE, PosixFilePermissions.toString(view.readAttributes().permissions()));
                            attributes.put(FILE_GROUP_ATTRIBUTE, view.readAttributes().group().getName());
                        }
                        catch (Exception exception) {
                            // empty catch block
                        }
                    }
                });
            }
            catch (IOException ioe) {
                this.getLogger().warn("Error collecting attributes for file {}", new Object[]{absPathString, ioe});
            }
        }
        return attributes;
    }

    protected String getPath(ProcessContext context) {
        return context.getProperty(DIRECTORY).evaluateAttributeExpressions().getValue();
    }

    protected Scope getStateScope(PropertyContext context) {
        String location = context.getProperty(DIRECTORY_LOCATION).getValue();
        if (LOCATION_REMOTE.getValue().equalsIgnoreCase(location)) {
            return Scope.CLUSTER;
        }
        return Scope.LOCAL;
    }

    protected RecordSchema getRecordSchema() {
        return FileInfo.getRecordSchema();
    }

    protected Integer countUnfilteredListing(ProcessContext context) throws IOException {
        return this.performListing(context, 0L, AbstractListProcessor.ListingMode.CONFIGURATION_VERIFICATION, false).size();
    }

    protected List<FileInfo> performListing(ProcessContext context, Long minTimestamp, AbstractListProcessor.ListingMode listingMode) throws IOException {
        return this.performListing(context, minTimestamp, listingMode, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private List<FileInfo> performListing(ProcessContext context, final Long minTimestamp, final AbstractListProcessor.ListingMode listingMode, final boolean applyFilters) throws IOException {
        BiPredicate<Path, BasicFileAttributes> fileFilter;
        PerformanceTracker performanceTracker;
        String evaluatedPath = this.getPath(context);
        if (evaluatedPath == null || evaluatedPath.isBlank()) {
            throw new IOException("Blank Directory is not valid: review configured expression for Directory property");
        }
        final Path basePath = new File(evaluatedPath).toPath();
        Boolean recurse = context.getProperty(RECURSE).asBoolean();
        final HashMap lastModifiedMap = new HashMap();
        if (listingMode == AbstractListProcessor.ListingMode.EXECUTION) {
            performanceTracker = this.performanceTracker;
            fileFilter = this.createFileFilter(context, performanceTracker, applyFilters, basePath);
        } else {
            long maxDiskOperationMillis = context.getProperty(MAX_DISK_OPERATION_TIME).evaluateAttributeExpressions().asTimePeriod(TimeUnit.MILLISECONDS);
            performanceTracker = new UntrackedPerformanceTracker(this.getLogger(), maxDiskOperationMillis);
            fileFilter = this.createFileFilter(context, performanceTracker, applyFilters, basePath);
        }
        int maxDepth = recurse != false ? Integer.MAX_VALUE : 1;
        final BiPredicate<Path, BasicFileAttributes> matcher = new BiPredicate<Path, BasicFileAttributes>(){
            private long lastTimestamp = System.currentTimeMillis();

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public boolean test(Path path, BasicFileAttributes attributes) {
                if (!ListFile.this.isScheduled() && listingMode == AbstractListProcessor.ListingMode.EXECUTION) {
                    throw new ProcessorStoppedException();
                }
                long now = System.currentTimeMillis();
                long timeToList = now - this.lastTimestamp;
                this.lastTimestamp = now;
                Path relativeDirectory = basePath.relativize(path).getParent();
                String relativePath = relativeDirectory == null ? "" : relativeDirectory.toString();
                String filename = path.getFileName().toString();
                performanceTracker.acceptOperation(DiskOperation.RETRIEVE_NEXT_FILE_FROM_OS, relativePath, filename, timeToList);
                boolean isDirectory = attributes.isDirectory();
                if (isDirectory) {
                    performanceTracker.setActiveDirectory(relativePath);
                }
                TimedOperationKey operationKey = performanceTracker.beginOperation(DiskOperation.FILTER, relativePath, filename);
                try {
                    boolean matchesFilters;
                    boolean bl = matchesFilters = (minTimestamp == null || attributes.lastModifiedTime().toMillis() >= minTimestamp) && fileFilter.test(path, attributes);
                    if (!(isDirectory || applyFilters && !matchesFilters)) {
                        lastModifiedMap.put(path, attributes);
                        boolean bl2 = true;
                        return bl2;
                    }
                    boolean bl3 = false;
                    return bl3;
                }
                finally {
                    performanceTracker.completeOperation(operationKey);
                    if (TimeUnit.MILLISECONDS.toMinutes(System.currentTimeMillis() - ListFile.this.performanceLoggingTimestamp) >= 5L) {
                        ListFile.this.logPerformance();
                    }
                }
            }
        };
        try {
            long start = System.currentTimeMillis();
            final LinkedList<FileInfo> result = new LinkedList<FileInfo>();
            Files.walkFileTree(basePath, Set.of(FileVisitOption.FOLLOW_LINKS), maxDepth, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attributes) {
                    if (Files.isReadable(dir)) {
                        return FileVisitResult.CONTINUE;
                    }
                    ListFile.this.getLogger().debug("The following directory is not readable: {}", new Object[]{dir});
                    return FileVisitResult.SKIP_SUBTREE;
                }

                @Override
                public FileVisitResult visitFile(Path path, BasicFileAttributes attributes) {
                    if (matcher.test(path, attributes)) {
                        File file = path.toFile();
                        BasicFileAttributes fileAttributes = (BasicFileAttributes)lastModifiedMap.get(path);
                        FileInfo fileInfo = new FileInfo.Builder().directory(false).filename(file.getName()).fullPathFileName(file.getAbsolutePath()).lastModifiedTime(fileAttributes.lastModifiedTime().toMillis()).size(fileAttributes.size()).build();
                        result.add(fileInfo);
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path path, IOException e) {
                    if (e instanceof AccessDeniedException) {
                        ListFile.this.getLogger().debug("The following file is not readable: {}", new Object[]{path});
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    ListFile.this.getLogger().error("Error during visiting file {}", new Object[]{path, e});
                    return FileVisitResult.TERMINATE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException e) {
                    if (e != null) {
                        ListFile.this.getLogger().error("Error during visiting directory {}", new Object[]{dir, e});
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
            long millis = System.currentTimeMillis() - start;
            this.getLogger().debug("Took {} milliseconds to perform listing and gather {} entries", new Object[]{millis, result.size()});
            LinkedList<FileInfo> linkedList = result;
            return linkedList;
        }
        catch (ProcessorStoppedException pse) {
            this.getLogger().info("Processor was stopped so will not complete listing of Files");
            List<FileInfo> list = Collections.emptyList();
            return list;
        }
        finally {
            if (performanceTracker != null) {
                performanceTracker.completeActiveDirectory();
            }
        }
    }

    protected String getListingContainerName(ProcessContext context) {
        return String.format("%s Directory [%s]", context.getProperty(DIRECTORY_LOCATION).getValue(), this.getPath(context));
    }

    protected boolean isListingResetNecessary(PropertyDescriptor property) {
        return DIRECTORY.equals((Object)property) || RECURSE.equals((Object)property) || FILE_FILTER.equals((Object)property) || PATH_FILTER.equals((Object)property) || MIN_AGE.equals((Object)property) || MAX_AGE.equals((Object)property) || MIN_SIZE.equals((Object)property) || MAX_SIZE.equals((Object)property) || IGNORE_HIDDEN_FILES.equals((Object)property);
    }

    private String formatDateTime(long dateTime) {
        ZonedDateTime zonedDateTime = Instant.ofEpochMilli(dateTime).atZone(ZoneId.systemDefault());
        return DATE_TIME_FORMATTER.format(zonedDateTime);
    }

    private BiPredicate<Path, BasicFileAttributes> createFileFilter(ProcessContext context, PerformanceTracker performanceTracker, boolean applyFilters, Path basePath) {
        long minSize = context.getProperty(MIN_SIZE).asDataSize(DataUnit.B).longValue();
        Double maxSize = context.getProperty(MAX_SIZE).asDataSize(DataUnit.B);
        long minAge = context.getProperty(MIN_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        Long maxAge = context.getProperty(MAX_AGE).asTimePeriod(TimeUnit.MILLISECONDS);
        boolean ignoreHidden = context.getProperty(IGNORE_HIDDEN_FILES).asBoolean();
        String fileFilter = context.getProperty(FILE_FILTER).getValue();
        Pattern filePattern = Pattern.compile(fileFilter);
        boolean recurseDirs = context.getProperty(RECURSE).asBoolean();
        String pathPatternStr = context.getProperty(PATH_FILTER).getValue();
        Pattern pathPattern = !recurseDirs || pathPatternStr == null ? null : Pattern.compile(pathPatternStr);
        return (path, attributes) -> {
            if (!applyFilters) {
                return true;
            }
            if (minSize > attributes.size()) {
                return false;
            }
            if (maxSize != null && maxSize < (double)attributes.size()) {
                return false;
            }
            long fileAge = System.currentTimeMillis() - attributes.lastModifiedTime().toMillis();
            if (minAge > fileAge) {
                return false;
            }
            if (maxAge != null && maxAge < fileAge) {
                return false;
            }
            Path relativePath = basePath.relativize((Path)path);
            Path relativePathParent = relativePath.getParent();
            String relativeDir = relativePathParent == null ? "" : relativePathParent.toString();
            String filename = path.getFileName().toString();
            TimingInfo timingInfo = performanceTracker.getTimingInfo(relativeDir, filename);
            File file = path.toFile();
            if (pathPattern != null && !pathPattern.matcher(relativeDir).matches()) {
                return false;
            }
            boolean matchesFilter = filePattern.matcher(filename).matches();
            if (!matchesFilter) {
                return false;
            }
            if (!timingInfo.timeOperation(DiskOperation.CHECK_READABLE, () -> Files.isReadable(path)).booleanValue()) {
                return false;
            }
            if (ignoreHidden) {
                if (timingInfo.timeOperation(DiskOperation.CHECK_HIDDEN, file::isHidden).booleanValue()) {
                    return false;
                }
            }
            return true;
        };
    }

    public static final class RollingMetricPerformanceTracker
    implements PerformanceTracker {
        private final Map<String, String> directoryCanonicalization = new HashMap<String, String>();
        private final Map<Tuple<String, String>, TimingInfo> directoryToTimingInfo;
        private TimedOperationKey activeOperation;
        private long earliestTimestamp = System.currentTimeMillis();
        private final long maxDiskOperationMillis;
        private final ComponentLog logger;
        private String activeDirectory;
        private long activeDirectoryStartTime = -1L;

        public RollingMetricPerformanceTracker(ComponentLog logger, long maxDiskOperationMillis, final int maxEntries) {
            this.logger = logger;
            this.maxDiskOperationMillis = maxDiskOperationMillis;
            this.directoryToTimingInfo = new LinkedHashMap<Tuple<String, String>, TimingInfo>(){

                @Override
                protected boolean removeEldestEntry(Map.Entry<Tuple<String, String>, TimingInfo> eldest) {
                    return this.size() > maxEntries;
                }
            };
        }

        @Override
        public synchronized TimedOperationKey beginOperation(DiskOperation operation, String directory, String filename) {
            return new TimedOperationKey(operation, directory, filename, System.currentTimeMillis());
        }

        @Override
        public synchronized void completeOperation(TimedOperationKey operationKey) {
            TimingInfo timingInfo = this.getTimingInfo(operationKey.getDirectory(), operationKey.getFilename());
            timingInfo.accept(operationKey.getOperation(), System.currentTimeMillis() - operationKey.getStartTime());
        }

        @Override
        public synchronized void acceptOperation(DiskOperation operation, String directory, String filename, long millis) {
            String canonicalDirectory = this.directoryCanonicalization.computeIfAbsent(directory, key -> directory);
            Tuple key2 = new Tuple((Object)canonicalDirectory, (Object)filename);
            TimingInfo timingInfo = this.directoryToTimingInfo.computeIfAbsent((Tuple<String, String>)key2, k -> new TimingInfo(directory, filename, this, this.logger, this.maxDiskOperationMillis));
            timingInfo.accept(operation, millis);
        }

        @Override
        public synchronized TimingInfo getTimingInfo(String directory, String filename) {
            String canonicalDirectory = this.directoryCanonicalization.computeIfAbsent(directory, key -> directory);
            Tuple key2 = new Tuple((Object)canonicalDirectory, (Object)filename);
            TimingInfo timingInfo = this.directoryToTimingInfo.computeIfAbsent((Tuple<String, String>)key2, k -> new TimingInfo(directory, filename, this, this.logger, this.maxDiskOperationMillis));
            return timingInfo;
        }

        @Override
        public void setActiveOperation(TimedOperationKey activeOperation) {
            this.activeOperation = activeOperation;
        }

        @Override
        public void completeActiveOperation() {
            this.activeOperation = null;
        }

        @Override
        public synchronized TimedOperationKey getActiveOperation() {
            return this.activeOperation;
        }

        @Override
        public synchronized void setActiveDirectory(String directory) {
            this.activeDirectory = directory;
            this.activeDirectoryStartTime = System.currentTimeMillis();
        }

        @Override
        public synchronized void completeActiveDirectory() {
            this.activeDirectory = null;
            this.activeDirectoryStartTime = -1L;
        }

        @Override
        public synchronized long getActiveDirectoryStartTime() {
            return this.activeDirectoryStartTime;
        }

        @Override
        public synchronized String getActiveDirectory() {
            return this.activeDirectory;
        }

        @Override
        public synchronized int getTrackedFileCount() {
            return this.directoryToTimingInfo.size();
        }

        @Override
        public synchronized void purgeTimingInfo(long cutoff) {
            this.logger.debug("Purging any entries from Performance Tracker that is older than {}", new Object[]{new Date(cutoff)});
            Iterator<Map.Entry<Tuple<String, String>, TimingInfo>> itr = this.directoryToTimingInfo.entrySet().iterator();
            int purgedCount = 0;
            long earliestTimestamp = System.currentTimeMillis();
            while (itr.hasNext()) {
                Map.Entry<Tuple<String, String>, TimingInfo> entry = itr.next();
                TimingInfo timingInfo = entry.getValue();
                long creationTime = timingInfo.getCreationTimestamp();
                if (creationTime < cutoff) {
                    itr.remove();
                    ++purgedCount;
                    this.directoryCanonicalization.remove(entry.getKey().getKey());
                    continue;
                }
                earliestTimestamp = Math.min(earliestTimestamp, creationTime);
            }
            this.earliestTimestamp = earliestTimestamp;
            this.logger.debug("Purged {} entries from Performance Tracker; now holding {} entries", new Object[]{purgedCount, this.directoryToTimingInfo.size()});
        }

        @Override
        public long getEarliestTimestamp() {
            return this.earliestTimestamp;
        }

        @Override
        public synchronized OperationStatistics getOperationStatistics(DiskOperation operation) {
            long count = 0L;
            long sum = 0L;
            long min = 0L;
            long max = 0L;
            for (TimingInfo timingInfo : this.directoryToTimingInfo.values()) {
                long operationTime = timingInfo.getOperationTime(operation);
                if (operationTime < 0L) continue;
                sum += operationTime;
                if (count++ == 0L) {
                    min = operationTime;
                    max = operationTime;
                    continue;
                }
                min = Math.min(min, operationTime);
                max = Math.max(max, operationTime);
            }
            if (count == 0L) {
                return OperationStatistics.EMPTY;
            }
            double average = (double)sum / (double)count;
            double stdDeviation = this.calculateStdDev(average, count, operation);
            double outlierCutoff = average + 2.0 * stdDeviation;
            HashMap<String, Long> outliers = new HashMap<String, Long>();
            for (TimingInfo timingInfo : this.directoryToTimingInfo.values()) {
                long operationTime = timingInfo.getOperationTime(operation);
                if (operationTime <= 2L || !((double)operationTime > outlierCutoff)) continue;
                String directory = timingInfo.getDirectory();
                String filename = timingInfo.getFilename();
                String fullPath = directory.endsWith("/") ? directory + filename : directory + "/" + filename;
                outliers.put(fullPath, operationTime);
            }
            return new StandardOperationStatistics(min, max, count, average, stdDeviation, outliers);
        }

        private double calculateStdDev(double average, double count, DiskOperation operation) {
            double squaredDifferenceSum = 0.0;
            for (TimingInfo timingInfo : this.directoryToTimingInfo.values()) {
                long operationTime = timingInfo.getOperationTime(operation);
                if (operationTime < 0L) continue;
                double differenceSquared = Math.pow((double)operationTime - average, 2.0);
                squaredDifferenceSum += differenceSquared;
            }
            double squaredDifferenceAverage = squaredDifferenceSum / count;
            double stdDeviation = Math.pow(squaredDifferenceAverage, 0.5);
            return stdDeviation;
        }
    }

    static interface PerformanceTracker {
        public TimedOperationKey beginOperation(DiskOperation var1, String var2, String var3);

        public void completeOperation(TimedOperationKey var1);

        public void acceptOperation(DiskOperation var1, String var2, String var3, long var4);

        public TimingInfo getTimingInfo(String var1, String var2);

        public OperationStatistics getOperationStatistics(DiskOperation var1);

        public void setActiveOperation(TimedOperationKey var1);

        public void completeActiveOperation();

        public TimedOperationKey getActiveOperation();

        public void purgeTimingInfo(long var1);

        public long getEarliestTimestamp();

        public void setActiveDirectory(String var1);

        public void completeActiveDirectory();

        public String getActiveDirectory();

        public long getActiveDirectoryStartTime();

        public int getTrackedFileCount();
    }

    public static class UntrackedPerformanceTracker
    implements PerformanceTracker {
        private TimedOperationKey activeOperation = null;
        private String activeDirectory;
        private long activeDirectoryStartTime = -1L;
        private final ComponentLog logger;
        private final long maxDiskOperationMillis;

        public UntrackedPerformanceTracker(ComponentLog logger, long maxDiskOperationMillis) {
            this.logger = logger;
            this.maxDiskOperationMillis = maxDiskOperationMillis;
        }

        @Override
        public TimedOperationKey beginOperation(DiskOperation operation, String directory, String filename) {
            return null;
        }

        @Override
        public void completeOperation(TimedOperationKey operationKey) {
        }

        @Override
        public void acceptOperation(DiskOperation operation, String directory, String filename, long millis) {
        }

        @Override
        public TimingInfo getTimingInfo(String directory, String filename) {
            return new TimingInfo(directory, filename, this, this.logger, this.maxDiskOperationMillis);
        }

        @Override
        public OperationStatistics getOperationStatistics(DiskOperation operation) {
            return OperationStatistics.EMPTY;
        }

        @Override
        public synchronized void setActiveOperation(TimedOperationKey operationKey) {
            this.activeOperation = operationKey;
        }

        @Override
        public synchronized void completeActiveOperation() {
            this.activeOperation = null;
        }

        @Override
        public synchronized TimedOperationKey getActiveOperation() {
            return this.activeOperation;
        }

        @Override
        public void purgeTimingInfo(long cutoff) {
        }

        @Override
        public long getEarliestTimestamp() {
            return System.currentTimeMillis();
        }

        @Override
        public synchronized void setActiveDirectory(String directory) {
            this.activeDirectory = directory;
            this.activeDirectoryStartTime = System.currentTimeMillis();
        }

        @Override
        public synchronized void completeActiveDirectory() {
            this.activeDirectory = null;
            this.activeDirectoryStartTime = -1L;
        }

        @Override
        public synchronized long getActiveDirectoryStartTime() {
            return this.activeDirectoryStartTime;
        }

        @Override
        public synchronized String getActiveDirectory() {
            return this.activeDirectory;
        }

        @Override
        public int getTrackedFileCount() {
            return 0;
        }
    }

    static class MonitorActiveTasks
    implements Runnable {
        private final PerformanceTracker performanceTracker;
        private final ComponentLog logger;
        private final long maxDiskOperationMillis;
        private final long maxListingMillis;
        private final long millisToKeepStats;
        private long lastPurgeTimestamp = 0L;

        public MonitorActiveTasks(PerformanceTracker tracker, ComponentLog logger, long maxDiskOperationMillis, long maxListingMillis, long millisToKeepStats) {
            this.performanceTracker = tracker;
            this.logger = logger;
            this.maxDiskOperationMillis = maxDiskOperationMillis;
            this.maxListingMillis = maxListingMillis;
            this.millisToKeepStats = millisToKeepStats;
        }

        @Override
        public void run() {
            this.monitorActiveOperation();
            this.monitorActiveDirectory();
            long now = System.currentTimeMillis();
            long millisSincePurge = now - this.lastPurgeTimestamp;
            if (millisSincePurge > TimeUnit.SECONDS.toMillis(60L)) {
                this.performanceTracker.purgeTimingInfo(now - this.millisToKeepStats);
                this.lastPurgeTimestamp = System.currentTimeMillis();
            }
        }

        private void monitorActiveOperation() {
            TimedOperationKey activeOperation = this.performanceTracker.getActiveOperation();
            if (activeOperation == null) {
                return;
            }
            long activeTime = System.currentTimeMillis() - activeOperation.getStartTime();
            if (activeTime > this.maxDiskOperationMillis) {
                String directory = activeOperation.getDirectory();
                String filename = activeOperation.getFilename();
                Object fullPath = directory.isEmpty() ? filename : (directory.endsWith("/") ? directory + filename : directory + "/" + filename);
                this.logger.warn("This Processor has currently spent {} milliseconds performing the {} action on {}, which exceeds the configured threshold of {} milliseconds", new Object[]{activeTime, activeOperation.getOperation(), fullPath, this.maxDiskOperationMillis});
            }
        }

        private void monitorActiveDirectory() {
            String activeDirectory = this.performanceTracker.getActiveDirectory();
            long startTime = this.performanceTracker.getActiveDirectoryStartTime();
            if (startTime <= 0L) {
                return;
            }
            long activeMillis = System.currentTimeMillis() - startTime;
            if (activeMillis > this.maxListingMillis) {
                String fullPath = activeDirectory.isEmpty() ? "the base directory" : activeDirectory;
                this.logger.warn("This processor has currently spent {} milliseconds performing the listing of {}, which exceeds the configured threshold of {} milliseconds", new Object[]{activeMillis, fullPath, this.maxListingMillis});
            }
        }
    }

    private static enum DiskOperation {
        RETRIEVE_BASIC_ATTRIBUTES,
        RETRIEVE_OWNER_ATTRIBUTES,
        RETRIEVE_POSIX_ATTRIBUTES,
        CHECK_HIDDEN,
        CHECK_READABLE,
        FILTER,
        RETRIEVE_NEXT_FILE_FROM_OS;

    }

    static interface OperationStatistics {
        public static final OperationStatistics EMPTY = new OperationStatistics(){

            @Override
            public long getMin() {
                return 0L;
            }

            @Override
            public long getMax() {
                return 0L;
            }

            @Override
            public long getCount() {
                return 0L;
            }

            @Override
            public double getAverage() {
                return 0.0;
            }

            @Override
            public double getStandardDeviation() {
                return 0.0;
            }

            @Override
            public Map<String, Long> getOutliers() {
                return Collections.emptyMap();
            }
        };

        public long getMin();

        public long getMax();

        public long getCount();

        public double getAverage();

        public double getStandardDeviation();

        public Map<String, Long> getOutliers();
    }

    private static class TimingInfo {
        private final String directory;
        private final String filename;
        private final int[] operationTimes;
        private final PerformanceTracker tracker;
        private final long creationTimestamp;
        private final ComponentLog logger;
        private final long maxDiskOperationMillis;

        public TimingInfo(String directory, String filename, PerformanceTracker tracker, ComponentLog logger, long maxDiskOperationMillis) {
            this.directory = directory;
            this.filename = filename;
            this.tracker = tracker;
            this.logger = logger;
            this.maxDiskOperationMillis = maxDiskOperationMillis;
            this.creationTimestamp = System.currentTimeMillis();
            this.operationTimes = new int[DiskOperation.values().length];
            Arrays.fill(this.operationTimes, -1);
        }

        public String getDirectory() {
            return this.directory;
        }

        public String getFilename() {
            return this.filename;
        }

        public void accept(DiskOperation operation, long duration) {
            this.operationTimes[operation.ordinal()] = (int)duration;
            if (duration > this.maxDiskOperationMillis) {
                String fullPath = this.getFullPath();
                this.logger.warn("This Processor completed action {} on {} in {} milliseconds, which exceeds the configured threshold of {} milliseconds", new Object[]{operation, fullPath, duration, this.maxDiskOperationMillis});
            }
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Performing operation {} on {} took {} milliseconds", new Object[]{operation, this.getFullPath(), duration});
            }
        }

        private String getFullPath() {
            if (this.directory.isEmpty()) {
                return this.filename;
            }
            return this.directory.endsWith("/") ? this.directory + this.filename : this.directory + "/" + this.filename;
        }

        public long getOperationTime(DiskOperation operation) {
            return this.operationTimes[operation.ordinal()];
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private <T> T timeOperation(DiskOperation operation, Supplier<T> function) {
            long start = System.currentTimeMillis();
            TimedOperationKey operationKey = new TimedOperationKey(operation, this.directory, this.filename, start);
            this.tracker.setActiveOperation(operationKey);
            try {
                T value = function.get();
                long millis = System.currentTimeMillis() - start;
                this.accept(operation, millis);
                T t = value;
                return t;
            }
            finally {
                this.tracker.completeActiveOperation();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void timeOperation(DiskOperation operation, Runnable task) {
            long start = System.currentTimeMillis();
            TimedOperationKey operationKey = new TimedOperationKey(operation, this.directory, this.filename, start);
            this.tracker.setActiveOperation(operationKey);
            try {
                task.run();
                long millis = System.currentTimeMillis() - start;
                this.accept(operation, millis);
            }
            finally {
                this.tracker.completeActiveOperation();
            }
        }

        public long getCreationTimestamp() {
            return this.creationTimestamp;
        }
    }

    private static class ProcessorStoppedException
    extends RuntimeException {
        private ProcessorStoppedException() {
        }
    }

    private static class TimedOperationKey {
        private final DiskOperation operation;
        private final String directory;
        private final String filename;
        private final long startTime;

        public TimedOperationKey(DiskOperation operation, String directory, String filename, long startTime) {
            this.operation = operation;
            this.startTime = startTime;
            this.directory = directory;
            this.filename = filename;
        }

        public DiskOperation getOperation() {
            return this.operation;
        }

        public String getDirectory() {
            return this.directory;
        }

        public String getFilename() {
            return this.filename;
        }

        public long getStartTime() {
            return this.startTime;
        }
    }

    private static class StandardOperationStatistics
    implements OperationStatistics {
        private final long min;
        private final long max;
        private final long count;
        private final double average;
        private final double stdDev;
        private final Map<String, Long> outliers;

        public StandardOperationStatistics(long min, long max, long count, double average, double stdDev, Map<String, Long> outliers) {
            this.min = min;
            this.max = max;
            this.count = count;
            this.average = average;
            this.stdDev = stdDev;
            this.outliers = outliers;
        }

        @Override
        public long getMin() {
            return this.min;
        }

        @Override
        public long getMax() {
            return this.max;
        }

        @Override
        public long getCount() {
            return this.count;
        }

        @Override
        public double getAverage() {
            return this.average;
        }

        @Override
        public double getStandardDeviation() {
            return this.stdDev;
        }

        @Override
        public Map<String, Long> getOutliers() {
            return this.outliers;
        }
    }
}

