/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.questdb.embedded;

import io.questdb.cairo.CairoConfiguration;
import io.questdb.cairo.CairoEngine;
import io.questdb.cairo.DefaultCairoConfiguration;
import io.questdb.cairo.TableToken;
import io.questdb.cairo.sql.TableRecordMetadata;
import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.apache.commons.lang3.concurrent.BasicThreadFactory;
import org.apache.nifi.questdb.Client;
import org.apache.nifi.questdb.DatabaseException;
import org.apache.nifi.questdb.DatabaseManager;
import org.apache.nifi.questdb.QueryResultProcessor;
import org.apache.nifi.questdb.QueryRowContext;
import org.apache.nifi.questdb.embedded.ClientDisconnectedException;
import org.apache.nifi.questdb.embedded.ConditionAwareClient;
import org.apache.nifi.questdb.embedded.ConditionFailedException;
import org.apache.nifi.questdb.embedded.CorruptedDatabaseException;
import org.apache.nifi.questdb.embedded.EmbeddedClient;
import org.apache.nifi.questdb.embedded.EmbeddedDatabaseManagerContext;
import org.apache.nifi.questdb.embedded.EmbeddedDatabaseManagerStatus;
import org.apache.nifi.questdb.embedded.LockUnsuccessfulException;
import org.apache.nifi.questdb.embedded.LockedClient;
import org.apache.nifi.questdb.embedded.ManagedTableDefinition;
import org.apache.nifi.questdb.embedded.NoOpClient;
import org.apache.nifi.questdb.embedded.RetryingClient;
import org.apache.nifi.questdb.embedded.RolloverWorker;
import org.apache.nifi.util.file.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class EmbeddedDatabaseManager
implements DatabaseManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(EmbeddedDatabaseManager.class);
    private final String id = UUID.randomUUID().toString();
    private final AtomicReference<EmbeddedDatabaseManagerStatus> state = new AtomicReference<EmbeddedDatabaseManagerStatus>(EmbeddedDatabaseManagerStatus.UNINITIALIZED);
    private final ReadWriteLock databaseStructureLock = new ReentrantReadWriteLock();
    private final EmbeddedDatabaseManagerContext context;
    private final AtomicReference<CairoEngine> engine = new AtomicReference();
    private final List<ScheduledFuture<?>> scheduledFutures = new ArrayList();
    private final ScheduledExecutorService scheduledExecutorService = Executors.newScheduledThreadPool(2, (ThreadFactory)new BasicThreadFactory.Builder().namingPattern("EmbeddedQuestDbManagerWorker-" + this.id + "-%d").build());

    EmbeddedDatabaseManager(EmbeddedDatabaseManagerContext context) {
        this.context = context;
    }

    @Override
    public void init() {
        if (this.state.get() != EmbeddedDatabaseManagerStatus.UNINITIALIZED) {
            throw new IllegalStateException("Manager is already initialized");
        }
        this.ensureDatabaseIsReady();
        this.startRollover();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void ensureDatabaseIsReady() {
        boolean successful = false;
        try {
            block13: {
                this.databaseStructureLock.writeLock().lock();
                this.state.set(EmbeddedDatabaseManagerStatus.REPAIRING);
                try {
                    this.ensurePersistLocationIsAccessible();
                    this.ensureConnectionEstablished();
                    this.ensureTablesAreInPlaceAndHealthy();
                    successful = true;
                }
                catch (CorruptedDatabaseException e) {
                    boolean couldMoveOldToBackup = false;
                    try {
                        LOGGER.error("Database is corrupted. Recreation is triggered. Manager tries to move corrupted database files to the backup location: {}", (Object)this.context.getBackupLocation(), (Object)e);
                        File backupFolder = new File(this.context.getBackupLocationAsPath().toFile(), "backup_" + System.currentTimeMillis());
                        FileUtils.ensureDirectoryExistAndCanAccess((File)this.context.getBackupLocationAsPath().toFile());
                        Files.move(this.context.getPersistLocationAsPath(), backupFolder.toPath(), new CopyOption[0]);
                        couldMoveOldToBackup = true;
                    }
                    catch (IOException ex) {
                        LOGGER.error("Could not create backup", (Throwable)ex);
                    }
                    if (!couldMoveOldToBackup) {
                        try {
                            FileUtils.deleteFile((File)this.context.getPersistLocationAsPath().toFile(), (boolean)true);
                            couldMoveOldToBackup = true;
                        }
                        catch (IOException ex) {
                            LOGGER.error("Could not clean up corrupted database", (Throwable)ex);
                        }
                    }
                    if (!couldMoveOldToBackup) break block13;
                    try {
                        this.ensurePersistLocationIsAccessible();
                        this.ensureConnectionEstablished();
                        this.ensureTablesAreInPlaceAndHealthy();
                        successful = true;
                    }
                    catch (CorruptedDatabaseException ex) {
                        LOGGER.error("Could not create backup", (Throwable)ex);
                    }
                }
            }
            this.state.set(successful ? EmbeddedDatabaseManagerStatus.HEALTHY : EmbeddedDatabaseManagerStatus.CORRUPTED);
            if (!successful) {
                this.engine.set(null);
            }
            this.databaseStructureLock.writeLock().unlock();
        }
        catch (Throwable throwable) {
            this.state.set(successful ? EmbeddedDatabaseManagerStatus.HEALTHY : EmbeddedDatabaseManagerStatus.CORRUPTED);
            if (!successful) {
                this.engine.set(null);
            }
            this.databaseStructureLock.writeLock().unlock();
            throw throwable;
        }
    }

    private void ensurePersistLocationIsAccessible() throws CorruptedDatabaseException {
        try {
            FileUtils.ensureDirectoryExistAndCanAccess((File)this.context.getPersistLocationAsPath().toFile());
        }
        catch (Exception e) {
            throw new CorruptedDatabaseException(String.format("Database directory creation failed [%s]", this.context.getPersistLocationAsPath()), e);
        }
    }

    private void ensureConnectionEstablished() throws CorruptedDatabaseException {
        if (this.engine.get() != null) {
            ((CairoEngine)this.engine.getAndSet(null)).close();
        }
        String absolutePath = this.context.getPersistLocationAsPath().toFile().getAbsolutePath();
        DefaultCairoConfiguration configuration = new DefaultCairoConfiguration((CharSequence)absolutePath);
        try {
            CairoEngine engine = new CairoEngine((CairoConfiguration)configuration);
            LOGGER.info("Database connection successful [{}]", (Object)absolutePath);
            this.engine.set(engine);
        }
        catch (Exception e) {
            throw new CorruptedDatabaseException(String.format("Database connection failed [%s]", absolutePath), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void ensureTablesAreInPlaceAndHealthy() throws CorruptedDatabaseException {
        Map<String, File> databaseFiles = Arrays.stream(this.context.getPersistLocationAsPath().toFile().listFiles()).collect(Collectors.toMap(f -> f.getAbsolutePath().substring(this.context.getPersistLocationAsPath().toFile().getAbsolutePath().length() + 1), f -> f));
        Client client = this.getUnmanagedClient();
        try {
            for (ManagedTableDefinition tableDefinition : this.context.getTableDefinitions()) {
                if (!databaseFiles.containsKey(tableDefinition.getName())) {
                    try {
                        LOGGER.debug("Creating table {}", (Object)tableDefinition.getName());
                        client.execute(tableDefinition.getDefinition());
                        LOGGER.debug("Table {} is created", (Object)tableDefinition.getName());
                        continue;
                    }
                    catch (DatabaseException e) {
                        throw new CorruptedDatabaseException(String.format("Creating table [%s] has failed", tableDefinition.getName()), e);
                    }
                }
                if (databaseFiles.get(tableDefinition.getName()).isDirectory()) continue;
                throw new CorruptedDatabaseException(String.format("Table %s cannot be created because there is already a file exists with the given name", tableDefinition.getName()));
            }
            for (ManagedTableDefinition tableDefinition : this.context.getTableDefinitions()) {
                try {
                    TableToken tableToken = this.engine.get().getTableTokenIfExists((CharSequence)tableDefinition.getName());
                    if (tableToken.isWal()) {
                        TableRecordMetadata metadata = this.engine.get().getSequencerMetadata(tableToken);
                        metadata.close();
                    }
                    client.query(String.format("SELECT * FROM %S LIMIT 1", tableDefinition.getName()), new QueryResultProcessor<Object>(this){

                        @Override
                        public void processRow(QueryRowContext context) {
                        }

                        @Override
                        public Object getResult() {
                            return null;
                        }
                    });
                }
                catch (Exception e) {
                    throw new CorruptedDatabaseException(e);
                    return;
                }
            }
        }
        finally {
            try {
                client.disconnect();
            }
            catch (DatabaseException e) {
                throw new CorruptedDatabaseException(e);
            }
        }
    }

    private void startRollover() {
        RolloverWorker rolloverWorker = new RolloverWorker(this.acquireClient(), this.context.getTableDefinitions());
        ScheduledFuture<?> rolloverFuture = this.scheduledExecutorService.scheduleWithFixedDelay(rolloverWorker, this.context.getRolloverFrequency().toMillis(), this.context.getRolloverFrequency().toMillis(), TimeUnit.MILLISECONDS);
        this.scheduledFutures.add(rolloverFuture);
        LOGGER.debug("Rollover started");
    }

    private void stopRollover() {
        LOGGER.debug("Rollover shutdown started");
        int cancelCompleted = 0;
        int cancelFailed = 0;
        for (ScheduledFuture<?> scheduledFuture : this.scheduledFutures) {
            boolean cancelled = scheduledFuture.cancel(true);
            if (cancelled) {
                ++cancelCompleted;
                continue;
            }
            ++cancelFailed;
        }
        LOGGER.debug("Rollover shutdown task cancellation status: completed [{}] failed [{}]", (Object)cancelCompleted, (Object)cancelFailed);
        List<Runnable> tasks = this.scheduledExecutorService.shutdownNow();
        LOGGER.debug("Rollover Scheduled Task Service shutdown remaining tasks [{}]", (Object)tasks.size());
    }

    private Client getUnmanagedClient() {
        return new EmbeddedClient(() -> this.engine.get());
    }

    @Override
    public Client acquireClient() {
        this.checkIfManagerIsInitialised();
        NoOpClient fallback = new NoOpClient();
        if (this.state.get() == EmbeddedDatabaseManagerStatus.CORRUPTED) {
            LOGGER.error("The database is corrupted: Status History will not be stored");
            return fallback;
        }
        LockedClient lockedClient = new LockedClient(this.databaseStructureLock.readLock(), this.context.getLockAttemptTime(), new ConditionAwareClient(() -> this.state.get() == EmbeddedDatabaseManagerStatus.HEALTHY, this.getUnmanagedClient()));
        return RetryingClient.getInstance(this.context.getNumberOfAttemptedRetries(), this::errorAction, lockedClient, fallback);
    }

    private void checkIfManagerIsInitialised() {
        if (this.state.get() == EmbeddedDatabaseManagerStatus.UNINITIALIZED) {
            throw new IllegalStateException("The state of the database manager is not initialized");
        }
    }

    private void errorAction(int attemptNumber, Throwable throwable) {
        if (this.shouldRestoreDatabase(attemptNumber, throwable)) {
            LOGGER.error("Database manager tries to restore database after the first failed attempt if necessary");
            this.ensureDatabaseIsReady();
        } else {
            LOGGER.warn("Error happened at attempt: {}", (Object)attemptNumber, (Object)throwable);
        }
    }

    private boolean shouldRestoreDatabase(int attemptNumber, Throwable throwable) {
        if (this.state.get() == EmbeddedDatabaseManagerStatus.CORRUPTED || this.state.get() == EmbeddedDatabaseManagerStatus.CLOSED) {
            return false;
        }
        if (throwable instanceof ConditionFailedException || throwable instanceof LockUnsuccessfulException || throwable instanceof ClientDisconnectedException) {
            return false;
        }
        return attemptNumber == 1;
    }

    @Override
    public void close() {
        this.databaseStructureLock.writeLock().lock();
        this.checkIfManagerIsInitialised();
        this.stopRollover();
        this.state.set(EmbeddedDatabaseManagerStatus.CLOSED);
        CairoEngine engine = this.engine.get();
        if (engine != null) {
            engine.close();
        }
        this.databaseStructureLock.writeLock().unlock();
    }
}

