/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.hadoop.inotify;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hdfs.DFSInotifyEventInputStream;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hdfs.inotify.Event;
import org.apache.hadoop.hdfs.inotify.EventBatch;
import org.apache.hadoop.hdfs.inotify.MissingEventsException;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.Stateful;
import org.apache.nifi.annotation.behavior.TriggerSerially;
import org.apache.nifi.annotation.behavior.TriggerWhenEmpty;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.Validator;
import org.apache.nifi.components.state.Scope;
import org.apache.nifi.components.state.StateMap;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.hadoop.AbstractHadoopProcessor;
import org.apache.nifi.processors.hadoop.FetchHDFS;
import org.apache.nifi.processors.hadoop.GetHDFS;
import org.apache.nifi.processors.hadoop.ListHDFS;
import org.apache.nifi.processors.hadoop.PutHDFS;
import org.apache.nifi.processors.hadoop.inotify.EventTypeValidator;
import org.apache.nifi.processors.hadoop.inotify.NotificationEventPathFilter;

@TriggerSerially
@TriggerWhenEmpty
@Tags(value={"hadoop", "events", "inotify", "notifications", "filesystem"})
@WritesAttributes(value={@WritesAttribute(attribute="mime.type", description="This is always application/json."), @WritesAttribute(attribute="hdfs.inotify.event.type", description="This will specify the specific HDFS notification event type. Currently there are six types of events (append, close, create, metadata, rename, and unlink)."), @WritesAttribute(attribute="hdfs.inotify.event.path", description="The specific path that the event is tied to.")})
@InputRequirement(value=InputRequirement.Requirement.INPUT_FORBIDDEN)
@CapabilityDescription(value="This processor polls the notification events provided by the HdfsAdmin API. Since this uses the HdfsAdmin APIs it is required to run as an HDFS super user. Currently there are six types of events (append, close, create, metadata, rename, and unlink). Please see org.apache.hadoop.hdfs.inotify.Event documentation for full explanations of each event. This processor will poll for new events based on a defined duration. For each event received a new flow file will be created with the expected attributes and the event itself serialized to JSON and written to the flow file's content. For example, if event.type is APPEND then the content of the flow file will contain a JSON file containing the information about the append event. If successful the flow files are sent to the 'success' relationship. Be careful of where the generated flow files are stored. If the flow files are stored in one of processor's watch directories there will be a never ending flow of events. It is also important to be aware that this processor must consume all events. The filtering must happen within the processor. This is because the HDFS admin's event notifications API does not have filtering.")
@Stateful(scopes={Scope.CLUSTER}, description="The last used transaction id is stored. This is used ")
@SeeAlso(value={GetHDFS.class, FetchHDFS.class, PutHDFS.class, ListHDFS.class})
public class GetHDFSEvents
extends AbstractHadoopProcessor {
    static final PropertyDescriptor POLL_DURATION = new PropertyDescriptor.Builder().name("Poll Duration").description("The time before the polling method returns with the next batch of events if they exist. It may exceed this amount of time by up to the time required for an RPC to the NameNode.").defaultValue("1 second").required(true).addValidator(StandardValidators.TIME_PERIOD_VALIDATOR).build();
    static final PropertyDescriptor HDFS_PATH_TO_WATCH = new PropertyDescriptor.Builder().name("HDFS Path to Watch").description("The HDFS path to get event notifications for. This property accepts both expression language and regular expressions. This will be evaluated during the OnScheduled phase.").required(true).expressionLanguageSupported(ExpressionLanguageScope.ENVIRONMENT).addValidator(StandardValidators.createRegexValidator((int)0, (int)Integer.MAX_VALUE, (boolean)true)).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    static final PropertyDescriptor IGNORE_HIDDEN_FILES = new PropertyDescriptor.Builder().name("Ignore Hidden Files").description("If true and the final component of the path associated with a given event starts with a '.' then that event will not be processed.").required(true).addValidator(StandardValidators.BOOLEAN_VALIDATOR).allowableValues(new String[]{"true", "false"}).defaultValue("false").build();
    static final PropertyDescriptor EVENT_TYPES = new PropertyDescriptor.Builder().name("Event Types to Filter On").description("A comma-separated list of event types to process. Valid event types are: append, close, create, metadata, rename, and unlink. Case does not matter.").addValidator((Validator)new EventTypeValidator()).required(true).defaultValue("append, close, create, metadata, rename, unlink").build();
    static final PropertyDescriptor NUMBER_OF_RETRIES_FOR_POLL = new PropertyDescriptor.Builder().name("IOException Retries During Event Polling").description("According to the HDFS admin API for event polling it is good to retry at least a few times. This number defines how many times the poll will be retried if it throws an IOException.").addValidator(StandardValidators.NON_NEGATIVE_INTEGER_VALIDATOR).required(true).defaultValue("3").build();
    static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("A flow file with updated information about a specific event will be sent to this relationship.").build();
    private static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = Stream.concat(GetHDFSEvents.getCommonPropertyDescriptors().stream(), Stream.of(POLL_DURATION, HDFS_PATH_TO_WATCH, IGNORE_HIDDEN_FILES, EVENT_TYPES, NUMBER_OF_RETRIES_FOR_POLL)).toList();
    private static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS);
    private static final String LAST_TX_ID = "last.tx.id";
    private volatile long lastTxId = -1L;
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private NotificationConfig notificationConfig;

    protected List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    @OnScheduled
    public void onSchedule(ProcessContext context) {
        this.notificationConfig = new NotificationConfig(context);
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        try {
            StateMap state = session.getState(Scope.CLUSTER);
            String txIdAsString = state.get(LAST_TX_ID);
            if (txIdAsString != null && !"".equals(txIdAsString)) {
                this.lastTxId = Long.parseLong(txIdAsString);
            }
        }
        catch (IOException e) {
            this.getLogger().error("Unable to retrieve last transaction ID. Must retrieve last processed transaction ID before processing can occur.", (Throwable)e);
            context.yield();
            return;
        }
        try {
            int retries = context.getProperty(NUMBER_OF_RETRIES_FOR_POLL).asInteger();
            TimeUnit pollDurationTimeUnit = TimeUnit.MICROSECONDS;
            long pollDuration = context.getProperty(POLL_DURATION).asTimePeriod(pollDurationTimeUnit);
            DFSInotifyEventInputStream eventStream = this.lastTxId == -1L ? this.getHdfsAdmin().getInotifyEventStream() : this.getHdfsAdmin().getInotifyEventStream(this.lastTxId);
            EventBatch eventBatch = this.getEventBatch(eventStream, pollDuration, pollDurationTimeUnit, retries);
            if (eventBatch != null && eventBatch.getEvents() != null) {
                if (eventBatch.getEvents().length > 0) {
                    ArrayList<FlowFile> flowFiles = new ArrayList<FlowFile>(eventBatch.getEvents().length);
                    for (Event e : eventBatch.getEvents()) {
                        if (!this.toProcessEvent(context, e)) continue;
                        this.getLogger().debug("Creating flow file for event: {}.", new Object[]{e});
                        String path = this.getPath(e);
                        FlowFile flowFile = session.create();
                        flowFile = session.putAttribute(flowFile, CoreAttributes.MIME_TYPE.key(), "application/json");
                        flowFile = session.putAttribute(flowFile, "hdfs.inotify.event.type", e.getEventType().name());
                        flowFile = session.putAttribute(flowFile, "hdfs.inotify.event.path", path);
                        flowFile = session.write(flowFile, out -> out.write(OBJECT_MAPPER.writeValueAsBytes((Object)e)));
                        flowFiles.add(flowFile);
                    }
                    for (FlowFile flowFile : flowFiles) {
                        String path = flowFile.getAttribute("hdfs.inotify.event.path");
                        String transitUri = path.startsWith("/") ? "hdfs:/" + path : "hdfs://" + path;
                        this.getLogger().debug("Transferring flow file {} and creating provenance event with URI {}.", new Object[]{flowFile, transitUri});
                        session.transfer(flowFile, REL_SUCCESS);
                        session.getProvenanceReporter().receive(flowFile, transitUri);
                    }
                }
                this.lastTxId = eventBatch.getTxid();
            }
        }
        catch (IOException | InterruptedException e) {
            this.getLogger().error("Unable to get notification information", (Throwable)e);
            context.yield();
            return;
        }
        catch (MissingEventsException e) {
            this.lastTxId = -1L;
            this.getLogger().error("Unable to get notification information. Setting transaction id to -1. This may cause some events to get missed. Please see javadoc for org.apache.hadoop.hdfs.client.HdfsAdmin#getInotifyEventStream", (Throwable)e);
        }
        this.updateClusterStateForTxId(session);
    }

    private EventBatch getEventBatch(DFSInotifyEventInputStream eventStream, long duration, TimeUnit timeUnit, int retries) throws IOException, InterruptedException, MissingEventsException {
        int i = 0;
        while (true) {
            try {
                ++i;
                return eventStream.poll(duration, timeUnit);
            }
            catch (IOException e) {
                if (i > retries) {
                    this.getLogger().debug("Failed to poll for event batch. Reached max retry times.", (Throwable)e);
                    throw e;
                }
                this.getLogger().debug("Attempt {} failed to poll for event batch. Retrying.", new Object[]{i});
                continue;
            }
            break;
        }
    }

    private void updateClusterStateForTxId(ProcessSession session) {
        try {
            HashMap<String, String> newState = new HashMap<String, String>(session.getState(Scope.CLUSTER).toMap());
            newState.put(LAST_TX_ID, String.valueOf(this.lastTxId));
            session.setState(newState, Scope.CLUSTER);
        }
        catch (IOException e) {
            this.getLogger().warn("Failed to update cluster state for last txId. It is possible data replication may occur.", (Throwable)e);
        }
    }

    protected HdfsAdmin getHdfsAdmin() {
        try {
            return new HdfsAdmin(this.getFileSystem().getUri(), this.getFileSystem().getConf());
        }
        catch (IOException e) {
            this.getLogger().error("Unable to get and instance of HDFS admin. You must be an HDFS super user to view HDFS events.");
            throw new ProcessException((Throwable)e);
        }
    }

    private boolean toProcessEvent(ProcessContext context, Event event) {
        String[] eventTypes;
        for (String name : eventTypes = context.getProperty(EVENT_TYPES).getValue().split(",")) {
            if (!name.trim().equalsIgnoreCase(event.getEventType().name())) continue;
            return this.notificationConfig.getPathFilter().accept(new Path(this.getPath(event)));
        }
        return false;
    }

    private String getPath(Event event) {
        if (event == null || event.getEventType() == null) {
            throw new IllegalArgumentException("Event and event type must not be null.");
        }
        return switch (event.getEventType()) {
            case Event.EventType.CREATE -> ((Event.CreateEvent)event).getPath();
            case Event.EventType.CLOSE -> ((Event.CloseEvent)event).getPath();
            case Event.EventType.APPEND -> ((Event.AppendEvent)event).getPath();
            case Event.EventType.RENAME -> ((Event.RenameEvent)event).getSrcPath();
            case Event.EventType.METADATA -> ((Event.MetadataUpdateEvent)event).getPath();
            case Event.EventType.UNLINK -> ((Event.UnlinkEvent)event).getPath();
            default -> throw new IllegalArgumentException("Unsupported event type.");
        };
    }

    private static class NotificationConfig {
        private final PathFilter pathFilter;

        NotificationConfig(ProcessContext context) {
            boolean toIgnoreHiddenFiles = context.getProperty(IGNORE_HIDDEN_FILES).asBoolean();
            Pattern watchDirectory = Pattern.compile(context.getProperty(HDFS_PATH_TO_WATCH).evaluateAttributeExpressions().getValue());
            this.pathFilter = new NotificationEventPathFilter(watchDirectory, toIgnoreHiddenFiles);
        }

        PathFilter getPathFilter() {
            return this.pathFilter;
        }
    }
}

