/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.gcp.drive;

import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.client.http.AbstractInputStreamContent;
import com.google.api.client.http.GenericUrl;
import com.google.api.client.http.HttpResponse;
import com.google.api.client.http.HttpTransport;
import com.google.api.client.http.InputStreamContent;
import com.google.api.client.util.DateTime;
import com.google.api.services.drive.Drive;
import com.google.api.services.drive.DriveRequest;
import com.google.api.services.drive.model.File;
import com.google.api.services.drive.model.FileList;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.nifi.annotation.behavior.InputRequirement;
import org.apache.nifi.annotation.behavior.ReadsAttribute;
import org.apache.nifi.annotation.behavior.WritesAttribute;
import org.apache.nifi.annotation.behavior.WritesAttributes;
import org.apache.nifi.annotation.documentation.CapabilityDescription;
import org.apache.nifi.annotation.documentation.SeeAlso;
import org.apache.nifi.annotation.documentation.Tags;
import org.apache.nifi.annotation.lifecycle.OnScheduled;
import org.apache.nifi.components.PropertyDescriptor;
import org.apache.nifi.components.ValidationContext;
import org.apache.nifi.components.ValidationResult;
import org.apache.nifi.components.Validator;
import org.apache.nifi.context.PropertyContext;
import org.apache.nifi.expression.ExpressionLanguageScope;
import org.apache.nifi.flowfile.FlowFile;
import org.apache.nifi.flowfile.attributes.CoreAttributes;
import org.apache.nifi.migration.PropertyConfiguration;
import org.apache.nifi.migration.ProxyServiceMigration;
import org.apache.nifi.processor.AbstractProcessor;
import org.apache.nifi.processor.DataUnit;
import org.apache.nifi.processor.ProcessContext;
import org.apache.nifi.processor.ProcessSession;
import org.apache.nifi.processor.Relationship;
import org.apache.nifi.processor.exception.ProcessException;
import org.apache.nifi.processor.util.StandardValidators;
import org.apache.nifi.processors.conflict.resolution.ConflictResolutionStrategy;
import org.apache.nifi.processors.gcp.ProxyAwareTransportFactory;
import org.apache.nifi.processors.gcp.drive.FetchGoogleDrive;
import org.apache.nifi.processors.gcp.drive.GoogleDriveAttributes;
import org.apache.nifi.processors.gcp.drive.GoogleDriveTrait;
import org.apache.nifi.processors.gcp.drive.ListGoogleDrive;
import org.apache.nifi.processors.gcp.util.GoogleUtils;
import org.apache.nifi.proxy.ProxyConfiguration;
import org.apache.nifi.proxy.ProxySpec;
import org.json.JSONObject;

@SeeAlso(value={ListGoogleDrive.class, FetchGoogleDrive.class})
@InputRequirement(value=InputRequirement.Requirement.INPUT_REQUIRED)
@Tags(value={"google", "drive", "storage", "put"})
@CapabilityDescription(value="Writes the contents of a FlowFile as a file in Google Drive.")
@ReadsAttribute(attribute="filename", description="Uses the FlowFile's filename as the filename for the Google Drive object.")
@WritesAttributes(value={@WritesAttribute(attribute="drive.id", description="The id of the file"), @WritesAttribute(attribute="filename", description="The name of the file"), @WritesAttribute(attribute="mime.type", description="The MIME type of the file"), @WritesAttribute(attribute="drive.size", description="The size of the file. Set to 0 when the file size is not available (e.g. externally stored files)."), @WritesAttribute(attribute="drive.size.available", description="Indicates if the file size is known / available"), @WritesAttribute(attribute="drive.timestamp", description="The last modified time or created time (whichever is greater) of the file. The reason for this is that the original modified date of a file is preserved when uploaded to Google Drive. 'Created time' takes the time when the upload occurs. However uploaded files can still be modified later."), @WritesAttribute(attribute="drive.created.time", description="The file's creation time"), @WritesAttribute(attribute="drive.modified.time", description="The file's last modification time"), @WritesAttribute(attribute="error.code", description="The error code returned by Google Drive"), @WritesAttribute(attribute="error.message", description="The error message returned by Google Drive")})
public class PutGoogleDrive
extends AbstractProcessor
implements GoogleDriveTrait {
    public static final int MIN_ALLOWED_CHUNK_SIZE_IN_BYTES = 262144;
    public static final int MAX_ALLOWED_CHUNK_SIZE_IN_BYTES = 0x40000000;
    public static final PropertyDescriptor FOLDER_ID = new PropertyDescriptor.Builder().name("Folder ID").description("The ID of the shared folder. Please see Additional Details to set up access to Google Drive and obtain Folder ID.").addValidator(StandardValidators.NON_EMPTY_VALIDATOR).expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).required(true).build();
    public static final PropertyDescriptor FILE_NAME = new PropertyDescriptor.Builder().name("Filename").description("The name of the file to upload to the specified Google Drive folder.").required(true).defaultValue("${filename}").expressionLanguageSupported(ExpressionLanguageScope.FLOWFILE_ATTRIBUTES).addValidator(StandardValidators.NON_EMPTY_VALIDATOR).build();
    public static final PropertyDescriptor CONFLICT_RESOLUTION = new PropertyDescriptor.Builder().name("Conflict Resolution Strategy").description("Indicates what should happen when a file with the same name already exists in the specified Google Drive folder.").required(true).defaultValue(ConflictResolutionStrategy.FAIL.getValue()).allowableValues(ConflictResolutionStrategy.class).build();
    public static final PropertyDescriptor CHUNKED_UPLOAD_SIZE = new PropertyDescriptor.Builder().name("Chunked Upload Size").description("Defines the size of a chunk. Used when a FlowFile's size exceeds 'Chunked Upload Threshold' and content is uploaded in smaller chunks. Minimum allowed chunk size is 256 KB, maximum allowed chunk size is 1 GB.").addValidator(PutGoogleDrive.createChunkSizeValidator()).defaultValue("10 MB").required(false).build();
    public static final PropertyDescriptor CHUNKED_UPLOAD_THRESHOLD = new PropertyDescriptor.Builder().name("Chunked Upload Threshold").description("The maximum size of the content which is uploaded at once. FlowFiles larger than this threshold are uploaded in chunks.").defaultValue("100 MB").addValidator(StandardValidators.DATA_SIZE_VALIDATOR).required(false).build();
    public static final List<PropertyDescriptor> PROPERTY_DESCRIPTORS = List.of(GoogleUtils.GCP_CREDENTIALS_PROVIDER_SERVICE, FOLDER_ID, FILE_NAME, CONFLICT_RESOLUTION, CHUNKED_UPLOAD_THRESHOLD, CHUNKED_UPLOAD_SIZE, ProxyConfiguration.createProxyConfigPropertyDescriptor((ProxySpec[])ProxyAwareTransportFactory.PROXY_SPECS), CONNECT_TIMEOUT, READ_TIMEOUT);
    public static final Relationship REL_SUCCESS = new Relationship.Builder().name("success").description("Files that have been successfully written to Google Drive are transferred to this relationship.").build();
    public static final Relationship REL_FAILURE = new Relationship.Builder().name("failure").description("Files that could not be written to Google Drive for some reason are transferred to this relationship.").build();
    public static final Set<Relationship> RELATIONSHIPS = Set.of(REL_SUCCESS, REL_FAILURE);
    public static final String MULTIPART_UPLOAD_URL = "https://www.googleapis.com/upload/drive/v3/files?uploadType=multipart&supportsAllDrives=true";
    private volatile Drive driveService;

    public Set<Relationship> getRelationships() {
        return RELATIONSHIPS;
    }

    public List<PropertyDescriptor> getSupportedPropertyDescriptors() {
        return PROPERTY_DESCRIPTORS;
    }

    public List<ValidationResult> customValidate(ValidationContext validationContext) {
        ArrayList<ValidationResult> results = new ArrayList<ValidationResult>(super.customValidate(validationContext));
        long chunkUploadThreshold = validationContext.getProperty(CHUNKED_UPLOAD_THRESHOLD).asDataSize(DataUnit.B).longValue();
        int uploadChunkSize = validationContext.getProperty(CHUNKED_UPLOAD_SIZE).asDataSize(DataUnit.B).intValue();
        if ((long)uploadChunkSize > chunkUploadThreshold) {
            results.add(new ValidationResult.Builder().subject(CHUNKED_UPLOAD_SIZE.getDisplayName()).explanation(String.format("%s should not be bigger than %s", CHUNKED_UPLOAD_SIZE.getDisplayName(), CHUNKED_UPLOAD_THRESHOLD.getDisplayName())).valid(false).build());
        }
        return results;
    }

    public void onTrigger(ProcessContext context, ProcessSession session) throws ProcessException {
        FlowFile flowFile = session.get();
        if (flowFile == null) {
            return;
        }
        String folderId = context.getProperty(FOLDER_ID).evaluateAttributeExpressions(flowFile).getValue();
        String filename = context.getProperty(FILE_NAME).evaluateAttributeExpressions(flowFile).getValue();
        String mimeType = flowFile.getAttribute(CoreAttributes.MIME_TYPE.key());
        try {
            File uploadedFile;
            long startNanos = System.nanoTime();
            long size = flowFile.getSize();
            long chunkUploadThreshold = context.getProperty(CHUNKED_UPLOAD_THRESHOLD).asDataSize(DataUnit.B).longValue();
            int uploadChunkSize = context.getProperty(CHUNKED_UPLOAD_SIZE).asDataSize(DataUnit.B).intValue();
            ConflictResolutionStrategy conflictResolution = ConflictResolutionStrategy.forValue((String)context.getProperty(CONFLICT_RESOLUTION).getValue());
            Optional<File> alreadyExistingFile = this.checkFileExistence(filename, folderId);
            File fileMetadata = alreadyExistingFile.orElseGet(() -> this.createMetadata(filename, folderId));
            if (alreadyExistingFile.isPresent() && conflictResolution == ConflictResolutionStrategy.FAIL) {
                this.getLogger().error("File [{}] already exists in [{}] Folder, conflict resolution is [{}]", new Object[]{filename, folderId, ConflictResolutionStrategy.FAIL.getDisplayName()});
                flowFile = this.addAttributes(alreadyExistingFile.get(), flowFile, session);
                session.transfer(flowFile, REL_FAILURE);
                return;
            }
            if (alreadyExistingFile.isPresent() && conflictResolution == ConflictResolutionStrategy.IGNORE) {
                this.getLogger().info("File [{}] already exists in [{}] Folder, conflict resolution is [{}]", new Object[]{filename, folderId, ConflictResolutionStrategy.IGNORE.getDisplayName()});
                flowFile = this.addAttributes(alreadyExistingFile.get(), flowFile, session);
                session.transfer(flowFile, REL_SUCCESS);
                return;
            }
            try (InputStream rawIn = session.read(flowFile);
                 BufferedInputStream bufferedInputStream = new BufferedInputStream(rawIn);){
                InputStreamContent mediaContent = new InputStreamContent(mimeType, (InputStream)bufferedInputStream);
                mediaContent.setLength(size);
                DriveRequest<File> driveRequest = this.createDriveRequest(fileMetadata, mediaContent);
                uploadedFile = size > chunkUploadThreshold ? this.uploadFileInChunks(driveRequest, fileMetadata, uploadChunkSize, mediaContent) : (File)driveRequest.execute();
            }
            if (uploadedFile != null) {
                Map<String, String> attributes = this.createGoogleDriveFileInfoBuilder(uploadedFile).build().toAttributeMap();
                String url = "https://drive.google.com/open?id=" + uploadedFile.getId();
                flowFile = session.putAllAttributes(flowFile, attributes);
                long transferMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                session.getProvenanceReporter().send(flowFile, url, transferMillis);
            }
            session.transfer(flowFile, REL_SUCCESS);
        }
        catch (GoogleJsonResponseException e) {
            this.getLogger().error("Exception occurred while uploading File [{}] to [{}] Google Drive Folder", new Object[]{filename, folderId, e});
            this.handleExpectedError(session, flowFile, e);
        }
        catch (Exception e) {
            this.getLogger().error("Exception occurred while uploading File [{}] to [{}] Google Drive Folder", new Object[]{filename, folderId, e});
            if (e.getCause() != null && e.getCause() instanceof GoogleJsonResponseException) {
                this.handleExpectedError(session, flowFile, (GoogleJsonResponseException)e.getCause());
            }
            this.handleUnexpectedError(session, flowFile, e);
        }
    }

    @OnScheduled
    public void onScheduled(ProcessContext context) throws IOException {
        ProxyConfiguration proxyConfiguration = ProxyConfiguration.getConfiguration((PropertyContext)context);
        HttpTransport httpTransport = new ProxyAwareTransportFactory(proxyConfiguration).create();
        this.driveService = this.createDriveService(context, httpTransport, new String[]{"https://www.googleapis.com/auth/drive", "https://www.googleapis.com/auth/drive.metadata"});
    }

    public void migrateProperties(PropertyConfiguration config) {
        config.renameProperty("connect-timeout", CONNECT_TIMEOUT.getName());
        config.renameProperty("read-timeout", READ_TIMEOUT.getName());
        config.renameProperty("folder-id", FOLDER_ID.getName());
        config.renameProperty("file-name", FILE_NAME.getName());
        config.renameProperty("conflict-resolution-strategy", CONFLICT_RESOLUTION.getName());
        config.renameProperty("chunked-upload-size", CHUNKED_UPLOAD_SIZE.getName());
        config.renameProperty("chunked-upload-threshold", CHUNKED_UPLOAD_THRESHOLD.getName());
        config.renameProperty("gcp-credentials-provider-service", GoogleUtils.GCP_CREDENTIALS_PROVIDER_SERVICE.getName());
        ProxyServiceMigration.renameProxyConfigurationServiceProperty((PropertyConfiguration)config);
    }

    private FlowFile addAttributes(File file, FlowFile flowFile, ProcessSession session) {
        HashMap<String, String> attributes = new HashMap<String, String>();
        attributes.put("drive.id", file.getId());
        attributes.put(GoogleDriveAttributes.FILENAME, file.getName());
        return session.putAllAttributes(flowFile, attributes);
    }

    private DriveRequest<File> createDriveRequest(File fileMetadata, InputStreamContent mediaContent) throws IOException {
        if (fileMetadata.getId() == null) {
            return this.driveService.files().create(fileMetadata, (AbstractInputStreamContent)mediaContent).setSupportsAllDrives(Boolean.valueOf(true)).setFields("id, name, createdTime, modifiedTime, mimeType, size");
        }
        return this.driveService.files().update(fileMetadata.getId(), new File(), (AbstractInputStreamContent)mediaContent).setSupportsAllDrives(Boolean.valueOf(true)).setFields("id, name, createdTime, modifiedTime, mimeType, size");
    }

    private File uploadFileInChunks(DriveRequest<File> driveRequest, File fileMetadata, int chunkSize, InputStreamContent mediaContent) throws IOException {
        HttpResponse response = driveRequest.getMediaHttpUploader().setChunkSize(chunkSize).setDirectUploadEnabled(false).upload(new GenericUrl(MULTIPART_UPLOAD_URL));
        if (response.getStatusCode() == 200) {
            fileMetadata.setId(this.getUploadedFileId(response.getContent()));
            fileMetadata.setMimeType(mediaContent.getType());
            fileMetadata.setCreatedTime(new DateTime(System.currentTimeMillis()));
            fileMetadata.setSize(Long.valueOf(mediaContent.getLength()));
            return fileMetadata;
        }
        throw new ProcessException(String.format("Upload of File [%s] to Folder [%s] failed, HTTP error code: [%d]", fileMetadata.getName(), fileMetadata.getParents().stream().findFirst().orElse(""), response.getStatusCode()));
    }

    private String getUploadedFileId(InputStream content) {
        String contentAsString = new BufferedReader(new InputStreamReader(content, StandardCharsets.UTF_8)).lines().collect(Collectors.joining("\n"));
        return new JSONObject(contentAsString).getString("id");
    }

    private File createMetadata(String name, String parentId) {
        File metadata = new File();
        metadata.setName(name);
        metadata.setParents(Collections.singletonList(parentId));
        return metadata;
    }

    private Optional<File> checkFileExistence(String fileName, String parentId) throws IOException {
        FileList result = (FileList)this.driveService.files().list().setSupportsAllDrives(Boolean.valueOf(true)).setIncludeItemsFromAllDrives(Boolean.valueOf(true)).setQ(String.format("name='%s' and ('%s' in parents)", fileName, parentId)).setFields("files(name, id)").execute();
        return result.getFiles().stream().findFirst();
    }

    private void handleUnexpectedError(ProcessSession session, FlowFile flowFile, Exception e) {
        flowFile = session.putAttribute(flowFile, "error.message", e.getMessage());
        flowFile = session.penalize(flowFile);
        session.transfer(flowFile, REL_FAILURE);
    }

    private void handleExpectedError(ProcessSession session, FlowFile flowFile, GoogleJsonResponseException e) {
        flowFile = session.putAttribute(flowFile, "error.message", e.getMessage());
        flowFile = session.putAttribute(flowFile, "error.code", String.valueOf(e.getStatusCode()));
        flowFile = session.penalize(flowFile);
        session.transfer(flowFile, REL_FAILURE);
    }

    private static Validator createChunkSizeValidator() {
        return (subject, input, context) -> {
            ValidationResult vr = StandardValidators.createDataSizeBoundsValidator((long)262144L, (long)0x40000000L).validate(subject, input, context);
            if (!vr.isValid()) {
                return vr;
            }
            long dataSizeBytes = DataUnit.parseDataSize((String)input, (DataUnit)DataUnit.B).longValue();
            if (dataSizeBytes % 262144L != 0L) {
                return new ValidationResult.Builder().subject(subject).input(input).valid(false).explanation("Must be a positive multiple of 262144 bytes").build();
            }
            return new ValidationResult.Builder().subject(subject).input(input).valid(true).build();
        };
    }
}

