/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.cluster.coordination.http.replication.client;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.AnnotationIntrospector;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.jakarta.xmlbind.JakartaXmlBindAnnotationIntrospector;
import jakarta.ws.rs.core.MultivaluedHashMap;
import jakarta.ws.rs.core.MultivaluedMap;
import jakarta.ws.rs.core.Response;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.Set;
import java.util.function.Supplier;
import java.util.zip.GZIPInputStream;
import org.apache.nifi.cluster.coordination.http.replication.HttpReplicationClient;
import org.apache.nifi.cluster.coordination.http.replication.PreparedRequest;
import org.apache.nifi.cluster.coordination.http.replication.client.PreparedRequestHeader;
import org.apache.nifi.cluster.coordination.http.replication.client.StandardPreparedRequest;
import org.apache.nifi.cluster.coordination.http.replication.io.EntitySerializer;
import org.apache.nifi.cluster.coordination.http.replication.io.JsonEntitySerializer;
import org.apache.nifi.cluster.coordination.http.replication.io.ReplicatedResponse;
import org.apache.nifi.cluster.coordination.http.replication.io.XmlEntitySerializer;
import org.apache.nifi.web.client.api.HttpEntityHeaders;
import org.apache.nifi.web.client.api.HttpRequestBodySpec;
import org.apache.nifi.web.client.api.HttpRequestMethod;
import org.apache.nifi.web.client.api.HttpResponseEntity;
import org.apache.nifi.web.client.api.HttpUriBuilder;
import org.apache.nifi.web.client.api.WebClientService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StandardHttpReplicationClient
implements HttpReplicationClient {
    private static final Set<String> REQUEST_BODY_METHODS = Set.of("PATCH", "POST", "PUT");
    private static final Set<String> DISALLOWED_HEADERS = Set.of("connection", "content-length", "expect", "host", "upgrade");
    private static final int CONTENT_LENGTH_NOT_FOUND = -1;
    private static final char PSEUDO_HEADER_PREFIX = ':';
    private static final String GZIP_ENCODING = "gzip";
    private static final String QUERY_SEPARATOR = "&";
    private static final String QUERY_NAME_VALUE_SEPARATOR = "=";
    private static final String APPLICATION_JSON_CONTENT_TYPE = "application/json";
    private static final String APPLICATION_XML_CONTENT_TYPE = "application/xml";
    private static final String USER_AGENT_PRODUCT = "Apache NiFi";
    private static final String USER_AGENT_FORMAT = "%s/%s";
    private static final String USER_AGENT_VERSION = "SNAPSHOT";
    private static final String USER_AGENT;
    private static final Logger logger;
    private final WebClientService webClientService;
    private final Supplier<HttpUriBuilder> httpUriBuilderSupplier;
    private final EntitySerializer jsonSerializer;
    private final EntitySerializer xmlSerializer;
    private final ObjectMapper objectMapper = new ObjectMapper();

    public StandardHttpReplicationClient(WebClientService webClientService, Supplier<HttpUriBuilder> httpUriBuilderSupplier) {
        this.webClientService = Objects.requireNonNull(webClientService, "Web Client Service required");
        this.httpUriBuilderSupplier = Objects.requireNonNull(httpUriBuilderSupplier, "HTTP URI Builder supplier required");
        this.objectMapper.setDefaultPropertyInclusion(JsonInclude.Value.construct((JsonInclude.Include)JsonInclude.Include.NON_NULL, (JsonInclude.Include)JsonInclude.Include.ALWAYS));
        this.objectMapper.setAnnotationIntrospector((AnnotationIntrospector)new JakartaXmlBindAnnotationIntrospector(this.objectMapper.getTypeFactory()));
        this.jsonSerializer = new JsonEntitySerializer(this.objectMapper);
        this.xmlSerializer = new XmlEntitySerializer();
    }

    @Override
    public PreparedRequest prepareRequest(String method, Map<String, String> headers, Object requestEntity) {
        Map<String, String> preparedHeaders = this.getPreparedHeaders(headers, method);
        byte[] requestBody = this.getRequestBody(requestEntity, preparedHeaders);
        return new StandardPreparedRequest(method, preparedHeaders, requestEntity, requestBody);
    }

    @Override
    public Response replicate(PreparedRequest request, URI uri) throws IOException {
        if (request instanceof StandardPreparedRequest) {
            StandardPreparedRequest preparedRequest = (StandardPreparedRequest)request;
            return this.replicate(preparedRequest, uri);
        }
        throw new IllegalArgumentException("HTTP Prepared Request not provided");
    }

    private Map<String, String> getPreparedHeaders(Map<String, String> headers, String method) {
        LinkedHashMap<String, String> preparedHeaders = new LinkedHashMap<String, String>();
        for (Map.Entry<String, String> header : headers.entrySet()) {
            String headerName = header.getKey().toLowerCase();
            if (PreparedRequestHeader.ACCEPT_ENCODING.getHeader().equals(headerName)) continue;
            String headerValue = header.getValue();
            preparedHeaders.put(headerName, headerValue);
        }
        preparedHeaders.put(PreparedRequestHeader.ACCEPT_ENCODING.getHeader(), GZIP_ENCODING);
        this.processContentType(method, preparedHeaders);
        this.processUserAgent(preparedHeaders);
        return preparedHeaders;
    }

    private Response replicate(StandardPreparedRequest preparedRequest, URI location) throws IOException {
        HttpRequestMethod requestMethod = this.getRequestMethod(preparedRequest);
        URI requestUri = this.getRequestUri(preparedRequest, location);
        HttpRequestBodySpec httpRequestBodySpec = this.webClientService.method(requestMethod).uri(requestUri);
        Map<String, String> requestHeaders = preparedRequest.headers();
        for (Map.Entry<String, String> requestHeader : requestHeaders.entrySet()) {
            String headerName = requestHeader.getKey();
            String headerNameLowerCased = headerName.toLowerCase();
            if (DISALLOWED_HEADERS.contains(headerNameLowerCased)) continue;
            httpRequestBodySpec.header(headerName, requestHeader.getValue());
        }
        if (REQUEST_BODY_METHODS.contains(requestMethod.getMethod())) {
            byte[] requestBody = preparedRequest.requestBody();
            ByteArrayInputStream body = new ByteArrayInputStream(requestBody);
            OptionalLong contentLength = OptionalLong.of(requestBody.length);
            httpRequestBodySpec.body((InputStream)body, contentLength);
        }
        return this.replicate(httpRequestBodySpec, preparedRequest.method(), location);
    }

    private Response replicate(HttpRequestBodySpec httpRequestBodySpec, String method, URI location) throws IOException {
        long started = System.currentTimeMillis();
        HttpResponseEntity responseEntity = httpRequestBodySpec.retrieve();
        int statusCode = responseEntity.statusCode();
        HttpEntityHeaders headers = responseEntity.headers();
        MultivaluedMap<String, String> responseHeaders = this.getResponseHeaders(headers);
        int contentLength = this.getContentLength(headers);
        InputStream responseBody = this.getResponseBody(responseEntity.body(), headers);
        Runnable closeCallback = () -> {
            try {
                responseEntity.close();
            }
            catch (IOException e) {
                logger.warn("Close failed for Replicated {} {} HTTP {}", new Object[]{method, location, statusCode, e});
            }
        };
        long elapsed = System.currentTimeMillis() - started;
        logger.debug("Replicated {} {} HTTP {} in {} ms", new Object[]{method, location, statusCode, elapsed});
        return new ReplicatedResponse(this.objectMapper, responseBody, responseHeaders, location, statusCode, contentLength, closeCallback);
    }

    private URI getRequestUri(StandardPreparedRequest preparedRequest, URI location) {
        Object requestEntity;
        HttpUriBuilder httpUriBuilder = this.httpUriBuilderSupplier.get();
        httpUriBuilder.scheme(location.getScheme());
        httpUriBuilder.host(location.getHost());
        httpUriBuilder.port(location.getPort());
        httpUriBuilder.encodedPath(location.getPath());
        String query = location.getQuery();
        if (query != null) {
            String[] parameters;
            for (String parameter : parameters = query.split(QUERY_SEPARATOR)) {
                String parameterName;
                String[] parameterNameValue = parameter.split(QUERY_NAME_VALUE_SEPARATOR);
                if (parameterNameValue.length == 1) {
                    parameterName = parameterNameValue[0];
                    httpUriBuilder.addQueryParameter(parameterName, null);
                    continue;
                }
                if (parameterNameValue.length != 2) continue;
                parameterName = parameterNameValue[0];
                String parameterValue = parameterNameValue[1];
                httpUriBuilder.addQueryParameter(parameterName, parameterValue);
            }
        }
        if ((requestEntity = preparedRequest.entity()) instanceof MultivaluedMap) {
            MultivaluedMap parameterEntity = (MultivaluedMap)requestEntity;
            for (Object key : parameterEntity.keySet()) {
                String parameterName = key.toString();
                Object parameterValues = parameterEntity.get((Object)parameterName);
                if (!(parameterValues instanceof List)) continue;
                List values = (List)parameterValues;
                for (Object value : values) {
                    httpUriBuilder.addQueryParameter(parameterName, value.toString());
                }
            }
        }
        return httpUriBuilder.build();
    }

    private HttpRequestMethod getRequestMethod(PreparedRequest preparedRequest) {
        final String method = preparedRequest.getMethod();
        return new HttpRequestMethod(){

            public String getMethod() {
                return method;
            }

            public String toString() {
                return method;
            }
        };
    }

    private MultivaluedMap<String, String> getResponseHeaders(HttpEntityHeaders responseHeaders) {
        MultivaluedHashMap headers = new MultivaluedHashMap();
        for (String name : responseHeaders.getHeaderNames()) {
            if (name.charAt(0) == ':' || PreparedRequestHeader.CONTENT_ENCODING.getHeader().equalsIgnoreCase(name) || PreparedRequestHeader.CONTENT_LENGTH.getHeader().equalsIgnoreCase(name)) continue;
            List values = responseHeaders.getHeader(name);
            headers.addAll((Object)name, values);
        }
        return headers;
    }

    private InputStream getResponseBody(InputStream inputStream, HttpEntityHeaders responseHeaders) throws IOException {
        boolean gzipEncoded = this.isGzipEncoded(responseHeaders);
        return gzipEncoded ? new GZIPInputStream(inputStream) : inputStream;
    }

    private int getContentLength(HttpEntityHeaders headers) {
        int contentLength;
        Optional contentLengthFound = headers.getHeaderNames().stream().filter(PreparedRequestHeader.CONTENT_LENGTH.getHeader()::equalsIgnoreCase).findFirst().flatMap(arg_0 -> ((HttpEntityHeaders)headers).getFirstHeader(arg_0));
        if (contentLengthFound.isPresent()) {
            String contentLengthHeader = (String)contentLengthFound.get();
            try {
                contentLength = Integer.parseInt(contentLengthHeader);
            }
            catch (NumberFormatException e) {
                logger.warn("Replicated Header Content-Length [{}] parsing failed", (Object)contentLengthHeader, (Object)e);
                contentLength = -1;
            }
        } else {
            contentLength = -1;
        }
        return contentLength;
    }

    private byte[] getRequestBody(Object requestEntity, Map<String, String> headers) {
        Optional<String> contentTypeFound = this.getContentType(headers);
        String contentType = contentTypeFound.orElse(APPLICATION_JSON_CONTENT_TYPE);
        EntitySerializer serializer = this.getSerializer(contentType);
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            serializer.serialize(requestEntity, outputStream);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Request Entity serialization failed", e);
        }
        return outputStream.toByteArray();
    }

    private void processContentType(String method, Map<String, String> headers) {
        Optional<String> contentTypeHeaderFound;
        if (REQUEST_BODY_METHODS.contains(method) && (contentTypeHeaderFound = this.getHeaderName(headers, PreparedRequestHeader.CONTENT_TYPE)).isEmpty()) {
            headers.put(PreparedRequestHeader.CONTENT_TYPE.getHeader(), APPLICATION_JSON_CONTENT_TYPE);
        }
    }

    private void processUserAgent(Map<String, String> headers) {
        Optional<String> userAgentHeaderFound = this.getHeaderName(headers, PreparedRequestHeader.USER_AGENT);
        String userAgentHeader = userAgentHeaderFound.orElseGet(PreparedRequestHeader.USER_AGENT::getHeader);
        headers.put(userAgentHeader, USER_AGENT);
    }

    private EntitySerializer getSerializer(String contentType) {
        EntitySerializer serializer = APPLICATION_XML_CONTENT_TYPE.equalsIgnoreCase(contentType) ? this.xmlSerializer : this.jsonSerializer;
        return serializer;
    }

    private boolean isGzipEncoded(HttpEntityHeaders headers) {
        Optional<String> contentEncodingFound = headers.getHeaderNames().stream().filter(PreparedRequestHeader.CONTENT_ENCODING.getHeader()::equalsIgnoreCase).map(arg_0 -> ((HttpEntityHeaders)headers).getFirstHeader(arg_0)).filter(Optional::isPresent).map(Optional::get).findFirst();
        return contentEncodingFound.map(GZIP_ENCODING::equalsIgnoreCase).orElse(false);
    }

    private Optional<String> getContentType(Map<String, String> headers) {
        String header;
        Optional<String> headerNameFound = this.getHeaderName(headers, PreparedRequestHeader.CONTENT_TYPE);
        if (headerNameFound.isPresent()) {
            String name = headerNameFound.get();
            header = headers.get(name);
        } else {
            header = null;
        }
        return Optional.ofNullable(header);
    }

    private Optional<String> getHeaderName(Map<String, String> headers, PreparedRequestHeader httpHeader) {
        return headers.keySet().stream().filter(httpHeader.getHeader()::equalsIgnoreCase).findFirst();
    }

    static {
        logger = LoggerFactory.getLogger(StandardHttpReplicationClient.class);
        Package clientPackage = StandardHttpReplicationClient.class.getPackage();
        String userAgentVersion = clientPackage == null || clientPackage.getImplementationVersion() == null ? USER_AGENT_VERSION : clientPackage.getImplementationVersion();
        USER_AGENT = USER_AGENT_FORMAT.formatted(USER_AGENT_PRODUCT, userAgentVersion);
    }
}

