/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.aether.internal.test.util.http;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.ServerSocket;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.eclipse.aether.DefaultRepositoryCache;
import org.eclipse.aether.DefaultRepositorySystemSession;
import org.eclipse.aether.DefaultSessionData;
import org.eclipse.aether.RepositoryCache;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.SessionData;
import org.eclipse.aether.internal.impl.transport.http.DefaultChecksumExtractor;
import org.eclipse.aether.internal.impl.transport.http.Nx2ChecksumExtractor;
import org.eclipse.aether.internal.impl.transport.http.XChecksumExtractor;
import org.eclipse.aether.internal.test.util.TestFileUtils;
import org.eclipse.aether.internal.test.util.TestLocalRepositoryManager;
import org.eclipse.aether.internal.test.util.http.HttpServer;
import org.eclipse.aether.internal.test.util.http.RecordingTransportListener;
import org.eclipse.aether.repository.Authentication;
import org.eclipse.aether.repository.LocalRepositoryManager;
import org.eclipse.aether.repository.Proxy;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.spi.connector.transport.GetTask;
import org.eclipse.aether.spi.connector.transport.PeekTask;
import org.eclipse.aether.spi.connector.transport.PutTask;
import org.eclipse.aether.spi.connector.transport.TransportListener;
import org.eclipse.aether.spi.connector.transport.http.ChecksumExtractor;
import org.eclipse.aether.spi.connector.transport.http.HttpTransporter;
import org.eclipse.aether.spi.connector.transport.http.HttpTransporterException;
import org.eclipse.aether.spi.connector.transport.http.HttpTransporterFactory;
import org.eclipse.aether.spi.connector.transport.http.RFC9457.HttpRFC9457Exception;
import org.eclipse.aether.transfer.NoTransporterException;
import org.eclipse.aether.transfer.TransferCancelledException;
import org.eclipse.aether.util.repository.AuthenticationBuilder;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestInfo;
import org.junit.jupiter.api.Timeout;

public class HttpTransporterTest {
    protected static final Path KEY_STORE_PATH = Paths.get("target/keystore", new String[0]);
    protected static final Path KEY_STORE_SELF_SIGNED_PATH = Paths.get("target/keystore-self-signed", new String[0]);
    protected static final Path TRUST_STORE_PATH = Paths.get("target/trustStore", new String[0]);
    private final Supplier<HttpTransporterFactory> transporterFactorySupplier;
    protected DefaultRepositorySystemSession session;
    protected HttpTransporterFactory factory;
    protected HttpTransporter transporter;
    protected Runnable closer;
    protected File repoDir;
    protected HttpServer httpServer;
    protected Authentication auth;
    protected Proxy proxy;
    protected static final long OLD_FILE_TIMESTAMP = 160660800000L;

    protected HttpTransporterTest(Supplier<HttpTransporterFactory> transporterFactorySupplier) {
        this.transporterFactorySupplier = Objects.requireNonNull(transporterFactorySupplier);
        if (!Files.isRegularFile(KEY_STORE_PATH, new LinkOption[0])) {
            URL keyStoreUrl = HttpTransporterTest.class.getClassLoader().getResource("ssl/server-store");
            URL keyStoreSelfSignedUrl = HttpTransporterTest.class.getClassLoader().getResource("ssl/server-store-selfsigned");
            URL trustStoreUrl = HttpTransporterTest.class.getClassLoader().getResource("ssl/client-store");
            try (InputStream keyStoreStream = keyStoreUrl.openStream();
                 InputStream keyStoreSelfSignedStream = keyStoreSelfSignedUrl.openStream();
                 InputStream trustStoreStream = trustStoreUrl.openStream();){
                Files.copy(keyStoreStream, KEY_STORE_PATH, StandardCopyOption.REPLACE_EXISTING);
                Files.copy(keyStoreSelfSignedStream, KEY_STORE_SELF_SIGNED_PATH, StandardCopyOption.REPLACE_EXISTING);
                Files.copy(trustStoreStream, TRUST_STORE_PATH, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
    }

    protected static ChecksumExtractor standardChecksumExtractor() {
        HashMap<String, Object> strategies = new HashMap<String, Object>();
        strategies.put("1", new Nx2ChecksumExtractor());
        strategies.put("2", new XChecksumExtractor());
        return new DefaultChecksumExtractor(strategies);
    }

    protected RemoteRepository newRepo(String url) {
        return new RemoteRepository.Builder("test", "default", url).setAuthentication(this.auth).setProxy(this.proxy).build();
    }

    protected void newTransporter(String url) throws Exception {
        if (this.transporter != null) {
            this.transporter.close();
            this.transporter = null;
        }
        if (this.closer != null) {
            this.closer.run();
            this.closer = null;
        }
        this.session = new DefaultRepositorySystemSession((RepositorySystemSession)this.session);
        this.session.setData((SessionData)new DefaultSessionData());
        this.transporter = this.factory.newInstance((RepositorySystemSession)this.session, this.newRepo(url));
    }

    @BeforeEach
    protected void setUp(TestInfo testInfo) throws Exception {
        System.out.println("=== " + testInfo.getDisplayName() + " ===");
        this.session = new DefaultRepositorySystemSession(h -> {
            this.closer = h;
            return true;
        });
        this.session.setLocalRepositoryManager((LocalRepositoryManager)new TestLocalRepositoryManager());
        this.factory = this.transporterFactorySupplier.get();
        this.repoDir = TestFileUtils.createTempDir();
        TestFileUtils.writeString((File)new File(this.repoDir, "file.txt"), (String)"test");
        TestFileUtils.writeString((File)new File(this.repoDir, "dir/file.txt"), (String)"test");
        TestFileUtils.writeString((File)new File(this.repoDir, "dir/oldFile.txt"), (String)"oldTest", (long)160660800000L);
        TestFileUtils.writeString((File)new File(this.repoDir, "empty.txt"), (String)"");
        TestFileUtils.writeString((File)new File(this.repoDir, "some space.txt"), (String)"space");
        File resumable = new File(this.repoDir, "resume.txt");
        TestFileUtils.writeString((File)resumable, (String)"resumable");
        resumable.setLastModified(System.currentTimeMillis() - 90000L);
        this.httpServer = new HttpServer().setRepoDir(this.repoDir).start();
        this.newTransporter(this.httpServer.getHttpUrl());
    }

    @AfterEach
    protected void tearDown() throws Exception {
        if (this.transporter != null) {
            this.transporter.close();
            this.transporter = null;
        }
        if (this.closer != null) {
            this.closer.run();
            this.closer = null;
        }
        if (this.httpServer != null) {
            this.httpServer.stop();
            this.httpServer = null;
        }
        this.factory = null;
        this.session = null;
    }

    @Test
    protected void testClassify() {
        Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)new FileNotFoundException()));
        Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)new HttpTransporterException(403)));
        Assertions.assertEquals((int)1, (int)this.transporter.classify((Throwable)new HttpTransporterException(404)));
    }

    @Test
    protected void testPeek() throws Exception {
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testRetryHandler_defaultCount_positive() throws Exception {
        this.httpServer.setConnectionsToClose(3);
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testRetryHandler_defaultCount_negative() throws Exception {
        this.httpServer.setConnectionsToClose(4);
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    protected void testRetryHandler_explicitCount_positive() throws Exception {
        this.session.setConfigProperty("aether.transport.http.retryHandler.count", (Object)10);
        this.newTransporter(this.httpServer.getHttpUrl());
        this.httpServer.setConnectionsToClose(10);
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testRetryHandler_disabled() throws Exception {
        this.session.setConfigProperty("aether.transport.http.retryHandler.count", (Object)0);
        this.newTransporter(this.httpServer.getHttpUrl());
        this.httpServer.setConnectionsToClose(1);
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @Test
    protected void testPeek_NotFound() throws Exception {
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/missing.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)404, (int)e.getStatusCode());
            Assertions.assertEquals((int)1, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testPeek_Closed() throws Exception {
        this.transporter.close();
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/missing.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (IllegalStateException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testPeek_Authenticated() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testPeek_Unauthenticated() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)401, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testPeek_ProxyAuthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort(), this.auth);
        this.newTransporter("http://bad.localhost:1/");
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testPeek_ProxyUnauthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort());
        this.newTransporter("http://bad.localhost:1/");
        try {
            this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)407, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testPeek_SSL() throws Exception {
        this.httpServer.addSslConnector();
        this.newTransporter(this.httpServer.getHttpsUrl());
        this.transporter.peek(new PeekTask(URI.create("repo/file.txt")));
    }

    @Test
    protected void testPeek_Redirect() throws Exception {
        this.httpServer.addSslConnector();
        this.transporter.peek(new PeekTask(URI.create("redirect/file.txt")));
        this.transporter.peek(new PeekTask(URI.create("redirect/file.txt?scheme=https")));
    }

    @Test
    protected void testGet_ToMemory() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_ToFile() throws Exception {
        File file = TestFileUtils.createTempFile((String)"failure");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setDataPath(file.toPath()).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"test", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_ToFileTimestamp() throws Exception {
        File file = TestFileUtils.createTempFile((String)"failure");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/dir/oldFile.txt")).setDataPath(file.toPath()).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"oldTest", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)7L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"oldTest", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
        Assertions.assertEquals((long)160660800000L, (long)file.lastModified());
    }

    @Test
    protected void testGet_EmptyResource() throws Exception {
        File file = TestFileUtils.createTempFile((String)"failure");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/empty.txt")).setDataPath(file.toPath()).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)0L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
        Assertions.assertEquals((Object)"", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_EncodedResourcePath() throws Exception {
        GetTask task = new GetTask(URI.create("repo/some%20space.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"space", (Object)task.getDataString());
    }

    @Test
    protected void testGet_Authenticated() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_Unauthenticated() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)401, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testGet_ProxyAuthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        Authentication auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort(), auth);
        this.newTransporter("http://bad.localhost:1/");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_ProxyUnauthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort());
        this.newTransporter("http://bad.localhost:1/");
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)407, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testGet_RFC9457Response() throws Exception {
        try {
            this.transporter.get(new GetTask(URI.create("rfc9457/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpRFC9457Exception e) {
            Assertions.assertEquals((int)403, (int)e.getStatusCode());
            Assertions.assertEquals((Object)e.getPayload().getType(), (Object)URI.create("https://example.com/probs/out-of-credit"));
            Assertions.assertEquals((int)403, (Integer)e.getPayload().getStatus());
            Assertions.assertEquals((Object)"You do not have enough credit.", (Object)e.getPayload().getTitle());
            Assertions.assertEquals((Object)"Your current balance is 30, but that costs 50.", (Object)e.getPayload().getDetail());
            Assertions.assertEquals((Object)URI.create("/account/12345/msgs/abc"), (Object)e.getPayload().getInstance());
        }
    }

    @Test
    protected void testGet_RFC9457Response_with_missing_fields() throws Exception {
        try {
            this.transporter.get(new GetTask(URI.create("rfc9457/missing_fields.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpRFC9457Exception e) {
            Assertions.assertEquals((int)403, (int)e.getStatusCode());
            Assertions.assertEquals((Object)e.getPayload().getType(), (Object)URI.create("about:blank"));
            Assertions.assertNull((Object)e.getPayload().getStatus());
            Assertions.assertNull((Object)e.getPayload().getTitle());
            Assertions.assertNull((Object)e.getPayload().getDetail());
            Assertions.assertNull((Object)e.getPayload().getInstance());
        }
    }

    @Test
    protected void testGet_SSL() throws Exception {
        this.httpServer.addSslConnector();
        this.newTransporter(this.httpServer.getHttpsUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_SSL_WithServerErrors() throws Exception {
        this.httpServer.setServerErrorsBeforeWorks(1);
        this.httpServer.addSslConnector();
        this.newTransporter(this.httpServer.getHttpsUrl());
        for (int i = 1; i < 3; ++i) {
            try {
                RecordingTransportListener listener = new RecordingTransportListener();
                GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
                this.transporter.get(task);
                Assertions.assertEquals((Object)"test", (Object)task.getDataString());
                Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
                Assertions.assertEquals((long)4L, (long)listener.getDataLength());
                Assertions.assertEquals((int)1, (int)listener.getStartedCount());
                Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
                Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
                continue;
            }
            catch (HttpTransporterException e) {
                Assertions.assertEquals((int)500, (int)e.getStatusCode());
            }
        }
    }

    @Test
    protected void testGet_HTTPS_Unknown_SecurityMode() throws Exception {
        this.session.setConfigProperty("aether.transport.https.securityMode", (Object)"unknown");
        this.httpServer.addSelfSignedSslConnector();
        try {
            this.newTransporter(this.httpServer.getHttpsUrl());
            Assertions.fail((String)"Unsupported security mode");
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
    }

    @Test
    protected void testGet_HTTPS_Insecure_SecurityMode() throws Exception {
        this.session.setConfigProperty("aether.transport.https.securityMode", (Object)"insecure");
        this.httpServer.addSelfSignedSslConnector();
        this.newTransporter(this.httpServer.getHttpsUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_HTTPS_HTTP2Only_Insecure_SecurityMode() throws Exception {
        this.enableHttp2Protocol();
        this.session.setConfigProperty("aether.transport.https.securityMode", (Object)"insecure");
        this.httpServer.addSelfSignedSslConnectorHttp2Only();
        this.newTransporter(this.httpServer.getHttpsUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    protected void enableHttp2Protocol() {
    }

    @Test
    protected void testGet_Redirect() throws Exception {
        this.httpServer.addSslConnector();
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("redirect/file.txt?scheme=https")).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)task.getDataString(), (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_Resume() throws Exception {
        File file = TestFileUtils.createTempFile((String)"re");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/resume.txt")).setDataPath(file.toPath(), true).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"resumable", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)1L, (long)listener.getStartedCount());
        Assertions.assertEquals((long)2L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)9L, (long)listener.getDataLength());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"sumable", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_ResumeLocalContentsOutdated() throws Exception {
        File file = TestFileUtils.createTempFile((String)"re");
        file.setLastModified(System.currentTimeMillis() - 300000L);
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/resume.txt")).setDataPath(file.toPath(), true).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"resumable", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)1L, (long)listener.getStartedCount());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)9L, (long)listener.getDataLength());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"resumable", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_ResumeRangesNotSupportedByServer() throws Exception {
        this.httpServer.setRangeSupport(false);
        File file = TestFileUtils.createTempFile((String)"re");
        RecordingTransportListener listener = new RecordingTransportListener();
        GetTask task = new GetTask(URI.create("repo/resume.txt")).setDataPath(file.toPath(), true).setListener((TransportListener)listener);
        this.transporter.get(task);
        Assertions.assertEquals((Object)"resumable", (Object)TestFileUtils.readString((File)file));
        Assertions.assertEquals((long)1L, (long)listener.getStartedCount());
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)9L, (long)listener.getDataLength());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"resumable", (Object)listener.getBaos().toString(StandardCharsets.UTF_8));
    }

    @Test
    protected void testGet_Checksums_Nexus() throws Exception {
        this.httpServer.setChecksumHeader(HttpServer.ChecksumHeader.NEXUS);
        GetTask task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((Object)"a94a8fe5ccb19ba61c4c0873d391e987982fbbd3", task.getChecksums().get("SHA-1"));
    }

    @Test
    protected void testGet_Checksums_XChecksum() throws Exception {
        this.httpServer.setChecksumHeader(HttpServer.ChecksumHeader.XCHECKSUM);
        GetTask task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((Object)"a94a8fe5ccb19ba61c4c0873d391e987982fbbd3", task.getChecksums().get("SHA-1"));
    }

    @Test
    protected void testGet_FileHandleLeak() throws Exception {
        for (int i = 0; i < 100; ++i) {
            File file = TestFileUtils.createTempFile((String)"failure");
            this.transporter.get(new GetTask(URI.create("repo/file.txt")).setDataPath(file.toPath()));
            Assertions.assertTrue((boolean)file.delete(), (String)(i + ", " + file.getAbsolutePath()));
        }
    }

    @Test
    protected void testGet_NotFound() throws Exception {
        try {
            this.transporter.get(new GetTask(URI.create("repo/missing.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)404, (int)e.getStatusCode());
            Assertions.assertEquals((int)1, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testGet_Closed() throws Exception {
        this.transporter.close();
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (IllegalStateException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testGet_StartCancelled() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        listener.cancelStart();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        try {
            this.transporter.get(task);
            Assertions.fail((String)"Expected error");
        }
        catch (TransferCancelledException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
    }

    @Test
    protected void testGet_ProgressCancelled() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        listener.cancelProgress();
        GetTask task = new GetTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        try {
            this.transporter.get(task);
            Assertions.fail((String)"Expected error");
        }
        catch (TransferCancelledException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)4L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)1, (int)listener.getProgressedCount());
    }

    @Test
    protected void testPut_FromMemory() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_FromFile() throws Exception {
        File file = TestFileUtils.createTempFile((String)"upload");
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataPath(file.toPath());
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_EmptyResource() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener);
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)0L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
        Assertions.assertEquals((Object)"", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_EncodedResourcePath() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/some%20space.txt")).setListener((TransportListener)listener).setDataString("OK");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)2L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"OK", (Object)TestFileUtils.readString((File)new File(this.repoDir, "some space.txt")));
    }

    @Test
    protected void testPut_Authenticated_ExpectContinue() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_Authenticated_ExpectContinueBroken() throws Exception {
        this.session.setConfigProperty("aether.transport.http.supportWebDav", (Object)true);
        this.httpServer.setAuthentication("testuser", "testpass");
        this.httpServer.setExpectSupport(HttpServer.ExpectContinue.BROKEN);
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_Authenticated_ExpectContinueRejected() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.httpServer.setExpectSupport(HttpServer.ExpectContinue.FAIL);
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_Authenticated_ExpectContinueDisabled() throws Exception {
        this.session.setConfigProperty("aether.transport.http.expectContinue", (Object)false);
        this.httpServer.setAuthentication("testuser", "testpass");
        this.httpServer.setExpectSupport(HttpServer.ExpectContinue.FAIL);
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_Authenticated_ExpectContinueRejected_ExplicitlyConfiguredHeader() throws Exception {
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("Expect", "100-continue");
        this.session.setConfigProperty("aether.transport.http.headers.test", headers);
        this.httpServer.setAuthentication("testuser", "testpass");
        this.httpServer.setExpectSupport(HttpServer.ExpectContinue.FAIL);
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_Unauthenticated() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        try {
            this.transporter.put(task);
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)401, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((int)0, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
    }

    @Test
    protected void testPut_ProxyAuthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        Authentication auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort(), auth);
        this.newTransporter("http://bad.localhost:1/");
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_ProxyUnauthenticated() throws Exception {
        this.httpServer.setProxyAuthentication("testuser", "testpass");
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort());
        this.newTransporter("http://bad.localhost:1/");
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        try {
            this.transporter.put(task);
            Assertions.fail((String)"Expected error");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)407, (int)e.getStatusCode());
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((int)0, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
    }

    @Test
    protected void testPut_SSL() throws Exception {
        this.httpServer.addSslConnector();
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpsUrl());
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertTrue((listener.getProgressedCount() > 0 ? 1 : 0) != 0, (String)("Count: " + listener.getProgressedCount()));
        Assertions.assertEquals((Object)"upload", (Object)TestFileUtils.readString((File)new File(this.repoDir, "file.txt")));
    }

    @Test
    protected void testPut_FileHandleLeak() throws Exception {
        for (int i = 0; i < 100; ++i) {
            File src = TestFileUtils.createTempFile((String)"upload");
            File dst = new File(this.repoDir, "file.txt");
            this.transporter.put(new PutTask(URI.create("repo/file.txt")).setDataPath(src.toPath()));
            Assertions.assertTrue((boolean)src.delete(), (String)(i + ", " + src.getAbsolutePath()));
            Assertions.assertTrue((boolean)dst.delete(), (String)(i + ", " + dst.getAbsolutePath()));
        }
    }

    @Test
    protected void testPut_Closed() throws Exception {
        this.transporter.close();
        try {
            this.transporter.put(new PutTask(URI.create("repo/missing.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (IllegalStateException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    protected void testPut_StartCancelled() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        listener.cancelStart();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        try {
            this.transporter.put(task);
            Assertions.fail((String)"Expected error");
        }
        catch (TransferCancelledException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)0, (int)listener.getProgressedCount());
    }

    @Test
    protected void testPut_ProgressCancelled() throws Exception {
        RecordingTransportListener listener = new RecordingTransportListener();
        listener.cancelProgress();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        try {
            this.transporter.put(task);
            Assertions.fail((String)"Expected error");
        }
        catch (TransferCancelledException e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
        Assertions.assertEquals((long)0L, (long)listener.getDataOffset());
        Assertions.assertEquals((long)6L, (long)listener.getDataLength());
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
        Assertions.assertEquals((int)1, (int)listener.getProgressedCount());
    }

    @Test
    protected void testGetPut_AuthCache() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        GetTask get = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(get);
        RecordingTransportListener listener = new RecordingTransportListener();
        PutTask task = new PutTask(URI.create("repo/file.txt")).setListener((TransportListener)listener).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)1, (int)listener.getStartedCount());
    }

    @Test
    protected void testPut_PreemptiveIsDefault() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        PutTask task = new PutTask(URI.create("repo/file.txt")).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
    }

    @Test
    protected void testPut_AuthCache() throws Exception {
        this.session.setConfigProperty("aether.transport.http.preemptivePutAuth", (Object)false);
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        PutTask task = new PutTask(URI.create("repo/file.txt")).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)2, (int)this.httpServer.getLogEntries().size());
        this.httpServer.getLogEntries().clear();
        task = new PutTask(URI.create("repo/file.txt")).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
    }

    @Test
    protected void testPut_AuthCache_Preemptive() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.session.setConfigProperty("aether.transport.http.preemptiveAuth", (Object)true);
        this.newTransporter(this.httpServer.getHttpUrl());
        PutTask task = new PutTask(URI.create("repo/file.txt")).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
        this.httpServer.getLogEntries().clear();
        task = new PutTask(URI.create("repo/file.txt")).setDataString("upload");
        this.transporter.put(task);
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
    }

    @Test
    @Timeout(value=20L)
    protected void testConcurrency() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.newTransporter(this.httpServer.getHttpUrl());
        AtomicReference error = new AtomicReference();
        Thread[] threads = new Thread[20];
        for (int i = 0; i < threads.length; ++i) {
            String path = "repo/file.txt?i=" + i;
            threads[i] = new Thread(() -> {
                try {
                    for (int j = 0; j < 100; ++j) {
                        GetTask task = new GetTask(URI.create(path));
                        this.transporter.get(task);
                        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
                    }
                }
                catch (Throwable t) {
                    error.compareAndSet(null, t);
                    System.err.println(path);
                    t.printStackTrace();
                }
            });
            threads[i].setName("Task-" + i);
        }
        for (Thread thread : threads) {
            thread.start();
        }
        for (Thread thread : threads) {
            thread.join();
        }
        Assertions.assertNull(error.get(), (String)String.valueOf(error.get()));
    }

    @Test
    @Timeout(value=10L)
    protected void testConnectTimeout() throws Exception {
        this.session.setConfigProperty("aether.transport.http.connectTimeout", (Object)100);
        int port = 1;
        this.newTransporter("http://localhost:" + port);
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Expected error");
        }
        catch (Exception e) {
            Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
        }
    }

    @Test
    @Timeout(value=10L)
    protected void testRequestTimeout() throws Exception {
        ServerSocket server;
        this.session.setConfigProperty("aether.transport.http.requestTimeout", (Object)100);
        try (ServerSocket serverSocket = server = new ServerSocket(0);){
            this.newTransporter("http://localhost:" + server.getLocalPort());
            try {
                this.transporter.get(new GetTask(URI.create("repo/file.txt")));
                Assertions.fail((String)"Expected error");
            }
            catch (Exception e) {
                Assertions.assertTrue((boolean)e.getClass().getSimpleName().contains("Timeout"));
                Assertions.assertEquals((int)0, (int)this.transporter.classify((Throwable)e));
            }
        }
    }

    @Test
    protected void testUserAgent() throws Exception {
        this.session.setConfigProperty("aether.transport.http.userAgent", (Object)"SomeTest/1.0");
        this.newTransporter(this.httpServer.getHttpUrl());
        this.transporter.get(new GetTask(URI.create("repo/file.txt")));
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
        for (HttpServer.LogEntry log : this.httpServer.getLogEntries()) {
            Assertions.assertEquals((Object)"SomeTest/1.0", (Object)log.getHeaders().get("User-Agent"));
        }
    }

    @Test
    protected void testCustomHeaders() throws Exception {
        HashMap<String, String> headers = new HashMap<String, String>();
        headers.put("User-Agent", "Custom/1.0");
        headers.put("X-CustomHeader", "Custom-Value");
        this.session.setConfigProperty("aether.transport.http.userAgent", (Object)"SomeTest/1.0");
        this.session.setConfigProperty("aether.transport.http.headers.test", headers);
        this.newTransporter(this.httpServer.getHttpUrl());
        this.transporter.get(new GetTask(URI.create("repo/file.txt")));
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
        for (HttpServer.LogEntry log : this.httpServer.getLogEntries()) {
            for (Map.Entry entry : headers.entrySet()) {
                Assertions.assertEquals(entry.getValue(), (Object)log.getHeaders().get(entry.getKey()), (String)((String)entry.getKey()));
            }
        }
    }

    @Test
    protected void testServerAuthScope_NotUsedForProxy() throws Exception {
        String username = "testuser";
        String password = "testpass";
        this.httpServer.setProxyAuthentication(username, password);
        this.auth = new AuthenticationBuilder().addUsername(username).addPassword(password).build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort());
        this.newTransporter("http://" + this.httpServer.getHost() + ":12/");
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Server auth must not be used as proxy auth");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)407, (int)e.getStatusCode());
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    protected void testProxyAuthScope_NotUsedForServer() throws Exception {
        String username = "testuser";
        String password = "testpass";
        this.httpServer.setAuthentication(username, password);
        Authentication auth = new AuthenticationBuilder().addUsername(username).addPassword(password).build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort(), auth);
        this.newTransporter("http://" + this.httpServer.getHost() + ":12/");
        try {
            this.transporter.get(new GetTask(URI.create("repo/file.txt")));
            Assertions.fail((String)"Proxy auth must not be used as server auth");
        }
        catch (HttpTransporterException e) {
            Assertions.assertEquals((int)401, (int)e.getStatusCode());
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    @Test
    protected void testAuthSchemeReuse() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.httpServer.setProxyAuthentication("proxyuser", "proxypass");
        this.session.setCache((RepositoryCache)new DefaultRepositoryCache());
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        Authentication auth = new AuthenticationBuilder().addUsername("proxyuser").addPassword("proxypass").build();
        this.proxy = new Proxy("http", this.httpServer.getHost(), this.httpServer.getHttpPort(), auth);
        this.newTransporter("http://bad.localhost:1/");
        GetTask task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((int)3, (int)this.httpServer.getLogEntries().size());
        this.httpServer.getLogEntries().clear();
        this.newTransporter("http://bad.localhost:1/");
        task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
        Assertions.assertNotNull((Object)this.httpServer.getLogEntries().get(0).getHeaders().get("Authorization"));
        Assertions.assertNotNull((Object)this.httpServer.getLogEntries().get(0).getHeaders().get("Proxy-Authorization"));
    }

    @Test
    protected void testAuthSchemePreemptive() throws Exception {
        this.httpServer.setAuthentication("testuser", "testpass");
        this.session.setCache((RepositoryCache)new DefaultRepositoryCache());
        this.auth = new AuthenticationBuilder().addUsername("testuser").addPassword("testpass").build();
        this.session.setConfigProperty("aether.transport.http.preemptiveAuth", (Object)false);
        this.newTransporter(this.httpServer.getHttpUrl());
        GetTask task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((int)2, (int)this.httpServer.getLogEntries().size());
        this.httpServer.getLogEntries().clear();
        this.session.setConfigProperty("aether.transport.http.preemptiveAuth", (Object)true);
        this.newTransporter(this.httpServer.getHttpUrl());
        task = new GetTask(URI.create("repo/file.txt"));
        this.transporter.get(task);
        Assertions.assertEquals((Object)"test", (Object)task.getDataString());
        Assertions.assertEquals((int)1, (int)this.httpServer.getLogEntries().size());
    }

    @Test
    void testInit_BadProtocol() {
        Assertions.assertThrows(NoTransporterException.class, () -> this.newTransporter("bad:/void"));
    }

    @Test
    void testInit_BadUrl() {
        Assertions.assertThrows(NoTransporterException.class, () -> this.newTransporter("http://localhost:NaN"));
    }

    @Test
    void testInit_CaseInsensitiveProtocol() throws Exception {
        this.newTransporter("http://localhost");
        this.newTransporter("HTTP://localhost");
        this.newTransporter("Http://localhost");
        this.newTransporter("https://localhost");
        this.newTransporter("HTTPS://localhost");
        this.newTransporter("HttpS://localhost");
    }

    static {
        System.setProperty("javax.net.ssl.trustStore", KEY_STORE_PATH.toAbsolutePath().toString());
        System.setProperty("javax.net.ssl.trustStorePassword", "server-pwd");
        System.setProperty("javax.net.ssl.keyStore", TRUST_STORE_PATH.toAbsolutePath().toString());
        System.setProperty("javax.net.ssl.keyStorePassword", "client-pwd");
        System.setProperty("javax.net.ssl.trustStoreType", "jks");
        System.setProperty("javax.net.ssl.keyStoreType", "jks");
    }
}

