/*
 * $Id$
 */

package org.apache.maven.shared.release.config;

  //---------------------------------/
 //- Imported classes and packages -/
//---------------------------------/

import java.util.Date;

/**
 * Class ReleaseDescriptor.
 * 
 * @version $Revision$ $Date$
 */
public class ReleaseDescriptor implements java.io.Serializable {


      //--------------------------/
     //- Class/Member Variables -/
    //--------------------------/

    /**
     * 
     *             The name of this release configuration
     *           .
     */
    private String name;

    /**
     * 
     *             The description of this release configuration
     *           .
     */
    private String description;

    /**
     * 
     *             The prefix of SCM modification messages
     *           .
     */
    private String ScmCommentPrefix = "[maven-release-manager] ";

    /**
     * 
     *             This is a MavenSCM of where you're going to get
     * the sources to make the release with.
     *             Example:
     * scm:svn:https://svn.apache.org/repos/asf/maven/plugins/trunk/maven-release-plugin
     *           .
     */
    private String scmSourceUrl;

    /**
     * 
     *             Where you are going to put your tagged sources
     *             Example
     * https://svn.apache.org/repos/asf/maven/plugins/tags
     *           .
     */
    private String scmTagBase;

    /**
     * 
     *             tag name: the identifier for the tag.
     *             Example: maven-release-plugin-2.0
     *           .
     */
    private String scmReleaseLabel;

    /**
     * 
     *             The user name to interact with the scm
     *           .
     */
    private String scmUsername;

    /**
     * 
     *             The password for the user interacting with the
     * scm
     *           .
     */
    private String scmPassword;

    /**
     * 
     *             Private key for an SSH based SCM repository.
     *           
     */
    private String scmPrivateKey;

    /**
     * 
     *             Pass phrase for the private key.
     *           
     */
    private String scmPrivateKeyPassPhrase;

    /**
     * 
     *             Whether to use edit mode when making SCM
     * modifications. This setting is disregarded
     *             if the SCM does not support edit mode, or if
     * edit mode is compulsory for the given SCM.
     *           
     */
    private boolean scmUseEditMode = false;

    /**
     * 
     *             Relative path of the project returned by the
     * checkout command.
     *           
     */
    private String scmRelativePathProjectDirectory;

    /**
     * Field releaseVersions.
     */
    private java.util.Map releaseVersions;

    /**
     * Field developmentVersions.
     */
    private java.util.Map developmentVersions;

    /**
     * Field resolvedSnapshotDependencies.
     */
    private java.util.Map resolvedSnapshotDependencies;

    /**
     * 
     *             If updateDependencies is false, dependencies
     * version won't be updated to the next development version.
     *           
     */
    private boolean updateDependencies = true;

    /**
     * 
     *             Whether to use the release profile that adds
     * sources and javadocs to the released artifact, if
     * appropriate.
     *           
     */
    private boolean useReleaseProfile = true;

    /**
     * Field originalScmInfo.
     */
    private java.util.Map originalScmInfo;

    /**
     * 
     *             The directory where the release is performed.
     *           
     */
    private String workingDirectory;

    /**
     * 
     *             The directory where the tag will be checked out.
     *           
     */
    private String checkoutDirectory;

    /**
     * 
     *             The last completed phase.
     *           
     */
    private String completedPhase;

    /**
     * 
     *             Whether to add the model schema to the top of
     * the rewritten POM if it wasn't there already.
     *             If <code>false</code> then the root element will
     * remain untouched.
     *           
     */
    private boolean addSchema = false;

    /**
     * 
     *             Whether to generate release POMs.
     *           
     */
    private boolean generateReleasePoms = false;

    /**
     * 
     *             Whether to use the parent pom version for
     * submodule verstions.
     *           
     */
    private boolean autoVersionSubmodules = false;

    /**
     * 
     *             Whether the release process is interactive and
     * the release manager should be prompted to
     *             confirm values, or whether the defaults are used
     * regardless.
     *           
     */
    private boolean interactive = true;

    /**
     * 
     *             Whether a SNAPSHOT of the release plugin is
     * allowed.
     *           
     */
    private boolean snapshotReleasePluginAllowed = false;

    /**
     * 
     *             Additional arguments to pass to any executed
     * Maven process.
     *           
     */
    private String additionalArguments;

    /**
     * 
     *             The file name of the POM to pass to any executed
     * Maven process.
     *           
     */
    private String pomFileName;

    /**
     * 
     *             The goals to execute in preparation for the
     * release.
     *           
     */
    private String preparationGoals;

    /**
     * 
     *             The goals to execute in perform phase for the
     * release.
     *           
     */
    private String performGoals;

    /**
     * 
     *             The commits must be done by modules or not. Set
     * it to true in case of flat directory structure.
     *           
     */
    private boolean commitByProject = false;

    /**
     * 
     *             Whether to create a branch instead of do a
     * release.
     *           
     */
    private boolean branchCreation = false;

    /**
     * 
     *             Whether to update branch POM versions.
     *           
     */
    private boolean updateBranchVersions = false;

    /**
     * 
     *             Whether to update working copy POM versions.
     *           
     */
    private boolean updateWorkingCopyVersions = false;

    /**
     * 
     *             Whether to update branch versions to SNAPSHOT.
     *           
     */
    private boolean updateVersionsToSnapshot = false;

    /**
     * 
     *             Should timestamped SNAPSHOT dependencies be
     * allowed?  Default is to fail when any SNAPSHOT
     *               dependency is found.
     *           
     */
    private boolean allowTimestampedSnapshots = false;

    /**
     * 
     *             Default version to use for the tagged release or
     * the new branch.
     *           
     */
    private String defaultReleaseVersion;

    /**
     * 
     *             Default version to use for new working copy.
     *           
     */
    private String defaultDevelopmentVersion;

    /**
     * 
     *             NOTE : currently only implemented with svn scm.
     * Enable a workaround to prevent issue due to svn client >
     * 1.5.0 
     *             (http://jira.codehaus.org/browse/SCM-406)
     *           .
     */
    private boolean remoteTagging = false;

    /**
     * 
     *             NOTE : currently only implemented with svn scm.
     * It contains the revision of the committed released pom to
     * remotely tag
     *             the source code with this revision.
     *           
     */
    private String scmReleasedPomRevision;


      //-----------/
     //- Methods -/
    //-----------/

    /**
     * Method addDevelopmentVersion.
     * 
     * @param key
     * @param value
     */
    public void addDevelopmentVersion( Object key, String value )
    {
        getDevelopmentVersions().put( key, value );
    } //-- void addDevelopmentVersion( Object, String ) 

    /**
     * Method addOriginalScmInfo.
     * 
     * @param key
     * @param value
     */
    public void addOriginalScmInfo( Object key, String value )
    {
        getOriginalScmInfo().put( key, value );
    } //-- void addOriginalScmInfo( Object, String ) 

    /**
     * Method addReleaseVersion.
     * 
     * @param key
     * @param value
     */
    public void addReleaseVersion( Object key, String value )
    {
        getReleaseVersions().put( key, value );
    } //-- void addReleaseVersion( Object, String ) 

    /**
     * Method addResolvedSnapshotDependency.
     * 
     * @param key
     * @param value
     */
    public void addResolvedSnapshotDependency( Object key, String value )
    {
        getResolvedSnapshotDependencies().put( key, value );
    } //-- void addResolvedSnapshotDependency( Object, String ) 

    /**
     * Get 
     *             Additional arguments to pass to any executed
     * Maven process.
     *           
     * 
     * @return String
     */
    public String getAdditionalArguments()
    {
        return this.additionalArguments;
    } //-- String getAdditionalArguments() 

    /**
     * Get 
     *             The directory where the tag will be checked out.
     *           
     * 
     * @return String
     */
    public String getCheckoutDirectory()
    {
        return this.checkoutDirectory;
    } //-- String getCheckoutDirectory() 

    /**
     * Get 
     *             The last completed phase.
     *           
     * 
     * @return String
     */
    public String getCompletedPhase()
    {
        return this.completedPhase;
    } //-- String getCompletedPhase() 

    /**
     * Get 
     *             Default version to use for new working copy.
     *           
     * 
     * @return String
     */
    public String getDefaultDevelopmentVersion()
    {
        return this.defaultDevelopmentVersion;
    } //-- String getDefaultDevelopmentVersion() 

    /**
     * Get 
     *             Default version to use for the tagged release or
     * the new branch.
     *           
     * 
     * @return String
     */
    public String getDefaultReleaseVersion()
    {
        return this.defaultReleaseVersion;
    } //-- String getDefaultReleaseVersion() 

    /**
     * Get 
     *             The description of this release configuration
     *           .
     * 
     * @return String
     */
    public String getDescription()
    {
        return this.description;
    } //-- String getDescription() 

    /**
     * Method getDevelopmentVersions.
     * 
     * @return java.util.Map
     */
    public java.util.Map getDevelopmentVersions()
    {
        if ( this.developmentVersions == null )
        {
            this.developmentVersions = new java.util.HashMap();
        }
    
        return this.developmentVersions;
    } //-- java.util.Map getDevelopmentVersions() 

    /**
     * Get 
     *             The name of this release configuration
     *           .
     * 
     * @return String
     */
    public String getName()
    {
        return this.name;
    } //-- String getName() 

    /**
     * Method getOriginalScmInfo.
     * 
     * @return java.util.Map
     */
    public java.util.Map getOriginalScmInfo()
    {
        if ( this.originalScmInfo == null )
        {
            this.originalScmInfo = new java.util.HashMap();
        }
    
        return this.originalScmInfo;
    } //-- java.util.Map getOriginalScmInfo() 

    /**
     * Get 
     *             The goals to execute in perform phase for the
     * release.
     *           
     * 
     * @return String
     */
    public String getPerformGoals()
    {
        return this.performGoals;
    } //-- String getPerformGoals() 

    /**
     * Get 
     *             The file name of the POM to pass to any executed
     * Maven process.
     *           
     * 
     * @return String
     */
    public String getPomFileName()
    {
        return this.pomFileName;
    } //-- String getPomFileName() 

    /**
     * Get 
     *             The goals to execute in preparation for the
     * release.
     *           
     * 
     * @return String
     */
    public String getPreparationGoals()
    {
        return this.preparationGoals;
    } //-- String getPreparationGoals() 

    /**
     * Method getReleaseVersions.
     * 
     * @return java.util.Map
     */
    public java.util.Map getReleaseVersions()
    {
        if ( this.releaseVersions == null )
        {
            this.releaseVersions = new java.util.HashMap();
        }
    
        return this.releaseVersions;
    } //-- java.util.Map getReleaseVersions() 

    /**
     * Method getResolvedSnapshotDependencies.
     * 
     * @return java.util.Map
     */
    public java.util.Map getResolvedSnapshotDependencies()
    {
        if ( this.resolvedSnapshotDependencies == null )
        {
            this.resolvedSnapshotDependencies = new java.util.HashMap();
        }
    
        return this.resolvedSnapshotDependencies;
    } //-- java.util.Map getResolvedSnapshotDependencies() 

    /**
     * Get 
     *             The prefix of SCM modification messages
     *           .
     * 
     * @return String
     */
    public String getScmCommentPrefix()
    {
        return this.ScmCommentPrefix;
    } //-- String getScmCommentPrefix() 

    /**
     * Get 
     *             The password for the user interacting with the
     * scm
     *           .
     * 
     * @return String
     */
    public String getScmPassword()
    {
        return this.scmPassword;
    } //-- String getScmPassword() 

    /**
     * Get 
     *             Private key for an SSH based SCM repository.
     *           
     * 
     * @return String
     */
    public String getScmPrivateKey()
    {
        return this.scmPrivateKey;
    } //-- String getScmPrivateKey() 

    /**
     * Get 
     *             Pass phrase for the private key.
     *           
     * 
     * @return String
     */
    public String getScmPrivateKeyPassPhrase()
    {
        return this.scmPrivateKeyPassPhrase;
    } //-- String getScmPrivateKeyPassPhrase() 

    /**
     * Get 
     *             Relative path of the project returned by the
     * checkout command.
     *           
     * 
     * @return String
     */
    public String getScmRelativePathProjectDirectory()
    {
        return this.scmRelativePathProjectDirectory;
    } //-- String getScmRelativePathProjectDirectory() 

    /**
     * Get 
     *             tag name: the identifier for the tag.
     *             Example: maven-release-plugin-2.0
     *           .
     * 
     * @return String
     */
    public String getScmReleaseLabel()
    {
        return this.scmReleaseLabel;
    } //-- String getScmReleaseLabel() 

    /**
     * Get 
     *             NOTE : currently only implemented with svn scm.
     * It contains the revision of the committed released pom to
     * remotely tag
     *             the source code with this revision.
     *           
     * 
     * @return String
     */
    public String getScmReleasedPomRevision()
    {
        return this.scmReleasedPomRevision;
    } //-- String getScmReleasedPomRevision() 

    /**
     * Get 
     *             This is a MavenSCM of where you're going to get
     * the sources to make the release with.
     *             Example:
     * scm:svn:https://svn.apache.org/repos/asf/maven/plugins/trunk/maven-release-plugin
     *           .
     * 
     * @return String
     */
    public String getScmSourceUrl()
    {
        return this.scmSourceUrl;
    } //-- String getScmSourceUrl() 

    /**
     * Get 
     *             Where you are going to put your tagged sources
     *             Example
     * https://svn.apache.org/repos/asf/maven/plugins/tags
     *           .
     * 
     * @return String
     */
    public String getScmTagBase()
    {
        return this.scmTagBase;
    } //-- String getScmTagBase() 

    /**
     * Get 
     *             The user name to interact with the scm
     *           .
     * 
     * @return String
     */
    public String getScmUsername()
    {
        return this.scmUsername;
    } //-- String getScmUsername() 

    /**
     * Get 
     *             The directory where the release is performed.
     *           
     * 
     * @return String
     */
    public String getWorkingDirectory()
    {
        return this.workingDirectory;
    } //-- String getWorkingDirectory() 

    /**
     * Get 
     *             Whether to add the model schema to the top of
     * the rewritten POM if it wasn't there already.
     *             If <code>false</code> then the root element will
     * remain untouched.
     *           
     * 
     * @return boolean
     */
    public boolean isAddSchema()
    {
        return this.addSchema;
    } //-- boolean isAddSchema() 

    /**
     * Get 
     *             Should timestamped SNAPSHOT dependencies be
     * allowed?  Default is to fail when any SNAPSHOT
     *               dependency is found.
     *           
     * 
     * @return boolean
     */
    public boolean isAllowTimestampedSnapshots()
    {
        return this.allowTimestampedSnapshots;
    } //-- boolean isAllowTimestampedSnapshots() 

    /**
     * Get 
     *             Whether to use the parent pom version for
     * submodule verstions.
     *           
     * 
     * @return boolean
     */
    public boolean isAutoVersionSubmodules()
    {
        return this.autoVersionSubmodules;
    } //-- boolean isAutoVersionSubmodules() 

    /**
     * Get 
     *             Whether to create a branch instead of do a
     * release.
     *           
     * 
     * @return boolean
     */
    public boolean isBranchCreation()
    {
        return this.branchCreation;
    } //-- boolean isBranchCreation() 

    /**
     * Get 
     *             The commits must be done by modules or not. Set
     * it to true in case of flat directory structure.
     *           
     * 
     * @return boolean
     */
    public boolean isCommitByProject()
    {
        return this.commitByProject;
    } //-- boolean isCommitByProject() 

    /**
     * Get 
     *             Whether to generate release POMs.
     *           
     * 
     * @return boolean
     */
    public boolean isGenerateReleasePoms()
    {
        return this.generateReleasePoms;
    } //-- boolean isGenerateReleasePoms() 

    /**
     * Get 
     *             Whether the release process is interactive and
     * the release manager should be prompted to
     *             confirm values, or whether the defaults are used
     * regardless.
     *           
     * 
     * @return boolean
     */
    public boolean isInteractive()
    {
        return this.interactive;
    } //-- boolean isInteractive() 

    /**
     * Get 
     *             NOTE : currently only implemented with svn scm.
     * Enable a workaround to prevent issue due to svn client >
     * 1.5.0 
     *             (http://jira.codehaus.org/browse/SCM-406)
     *           .
     * 
     * @return boolean
     */
    public boolean isRemoteTagging()
    {
        return this.remoteTagging;
    } //-- boolean isRemoteTagging() 

    /**
     * Get 
     *             Whether to use edit mode when making SCM
     * modifications. This setting is disregarded
     *             if the SCM does not support edit mode, or if
     * edit mode is compulsory for the given SCM.
     *           
     * 
     * @return boolean
     */
    public boolean isScmUseEditMode()
    {
        return this.scmUseEditMode;
    } //-- boolean isScmUseEditMode() 

    /**
     * Get 
     *             Whether a SNAPSHOT of the release plugin is
     * allowed.
     *           
     * 
     * @return boolean
     */
    public boolean isSnapshotReleasePluginAllowed()
    {
        return this.snapshotReleasePluginAllowed;
    } //-- boolean isSnapshotReleasePluginAllowed() 

    /**
     * Get 
     *             Whether to update branch POM versions.
     *           
     * 
     * @return boolean
     */
    public boolean isUpdateBranchVersions()
    {
        return this.updateBranchVersions;
    } //-- boolean isUpdateBranchVersions() 

    /**
     * Get 
     *             If updateDependencies is false, dependencies
     * version won't be updated to the next development version.
     *           
     * 
     * @return boolean
     */
    public boolean isUpdateDependencies()
    {
        return this.updateDependencies;
    } //-- boolean isUpdateDependencies() 

    /**
     * Get 
     *             Whether to update branch versions to SNAPSHOT.
     *           
     * 
     * @return boolean
     */
    public boolean isUpdateVersionsToSnapshot()
    {
        return this.updateVersionsToSnapshot;
    } //-- boolean isUpdateVersionsToSnapshot() 

    /**
     * Get 
     *             Whether to update working copy POM versions.
     *           
     * 
     * @return boolean
     */
    public boolean isUpdateWorkingCopyVersions()
    {
        return this.updateWorkingCopyVersions;
    } //-- boolean isUpdateWorkingCopyVersions() 

    /**
     * Get 
     *             Whether to use the release profile that adds
     * sources and javadocs to the released artifact, if
     * appropriate.
     *           
     * 
     * @return boolean
     */
    public boolean isUseReleaseProfile()
    {
        return this.useReleaseProfile;
    } //-- boolean isUseReleaseProfile() 

    /**
     * Set 
     *             Whether to add the model schema to the top of
     * the rewritten POM if it wasn't there already.
     *             If <code>false</code> then the root element will
     * remain untouched.
     *           
     * 
     * @param addSchema
     */
    public void setAddSchema( boolean addSchema )
    {
        this.addSchema = addSchema;
    } //-- void setAddSchema( boolean ) 

    /**
     * Set 
     *             Additional arguments to pass to any executed
     * Maven process.
     *           
     * 
     * @param additionalArguments
     */
    public void setAdditionalArguments( String additionalArguments )
    {
        this.additionalArguments = additionalArguments;
    } //-- void setAdditionalArguments( String ) 

    /**
     * Set 
     *             Should timestamped SNAPSHOT dependencies be
     * allowed?  Default is to fail when any SNAPSHOT
     *               dependency is found.
     *           
     * 
     * @param allowTimestampedSnapshots
     */
    public void setAllowTimestampedSnapshots( boolean allowTimestampedSnapshots )
    {
        this.allowTimestampedSnapshots = allowTimestampedSnapshots;
    } //-- void setAllowTimestampedSnapshots( boolean ) 

    /**
     * Set 
     *             Whether to use the parent pom version for
     * submodule verstions.
     *           
     * 
     * @param autoVersionSubmodules
     */
    public void setAutoVersionSubmodules( boolean autoVersionSubmodules )
    {
        this.autoVersionSubmodules = autoVersionSubmodules;
    } //-- void setAutoVersionSubmodules( boolean ) 

    /**
     * Set 
     *             Whether to create a branch instead of do a
     * release.
     *           
     * 
     * @param branchCreation
     */
    public void setBranchCreation( boolean branchCreation )
    {
        this.branchCreation = branchCreation;
    } //-- void setBranchCreation( boolean ) 

    /**
     * Set 
     *             The directory where the tag will be checked out.
     *           
     * 
     * @param checkoutDirectory
     */
    public void setCheckoutDirectory( String checkoutDirectory )
    {
        this.checkoutDirectory = checkoutDirectory;
    } //-- void setCheckoutDirectory( String ) 

    /**
     * Set 
     *             The commits must be done by modules or not. Set
     * it to true in case of flat directory structure.
     *           
     * 
     * @param commitByProject
     */
    public void setCommitByProject( boolean commitByProject )
    {
        this.commitByProject = commitByProject;
    } //-- void setCommitByProject( boolean ) 

    /**
     * Set 
     *             The last completed phase.
     *           
     * 
     * @param completedPhase
     */
    public void setCompletedPhase( String completedPhase )
    {
        this.completedPhase = completedPhase;
    } //-- void setCompletedPhase( String ) 

    /**
     * Set 
     *             Default version to use for new working copy.
     *           
     * 
     * @param defaultDevelopmentVersion
     */
    public void setDefaultDevelopmentVersion( String defaultDevelopmentVersion )
    {
        this.defaultDevelopmentVersion = defaultDevelopmentVersion;
    } //-- void setDefaultDevelopmentVersion( String ) 

    /**
     * Set 
     *             Default version to use for the tagged release or
     * the new branch.
     *           
     * 
     * @param defaultReleaseVersion
     */
    public void setDefaultReleaseVersion( String defaultReleaseVersion )
    {
        this.defaultReleaseVersion = defaultReleaseVersion;
    } //-- void setDefaultReleaseVersion( String ) 

    /**
     * Set 
     *             The description of this release configuration
     *           .
     * 
     * @param description
     */
    public void setDescription( String description )
    {
        this.description = description;
    } //-- void setDescription( String ) 

    /**
     * Set 
     *             A map of projects to versions to use when moving
     * the given projects back into devlopment after release.
     *           
     * 
     * @param developmentVersions
     */
    public void setDevelopmentVersions( java.util.Map developmentVersions )
    {
        this.developmentVersions = developmentVersions;
    } //-- void setDevelopmentVersions( java.util.Map ) 

    /**
     * Set 
     *             Whether to generate release POMs.
     *           
     * 
     * @param generateReleasePoms
     */
    public void setGenerateReleasePoms( boolean generateReleasePoms )
    {
        this.generateReleasePoms = generateReleasePoms;
    } //-- void setGenerateReleasePoms( boolean ) 

    /**
     * Set 
     *             Whether the release process is interactive and
     * the release manager should be prompted to
     *             confirm values, or whether the defaults are used
     * regardless.
     *           
     * 
     * @param interactive
     */
    public void setInteractive( boolean interactive )
    {
        this.interactive = interactive;
    } //-- void setInteractive( boolean ) 

    /**
     * Set 
     *             The name of this release configuration
     *           .
     * 
     * @param name
     */
    public void setName( String name )
    {
        this.name = name;
    } //-- void setName( String ) 

    /**
     * Set 
     *             A map of projects to original SCM information.
     *           
     * 
     * @param originalScmInfo
     */
    public void setOriginalScmInfo( java.util.Map originalScmInfo )
    {
        this.originalScmInfo = originalScmInfo;
    } //-- void setOriginalScmInfo( java.util.Map ) 

    /**
     * Set 
     *             The goals to execute in perform phase for the
     * release.
     *           
     * 
     * @param performGoals
     */
    public void setPerformGoals( String performGoals )
    {
        this.performGoals = performGoals;
    } //-- void setPerformGoals( String ) 

    /**
     * Set 
     *             The file name of the POM to pass to any executed
     * Maven process.
     *           
     * 
     * @param pomFileName
     */
    public void setPomFileName( String pomFileName )
    {
        this.pomFileName = pomFileName;
    } //-- void setPomFileName( String ) 

    /**
     * Set 
     *             The goals to execute in preparation for the
     * release.
     *           
     * 
     * @param preparationGoals
     */
    public void setPreparationGoals( String preparationGoals )
    {
        this.preparationGoals = preparationGoals;
    } //-- void setPreparationGoals( String ) 

    /**
     * Set 
     *             A map of projects to versions to use when
     * releasing the given projects.
     *           
     * 
     * @param releaseVersions
     */
    public void setReleaseVersions( java.util.Map releaseVersions )
    {
        this.releaseVersions = releaseVersions;
    } //-- void setReleaseVersions( java.util.Map ) 

    /**
     * Set 
     *             NOTE : currently only implemented with svn scm.
     * Enable a workaround to prevent issue due to svn client >
     * 1.5.0 
     *             (http://jira.codehaus.org/browse/SCM-406)
     *           .
     * 
     * @param remoteTagging
     */
    public void setRemoteTagging( boolean remoteTagging )
    {
        this.remoteTagging = remoteTagging;
    } //-- void setRemoteTagging( boolean ) 

    /**
     * Set 
     *             A map of resolved snapshot dependencies
     * versions.
     *           
     * 
     * @param resolvedSnapshotDependencies
     */
    public void setResolvedSnapshotDependencies( java.util.Map resolvedSnapshotDependencies )
    {
        this.resolvedSnapshotDependencies = resolvedSnapshotDependencies;
    } //-- void setResolvedSnapshotDependencies( java.util.Map ) 

    /**
     * Set 
     *             The prefix of SCM modification messages
     *           .
     * 
     * @param ScmCommentPrefix
     */
    public void setScmCommentPrefix( String ScmCommentPrefix )
    {
        this.ScmCommentPrefix = ScmCommentPrefix;
    } //-- void setScmCommentPrefix( String ) 

    /**
     * Set 
     *             The password for the user interacting with the
     * scm
     *           .
     * 
     * @param scmPassword
     */
    public void setScmPassword( String scmPassword )
    {
        this.scmPassword = scmPassword;
    } //-- void setScmPassword( String ) 

    /**
     * Set 
     *             Private key for an SSH based SCM repository.
     *           
     * 
     * @param scmPrivateKey
     */
    public void setScmPrivateKey( String scmPrivateKey )
    {
        this.scmPrivateKey = scmPrivateKey;
    } //-- void setScmPrivateKey( String ) 

    /**
     * Set 
     *             Pass phrase for the private key.
     *           
     * 
     * @param scmPrivateKeyPassPhrase
     */
    public void setScmPrivateKeyPassPhrase( String scmPrivateKeyPassPhrase )
    {
        this.scmPrivateKeyPassPhrase = scmPrivateKeyPassPhrase;
    } //-- void setScmPrivateKeyPassPhrase( String ) 

    /**
     * Set 
     *             Relative path of the project returned by the
     * checkout command.
     *           
     * 
     * @param scmRelativePathProjectDirectory
     */
    public void setScmRelativePathProjectDirectory( String scmRelativePathProjectDirectory )
    {
        this.scmRelativePathProjectDirectory = scmRelativePathProjectDirectory;
    } //-- void setScmRelativePathProjectDirectory( String ) 

    /**
     * Set 
     *             tag name: the identifier for the tag.
     *             Example: maven-release-plugin-2.0
     *           .
     * 
     * @param scmReleaseLabel
     */
    public void setScmReleaseLabel( String scmReleaseLabel )
    {
        this.scmReleaseLabel = scmReleaseLabel;
    } //-- void setScmReleaseLabel( String ) 

    /**
     * Set 
     *             NOTE : currently only implemented with svn scm.
     * It contains the revision of the committed released pom to
     * remotely tag
     *             the source code with this revision.
     *           
     * 
     * @param scmReleasedPomRevision
     */
    public void setScmReleasedPomRevision( String scmReleasedPomRevision )
    {
        this.scmReleasedPomRevision = scmReleasedPomRevision;
    } //-- void setScmReleasedPomRevision( String ) 

    /**
     * Set 
     *             This is a MavenSCM of where you're going to get
     * the sources to make the release with.
     *             Example:
     * scm:svn:https://svn.apache.org/repos/asf/maven/plugins/trunk/maven-release-plugin
     *           .
     * 
     * @param scmSourceUrl
     */
    public void setScmSourceUrl( String scmSourceUrl )
    {
        this.scmSourceUrl = scmSourceUrl;
    } //-- void setScmSourceUrl( String ) 

    /**
     * Set 
     *             Where you are going to put your tagged sources
     *             Example
     * https://svn.apache.org/repos/asf/maven/plugins/tags
     *           .
     * 
     * @param scmTagBase
     */
    public void setScmTagBase( String scmTagBase )
    {
        this.scmTagBase = scmTagBase;
    } //-- void setScmTagBase( String ) 

    /**
     * Set 
     *             Whether to use edit mode when making SCM
     * modifications. This setting is disregarded
     *             if the SCM does not support edit mode, or if
     * edit mode is compulsory for the given SCM.
     *           
     * 
     * @param scmUseEditMode
     */
    public void setScmUseEditMode( boolean scmUseEditMode )
    {
        this.scmUseEditMode = scmUseEditMode;
    } //-- void setScmUseEditMode( boolean ) 

    /**
     * Set 
     *             The user name to interact with the scm
     *           .
     * 
     * @param scmUsername
     */
    public void setScmUsername( String scmUsername )
    {
        this.scmUsername = scmUsername;
    } //-- void setScmUsername( String ) 

    /**
     * Set 
     *             Whether a SNAPSHOT of the release plugin is
     * allowed.
     *           
     * 
     * @param snapshotReleasePluginAllowed
     */
    public void setSnapshotReleasePluginAllowed( boolean snapshotReleasePluginAllowed )
    {
        this.snapshotReleasePluginAllowed = snapshotReleasePluginAllowed;
    } //-- void setSnapshotReleasePluginAllowed( boolean ) 

    /**
     * Set 
     *             Whether to update branch POM versions.
     *           
     * 
     * @param updateBranchVersions
     */
    public void setUpdateBranchVersions( boolean updateBranchVersions )
    {
        this.updateBranchVersions = updateBranchVersions;
    } //-- void setUpdateBranchVersions( boolean ) 

    /**
     * Set 
     *             If updateDependencies is false, dependencies
     * version won't be updated to the next development version.
     *           
     * 
     * @param updateDependencies
     */
    public void setUpdateDependencies( boolean updateDependencies )
    {
        this.updateDependencies = updateDependencies;
    } //-- void setUpdateDependencies( boolean ) 

    /**
     * Set 
     *             Whether to update branch versions to SNAPSHOT.
     *           
     * 
     * @param updateVersionsToSnapshot
     */
    public void setUpdateVersionsToSnapshot( boolean updateVersionsToSnapshot )
    {
        this.updateVersionsToSnapshot = updateVersionsToSnapshot;
    } //-- void setUpdateVersionsToSnapshot( boolean ) 

    /**
     * Set 
     *             Whether to update working copy POM versions.
     *           
     * 
     * @param updateWorkingCopyVersions
     */
    public void setUpdateWorkingCopyVersions( boolean updateWorkingCopyVersions )
    {
        this.updateWorkingCopyVersions = updateWorkingCopyVersions;
    } //-- void setUpdateWorkingCopyVersions( boolean ) 

    /**
     * Set 
     *             Whether to use the release profile that adds
     * sources and javadocs to the released artifact, if
     * appropriate.
     *           
     * 
     * @param useReleaseProfile
     */
    public void setUseReleaseProfile( boolean useReleaseProfile )
    {
        this.useReleaseProfile = useReleaseProfile;
    } //-- void setUseReleaseProfile( boolean ) 

    /**
     * Set 
     *             The directory where the release is performed.
     *           
     * 
     * @param workingDirectory
     */
    public void setWorkingDirectory( String workingDirectory )
    {
        this.workingDirectory = workingDirectory;
    } //-- void setWorkingDirectory( String ) 


    public static final String DEVELOPMENT_KEY = "dev";
    
    public static final String RELEASE_KEY = "rel";
    
    public static final String ORIGINAL_VERSION = "original";
                        
    /**
     * Map a given snapshot dependency to a specified version from when it is released.
     *
     */
    public void mapResolvedSnapshotDependencies( String artifactName, String releaseVersion, String developmentVersion )
    {
        if (resolvedSnapshotDependencies == null) {
            resolvedSnapshotDependencies = new java.util.HashMap();
        } else {
            assert !resolvedSnapshotDependencies.containsKey( artifactName );
        }

        java.util.Map versionsMap = new java.util.HashMap();

        versionsMap.put( DEVELOPMENT_KEY, developmentVersion );
        versionsMap.put( RELEASE_KEY, releaseVersion );
        resolvedSnapshotDependencies.put( artifactName, versionsMap );
    }
          
    /**
     * Map a given project to a specified version from when it is released.
     *
     * @param projectId   the project's group and artifact ID
     * @param nextVersion the version to map to
     */
    public void mapReleaseVersion( String projectId, String nextVersion )
    {
        if (releaseVersions == null) {
            releaseVersions = new java.util.HashMap();
        } else {
            assert !releaseVersions.containsKey( projectId );
        }

        releaseVersions.put( projectId, nextVersion );
    }

    /**
     * Map a given project to a specified version from when it is incremented and placed back into development.
     *
     * @param projectId   the project's group and artifact ID
     * @param nextVersion the version to map to
     */
    public void mapDevelopmentVersion( String projectId, String nextVersion )
    {
        if (developmentVersions == null) {
            developmentVersions = new java.util.HashMap();
        } else {
            assert !developmentVersions.containsKey( projectId );
        }

        developmentVersions.put( projectId, nextVersion );
    }

    /**
     * Map a given project to a specified set of SCM information.
     *
     * @param projectId the project's group and artifact ID
     * @param scm       the original SCM information to store
     */
    public void mapOriginalScmInfo( String projectId, org.apache.maven.model.Scm scm )
    {
        if (originalScmInfo == null) {
            originalScmInfo = new java.util.HashMap();
        } else {
            assert !originalScmInfo.containsKey( projectId );
        }

        originalScmInfo.put( projectId, scm );
    }

    /**
     * A map of projects to original versions before any transformation.
     */
    private java.util.Map originalVersions;

    /**
     * Retrieve the original version map, before transformation, keyed by project's versionless identifier.
     *
     * @param reactorProjects The reactor projects.
     *
     * @return the map of project IDs to versions.
     */
    public synchronized java.util.Map getOriginalVersions( java.util.List reactorProjects )
    {
        if ( originalVersions == null )
        {
            originalVersions = new java.util.HashMap();
            for ( java.util.Iterator i = reactorProjects.iterator(); i.hasNext(); )
            {
                org.apache.maven.project.MavenProject project = (org.apache.maven.project.MavenProject) i.next();
                originalVersions.put( org.apache.maven.artifact.ArtifactUtils.versionlessKey( project.getGroupId(), project.getArtifactId() ),
                                      project.getVersion() );
            }
        }
        return originalVersions;
    }

    /**
     * Retrieve the release version for the resolved snapshot dependency.
     *
     */
    public String getDependencyDevelopmentVersion( String artifactConflictId )
    {
        if ( ( resolvedSnapshotDependencies == null ) && ( resolvedSnapshotDependencies.containsKey( artifactConflictId ) ) )
        {
            java.util.Map versionMap = ( java.util.Map ) resolvedSnapshotDependencies.get( artifactConflictId );
            return ( String ) versionMap.get( DEVELOPMENT_KEY );
        }
        return null;
    }

    /**
     * Retrieve the release version for the resolved snapshot dependency.
     *
     */
    public String getDependencyReleaseVersion( String artifactConflictId )
    {
        if ( ( resolvedSnapshotDependencies == null ) && ( resolvedSnapshotDependencies.containsKey( artifactConflictId ) ) )
        {
            java.util.Map versionMap = ( java.util.Map ) resolvedSnapshotDependencies.get( artifactConflictId );
            return ( String ) versionMap.get( RELEASE_KEY );
        }
        return null;
    }

    public boolean equals( Object obj )
    {
        if ( this == obj )
        {
            return true;
        }
        if ( obj == null || getClass() != obj.getClass() )
        {
            return false;
        }

        ReleaseDescriptor that = (ReleaseDescriptor) obj;

        if ( addSchema != that.addSchema )
        {
            return false;
        }
        if ( generateReleasePoms != that.generateReleasePoms )
        {
            return false;
        }
        if ( interactive != that.interactive )
        {
            return false;
        }
        if ( scmUseEditMode != that.scmUseEditMode )
        {
            return false;
        }
        if ( completedPhase != null ? !completedPhase.equals( that.completedPhase ) : that.completedPhase != null )
        {
            return false;
        }
        if ( ( developmentVersions == null || ( developmentVersions != null && developmentVersions.size() == 0) ) &&
            ( that.developmentVersions == null || ( that.developmentVersions != null && that.developmentVersions.size() == 0) ) ) {
            // Do nothing.  This is a Modello workaround
        } else {
            if ( developmentVersions != null ? !developmentVersions.equals( that.developmentVersions )
                : that.developmentVersions != null )
            {
                return false;
            }
        }
        if ( ( originalScmInfo == null || (originalScmInfo != null && originalScmInfo.size() == 0 ) ) &&
            ( that.originalScmInfo == null || ( that.originalScmInfo != null && that.originalScmInfo.size() == 0 ) ) ) {
            // Do nothing.  This is a Modello workaround
        } else {
            if ( originalScmInfo != null ? !compareScmCollections( that.originalScmInfo ) : that.originalScmInfo != null )
            {
                return false;
            }
        }
        if ( ( releaseVersions == null || ( releaseVersions != null && releaseVersions.size() == 0 ) ) &&
            ( that.releaseVersions == null || ( that.releaseVersions != null && that.releaseVersions.size() == 0 ) ) ) {
            // Do nothing.  This is a Modello workaround
        } else {
            if ( releaseVersions != null ? !releaseVersions.equals( that.releaseVersions ) : that.releaseVersions != null )
            {
                return false;
            }
        }
        if ( additionalArguments != null ? !additionalArguments.equals( that.additionalArguments )
            : that.additionalArguments != null )
        {
            return false;
        }
        if ( preparationGoals != null ? !preparationGoals.equals( that.preparationGoals )
            : that.preparationGoals != null )
        {
            return false;
        }
        if ( pomFileName != null ? !pomFileName.equals( that.pomFileName ) : that.pomFileName != null )
        {
            return false;
        }
        if ( scmPrivateKeyPassPhrase != null ? !scmPrivateKeyPassPhrase.equals( that.scmPrivateKeyPassPhrase ) : that.scmPrivateKeyPassPhrase != null )
        {
            return false;
        }
        if ( scmPassword != null ? !scmPassword.equals( that.scmPassword ) : that.scmPassword != null )
        {
            return false;
        }
        if ( scmPrivateKey != null ? !scmPrivateKey.equals( that.scmPrivateKey ) : that.scmPrivateKey != null )
        {
            return false;
        }
        if ( scmReleaseLabel != null ? !scmReleaseLabel.equals( that.scmReleaseLabel ) : that.scmReleaseLabel != null )
        {
            return false;
        }
        if ( scmTagBase != null ? !scmTagBase.equals( that.scmTagBase ) : that.scmTagBase != null )
        {
            return false;
        }
        if ( scmSourceUrl != null ? !scmSourceUrl.equals( that.scmSourceUrl ) : that.scmSourceUrl != null )
        {
            return false;
        }
        if ( scmUsername != null ? !scmUsername.equals( that.scmUsername ) : that.scmUsername != null )
        {
            return false;
        }
        if ( workingDirectory != null ? !workingDirectory.equals( that.workingDirectory ) : that.workingDirectory != null )
        {
            return false;
        }

        return true;
    }

    private boolean compareScmCollections( java.util.Map that )
    {
        // Must manually compare as Scm doesn't have an equals method
        if (that == null && originalScmInfo == null) {
            return true;
        }

        if ( ( that == null && originalScmInfo != null ) || ( that != null && originalScmInfo == null ) ) {
            return false;
        }

        if ( that.size() != originalScmInfo.size() )
        {
            return false;
        }

        for ( java.util.Iterator i = originalScmInfo.entrySet().iterator(); i.hasNext(); )
        {
            java.util.Map.Entry entry = (java.util.Map.Entry) i.next();

            org.apache.maven.model.Scm thatScm = (org.apache.maven.model.Scm) that.get( entry.getKey() );

            org.apache.maven.model.Scm thisScm = (org.apache.maven.model.Scm) entry.getValue();
            if ( thatScm == null && thisScm == null )
            {
                return true;
            }
            else if ( thatScm == null )
            {
                return false;
            }
            else if ( thisScm == null )
            {
                return true;
            }

            if ( thisScm.getConnection() != null ? !thisScm.getConnection().equals( thatScm.getConnection() )
                : thatScm.getConnection() != null )
            {
                return false;
            }
            if ( thisScm.getDeveloperConnection() != null ? !thisScm.getDeveloperConnection().equals(
                thatScm.getDeveloperConnection() ) : thatScm.getDeveloperConnection() != null )
            {
                return false;
            }
            if ( thisScm.getUrl() != null ? !thisScm.getUrl().equals( thatScm.getUrl() ) : thatScm.getUrl() != null )
            {
                return false;
            }
            if ( thisScm.getTag() != null ? !thisScm.getTag().equals( thatScm.getTag() ) : thatScm.getTag() != null )
            {
                return false;
            }
        }

        return true;
    }

    public int hashCode()
    {
        int result = completedPhase != null ? completedPhase.hashCode() : 0;
        result = 29 * result + ( scmReleaseLabel != null ? scmReleaseLabel.hashCode() : 0 );
        result = 29 * result + ( additionalArguments != null ? additionalArguments.hashCode() : 0 );
        result = 29 * result + ( preparationGoals != null ? preparationGoals.hashCode() : 0 );
        result = 29 * result + ( pomFileName != null ? pomFileName.hashCode() : 0 );
        result = 29 * result + ( scmTagBase != null ? scmTagBase.hashCode() : 0 );
        result = 29 * result + ( scmUsername != null ? scmUsername.hashCode() : 0 );
        result = 29 * result + ( scmPassword != null ? scmPassword.hashCode() : 0 );
        result = 29 * result + ( scmSourceUrl != null ? scmSourceUrl.hashCode() : 0 );
        result = 29 * result + ( scmPrivateKey != null ? scmPrivateKey.hashCode() : 0 );
        result = 29 * result + ( scmPrivateKeyPassPhrase != null ? scmPrivateKeyPassPhrase.hashCode() : 0 );
        result = 29 * result + ( workingDirectory != null ? workingDirectory.hashCode() : 0 );
        result = 29 * result + ( scmUseEditMode ? 1 : 0 );
        result = 29 * result + ( addSchema ? 1 : 0 );
        result = 29 * result + ( generateReleasePoms ? 1 : 0 );
        result = 29 * result + ( interactive ? 1 : 0 );
        result = 29 * result + ( releaseVersions != null ? releaseVersions.hashCode() : 0 );
        result = 29 * result + ( developmentVersions != null ? developmentVersions.hashCode() : 0 );
        result = 29 * result + ( originalScmInfo != null ? originalScmInfo.hashCode() : 0 );
        return result;
    }

    private String modelEncoding = "UTF-8";

    /**
     * Set an encoding used for reading/writing the model.
     *
     * @param modelEncoding the encoding used when reading/writing the model.
     */
    public void setModelEncoding( String modelEncoding )
    {
        this.modelEncoding = modelEncoding;
    }

    /**
     * @return the current encoding used when reading/writing this model.
     */
    public String getModelEncoding()
    {
        return modelEncoding;
    }
}
