package org.apache.maven.plugin.idea;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.artifact.factory.ArtifactFactory;
import org.apache.maven.artifact.metadata.ArtifactMetadataSource;
import org.apache.maven.artifact.repository.ArtifactRepository;
import org.apache.maven.artifact.resolver.ArtifactResolver;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.FileUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;

import java.io.File;
import java.io.IOException;

/**
 * Creates the workspace file (*.iws) for IntelliJ IDEA.
 *
 * @author Edwin Punzalan
 * @goal workspace
 * @execute phase="process-resources"
 */
public class IdeaWorkspaceMojo
    extends AbstractIdeaMojo
{
    /**
     * Where a base workspaceTemplatePath file might be
     *
     * @parameter expression="${workspaceTemplatePath}" default-value="${idea.workspaceTemplatePath}"
     */
    private String workspaceTemplatePath;

    public void initParam(final MavenProject executedProject, final ArtifactFactory artifactFactory, final ArtifactRepository localRepo, final ArtifactResolver artifactResolver, final ArtifactMetadataSource artifactMetadataSource, final Log log, final boolean overwrite, final String workspaceTemplatePath)
    {
        super.initParam(executedProject, artifactFactory, localRepo, artifactResolver, artifactMetadataSource, log, overwrite);
        this.workspaceTemplatePath = workspaceTemplatePath;
    }


    /**
     * Create IDEA workspace (.iws) file.
     *
     * @throws org.apache.maven.plugin.MojoExecutionException
     *
     */
    public void execute()
        throws MojoExecutionException
    {
        try
        {
            doDependencyResolution( executedProject, localRepo );
        }
        catch ( Exception e )
        {
            throw new MojoExecutionException( "Unable to build project dependencies.", e );
        }

        rewriteWorkspace();
    }

    public void rewriteWorkspace()
        throws MojoExecutionException
    {
        File targetWorkspaceFile = new File( executedProject.getBasedir(), executedProject.getArtifactId() + ".iws" );

        if (!targetWorkspaceFile.exists())
        {
            getLog().info("Configuration for workspaceTemplatePath specified: " + workspaceTemplatePath);

            if (workspaceTemplatePath != null && workspaceTemplatePath.length() > 0 && new File(workspaceTemplatePath).isFile())
            {
                File workspaceTemplateFile = new File(workspaceTemplatePath);

                try
                {
                    getLog().info("Workspace template found at " + workspaceTemplateFile);

                    final String templateContents = FileUtils.fileRead(workspaceTemplateFile);
                    FileUtils.fileWrite(targetWorkspaceFile.getAbsolutePath(), templateContents);
                }
                catch (IOException e)
                {
                    throw new MojoExecutionException("Error reading IWS file: " + workspaceTemplateFile.getAbsolutePath(),
                                                     e);
                }

            }
            else
            {
                try
                {
                    Document document = readXmlDocument(targetWorkspaceFile, "workspace.xml");

                    Element module = document.getRootElement();

                    setProjectScmType(module);

                    writeXmlDocument(targetWorkspaceFile, document);
                }
                catch (DocumentException e)
                {
                    throw new MojoExecutionException("Error parsing existing IWS file: " + targetWorkspaceFile.getAbsolutePath(),
                                                     e);
                }
                catch (IOException e)
                {
                    throw new MojoExecutionException("Unable to create workspace file", e);
                }
            }
        }
        else
        {
            getLog().info("Existing workspace file found at " + targetWorkspaceFile.getAbsolutePath() + " ignoring.");
        }
    }

    /**
     * Sets the SCM type of the project
     */
    private void setProjectScmType( Element content )
    {
        String scmType = getScmType();

        if ( scmType != null )
        {
            Element component = findComponent( content, "VcsManagerConfiguration" );

            Element element = findElement( component, "option", "ACTIVE_VCS_NAME" );

            element.addAttribute( "value", scmType );
        }
    }

    /**
     * used to retrieve the SCM Type
     *
     * @return the Scm Type string used to connect to the SCM
     */
    protected String getScmType()
    {
        String scmType;

        if ( executedProject.getScm() == null )
        {
            return null;
        }
        scmType = getScmType( executedProject.getScm().getConnection() );

        if ( scmType != null )
        {
            return scmType;
        }
        scmType = getScmType( executedProject.getScm().getDeveloperConnection() );

        return scmType;
    }

    protected String getScmType( String connection )
    {
        String scmType;

        if ( connection != null )
        {
            if ( connection.length() > 0 )
            {
                int startIndex = connection.indexOf( ":" );

                int endIndex = connection.indexOf( "|", startIndex + 1 );

                if ( endIndex == -1 )
                {
                    endIndex = connection.indexOf( ":", startIndex + 1 );
                }

                if ( startIndex < endIndex )
                {
                    scmType = connection.substring( startIndex + 1, endIndex );

                    scmType = translateScmType( scmType );

                    return scmType;
                }
            }
        }
        return null;
    }

    /**
     * Translate the SCM type from the SCM connection URL to the format used by
     * IDEA as the value for ACTIVE_VCS_NAME.
     */
    protected String translateScmType( String scmType )
    {
        if ( "cvs".equals( scmType ) )
        {
            return "CVS";
        }
        else if ( "perforce".equals( scmType ) )
        {
            return "Perforce";
        }
        else if ( "starteam".equals( scmType ) )
        {
            return "StarTeam";
        }
        else if ( "vss".equals( scmType ) )
        {
            return "SourceSafe";
        }
        else
        {
            return scmType;
        }
    }

}
