/*
 * Decompiled with CFR 0.152.
 */
package org.apache.mahout.math.decomposer.lanczos;

import java.util.EnumMap;
import java.util.Map;
import org.apache.mahout.math.Matrix;
import org.apache.mahout.math.Vector;
import org.apache.mahout.math.VectorIterable;
import org.apache.mahout.math.decomposer.lanczos.LanczosState;
import org.apache.mahout.math.function.DoubleFunction;
import org.apache.mahout.math.function.PlusMult;
import org.apache.mahout.math.matrix.DoubleMatrix1D;
import org.apache.mahout.math.matrix.DoubleMatrix2D;
import org.apache.mahout.math.matrix.linalg.EigenvalueDecomposition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LanczosSolver {
    private static final Logger log = LoggerFactory.getLogger(LanczosSolver.class);
    public static final double SAFE_MAX = 1.0E150;
    private final Map<TimingSection, Long> startTimes = new EnumMap<TimingSection, Long>(TimingSection.class);
    private final Map<TimingSection, Long> times = new EnumMap<TimingSection, Long>(TimingSection.class);

    public void solve(LanczosState state, int desiredRank) {
        this.solve(state, desiredRank, false);
    }

    public void solve(LanczosState state, int desiredRank, boolean isSymmetric) {
        VectorIterable corpus = state.getCorpus();
        log.info("Finding {} singular vectors of matrix with {} rows, via Lanczos", (Object)desiredRank, (Object)corpus.numRows());
        int i = state.getIterationNumber();
        Vector currentVector = state.getBasisVector(i - 1);
        Vector previousVector = state.getBasisVector(i - 2);
        double beta = 0.0;
        Matrix triDiag = state.getDiagonalMatrix();
        while (i < desiredRank) {
            this.startTime(TimingSection.ITERATE);
            Vector nextVector = isSymmetric ? corpus.times(currentVector) : corpus.timesSquared(currentVector);
            log.info("{} passes through the corpus so far...", (Object)i);
            if (state.getScaleFactor() <= 0.0) {
                state.setScaleFactor(this.calculateScaleFactor(nextVector));
            }
            nextVector.assign(new Scale(1.0 / state.getScaleFactor()));
            if (previousVector != null) {
                nextVector.assign(previousVector, new PlusMult(-beta));
            }
            double alpha = currentVector.dot(nextVector);
            nextVector.assign(currentVector, new PlusMult(-alpha));
            this.endTime(TimingSection.ITERATE);
            this.startTime(TimingSection.ORTHOGANLIZE);
            this.orthoganalizeAgainstAllButLast(nextVector, state);
            this.endTime(TimingSection.ORTHOGANLIZE);
            beta = nextVector.norm(2.0);
            if (LanczosSolver.outOfRange(beta) || LanczosSolver.outOfRange(alpha)) {
                log.warn("Lanczos parameters out of range: alpha = {}, beta = {}.  Bailing out early!", (Object)alpha, (Object)beta);
                break;
            }
            nextVector.assign(new Scale(1.0 / beta));
            state.setBasisVector(i, nextVector);
            previousVector = currentVector;
            currentVector = nextVector;
            triDiag.set(i - 1, i - 1, alpha);
            if (i < desiredRank - 1) {
                triDiag.set(i - 1, i, beta);
                triDiag.set(i, i - 1, beta);
            }
            state.setIterationNumber(++i);
        }
        this.startTime(TimingSection.TRIDIAG_DECOMP);
        log.info("Lanczos iteration complete - now to diagonalize the tri-diagonal auxiliary matrix.");
        EigenvalueDecomposition decomp = new EigenvalueDecomposition(triDiag);
        DoubleMatrix2D eigenVects = decomp.getV();
        DoubleMatrix1D eigenVals = decomp.getRealEigenvalues();
        this.endTime(TimingSection.TRIDIAG_DECOMP);
        this.startTime(TimingSection.FINAL_EIGEN_CREATE);
        for (int row = 0; row < i; ++row) {
            Vector realEigen = null;
            DoubleMatrix1D ejCol = eigenVects.viewColumn(i - row - 1);
            int size = Math.min(ejCol.size(), state.getBasisSize());
            for (int j = 0; j < size; ++j) {
                double d = ejCol.get(j);
                Vector rowJ = state.getBasisVector(j);
                if (realEigen == null) {
                    realEigen = rowJ.like();
                }
                realEigen.assign(rowJ, new PlusMult(d));
            }
            realEigen = realEigen.normalize();
            state.setRightSingularVector(row, realEigen);
            double e = eigenVals.get(row) * state.getScaleFactor();
            if (!isSymmetric) {
                e = Math.sqrt(e);
            }
            log.info("Eigenvector {} found with eigenvalue {}", (Object)row, (Object)e);
            state.setSingularValue(row, e);
        }
        log.info("LanczosSolver finished.");
        this.endTime(TimingSection.FINAL_EIGEN_CREATE);
    }

    protected double calculateScaleFactor(Vector nextVector) {
        return nextVector.norm(2.0);
    }

    private static boolean outOfRange(double d) {
        return Double.isNaN(d) || d > 1.0E150 || -d > 1.0E150;
    }

    protected void orthoganalizeAgainstAllButLast(Vector nextVector, LanczosState state) {
        for (int i = 0; i < state.getIterationNumber(); ++i) {
            double alpha;
            Vector basisVector = state.getBasisVector(i);
            if (basisVector == null || (alpha = nextVector.dot(basisVector)) == 0.0) continue;
            nextVector.assign(basisVector, new PlusMult(-alpha));
        }
    }

    private void startTime(TimingSection section) {
        this.startTimes.put(section, System.nanoTime());
    }

    private void endTime(TimingSection section) {
        if (!this.times.containsKey((Object)section)) {
            this.times.put(section, 0L);
        }
        this.times.put(section, this.times.get((Object)section) + System.nanoTime() - this.startTimes.get((Object)section));
    }

    private static final class Scale
    implements DoubleFunction {
        private final double d;

        private Scale(double d) {
            this.d = d;
        }

        @Override
        public double apply(double arg1) {
            return arg1 * this.d;
        }
    }

    public static enum TimingSection {
        ITERATE,
        ORTHOGANLIZE,
        TRIDIAG_DECOMP,
        FINAL_EIGEN_CREATE;

    }
}

