/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache license, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the license for the specific language governing permissions and
 * limitations under the license.
 */
 package org.apache.logging.log4j.util;

import java.util.Locale;

/**
 * Helps convert English Strings to English Enum values.
 * <p>
 * Enum name arguments are converted internally to upper case with the {@linkplain Locale#ENGLISH ENGLISH} locale to
 * avoid problems on the Turkish locale. Do not use with Turkish enum values.
 * </p>
 */
public class EnglishEnums {

    /**
     * Returns the Result for the given string.
     * <p>
     * The {@code name} is converted internally to upper case with the {@linkplain Locale#ENGLISH ENGLISH} locale to
     * avoid problems on the Turkish locale. Do not use with Turkish enum values.
     * </p>
     * 
     * @param name
     *            The enum name, case-insensitive. If null, returns {@code defaultValue}
     * @return an enum value or null if {@code name} is null
     */
    public static <T extends Enum<T>> T valueOf(Class<T> enumType, String name) {
        return valueOf(enumType, name, null);
    }

    /**
     * Returns an enum value for the given string.
     * <p>
     * The {@code name} is converted internally to upper case with the {@linkplain Locale#ENGLISH ENGLISH} locale to
     * avoid problems on the Turkish locale. Do not use with Turkish enum values.
     * </p>
     * 
     * @param name
     *            The enum name, case-insensitive. If null, returns {@code defaultValue}
     * @param defaultValue
     *            the enum value to return if {@code name} is null
     * @return an enum value or {@code defaultValue} if {@code name} is null
     */
    public static <T extends Enum<T>> T valueOf(Class<T> enumType, String name, T defaultValue) {
        return name == null ? defaultValue : Enum.valueOf(enumType, name.toUpperCase(Locale.ENGLISH));
    }

}
