/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.util;

import java.nio.charset.StandardCharsets;
import java.util.BitSet;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.kudu.shaded.com.google.common.base.Preconditions;
import org.apache.kudu.shaded.com.sangupta.murmur.Murmur2;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

@InterfaceAudience.Public
@InterfaceStability.Unstable
@NotThreadSafe
public class BloomFilter {
    private final BitSet bitSet;
    private final int nHashes;
    private final byte[] byteBuffer;
    private final HashFunction hashFunction;
    private static final double DEFAULT_FP_RATE = 0.01;
    private static double kNaturalLog2 = 0.69314;

    private BloomFilter(BitSet bitSet, int nHashes, HashFunction hashFunction) {
        Preconditions.checkArgument(bitSet.size() >= 8, "Number of bits in bitset should be at least 8, but found %s.", bitSet.size());
        this.bitSet = bitSet;
        this.nHashes = nHashes;
        this.hashFunction = hashFunction;
        this.byteBuffer = new byte[8];
    }

    public static BloomFilter bySize(int nBytes) {
        return BloomFilter.bySizeAndFPRate(nBytes, 0.01);
    }

    public static BloomFilter bySizeAndFPRate(int nBytes, double fpRate) {
        return BloomFilter.bySizeAndFPRate(nBytes, fpRate, HashFunctions.MURMUR2);
    }

    public static BloomFilter bySizeAndFPRate(int nBytes, double fpRate, HashFunction hashFunction) {
        int nBits = nBytes * 8;
        int nHashes = BloomFilter.computeOptimalHashCount(nBits, BloomFilter.optimalExpectedCount(nBytes, fpRate));
        return new BloomFilter(new BitSet(nBits), nHashes, hashFunction);
    }

    public static BloomFilter byCount(int expectedCount) {
        return BloomFilter.byCountAndFPRate(expectedCount, 0.01);
    }

    public static BloomFilter byCountAndFPRate(int expectedCount, double fpRate) {
        return BloomFilter.byCountAndFPRate(expectedCount, fpRate, HashFunctions.MURMUR2);
    }

    public static BloomFilter byCountAndFPRate(int expectedCount, double fpRate, HashFunction hashFunction) {
        int nBytes = BloomFilter.optimalNumOfBytes(expectedCount, fpRate);
        int nBits = nBytes * 8;
        int nHashes = BloomFilter.computeOptimalHashCount(nBits, expectedCount);
        return new BloomFilter(new BitSet(nBits), nHashes, hashFunction);
    }

    public void put(byte[] data) {
        this.updateBitset(data, data.length);
    }

    public void put(boolean data) {
        this.byteBuffer[0] = (byte)(data ? 1 : 0);
        this.updateBitset(this.byteBuffer, 1);
    }

    public void put(byte data) {
        this.byteBuffer[0] = data;
        this.updateBitset(this.byteBuffer, 1);
    }

    public void put(short data) {
        this.byteBuffer[0] = (byte)(data >>> 0);
        this.byteBuffer[1] = (byte)(data >>> 8);
        this.updateBitset(this.byteBuffer, 2);
    }

    public void put(int data) {
        this.byteBuffer[0] = (byte)(data >>> 0);
        this.byteBuffer[1] = (byte)(data >>> 8);
        this.byteBuffer[2] = (byte)(data >>> 16);
        this.byteBuffer[3] = (byte)(data >>> 24);
        this.updateBitset(this.byteBuffer, 4);
    }

    public void put(long data) {
        this.byteBuffer[0] = (byte)(data >>> 0);
        this.byteBuffer[1] = (byte)(data >>> 8);
        this.byteBuffer[2] = (byte)(data >>> 16);
        this.byteBuffer[3] = (byte)(data >>> 24);
        this.byteBuffer[4] = (byte)(data >>> 32);
        this.byteBuffer[5] = (byte)(data >>> 40);
        this.byteBuffer[6] = (byte)(data >>> 48);
        this.byteBuffer[7] = (byte)(data >>> 56);
        this.updateBitset(this.byteBuffer, 8);
    }

    public void put(float data) {
        this.put(Float.floatToIntBits(data));
    }

    public void put(double data) {
        this.put(Double.doubleToLongBits(data));
    }

    public void put(String data) {
        this.put(data.getBytes(StandardCharsets.UTF_8));
    }

    public byte[] getBitSet() {
        return this.bitSet.toByteArray();
    }

    public int getNHashes() {
        return this.nHashes;
    }

    public String getHashFunctionName() {
        return this.hashFunction.toString();
    }

    private void updateBitset(byte[] byteBuffer, int length) {
        Preconditions.checkArgument(byteBuffer.length >= length);
        long h = Murmur2.hash64(byteBuffer, length, 0L);
        long h1 = 0xFFFFFFFFL & h;
        long h2 = h >>> 32;
        long tmp = h1;
        for (int i = 0; i < this.nHashes; ++i) {
            long bitPos = tmp % (long)this.bitSet.size();
            this.bitSet.set((int)bitPos);
            tmp += h2;
        }
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(byte[] data) {
        return this.checkIfContains(data);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(boolean data) {
        byte[] byteBuffer = new byte[]{data ? (byte)1 : 0};
        return this.checkIfContains(byteBuffer);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(byte data) {
        byte[] byteBuffer = new byte[]{data};
        return this.checkIfContains(byteBuffer);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(short data) {
        byte[] byteBuffer = new byte[]{(byte)(data >>> 0), (byte)(data >>> 8)};
        return this.checkIfContains(byteBuffer);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(int data) {
        byte[] byteBuffer = new byte[]{(byte)(data >>> 0), (byte)(data >>> 8), (byte)(data >>> 16), (byte)(data >>> 24)};
        return this.checkIfContains(byteBuffer);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(long data) {
        byte[] byteBuffer = new byte[]{(byte)(data >>> 0), (byte)(data >>> 8), (byte)(data >>> 16), (byte)(data >>> 24), (byte)(data >>> 32), (byte)(data >>> 40), (byte)(data >>> 48), (byte)(data >>> 56)};
        return this.checkIfContains(byteBuffer);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(float data) {
        return this.mayContain(Float.floatToIntBits(data));
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(double data) {
        return this.mayContain(Double.doubleToLongBits(data));
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    public boolean mayContain(String data) {
        return this.mayContain(data.getBytes(StandardCharsets.UTF_8));
    }

    private boolean checkIfContains(byte[] bytes) {
        long h = Murmur2.hash64(bytes, bytes.length, 0L);
        long h1 = 0xFFFFFFFFL & h;
        long h2 = h >>> 32;
        long tmp = h1;
        for (int remHashes = this.nHashes; remHashes != 0; --remHashes) {
            long bitPos = tmp % (long)this.bitSet.size();
            if (!this.bitSet.get((int)bitPos)) {
                return false;
            }
            tmp += h2;
        }
        return true;
    }

    private static int optimalNumOfBytes(int expectedCount, double fpRate) {
        if (fpRate == 0.0) {
            fpRate = Double.MIN_VALUE;
        }
        return (int)Math.ceil((double)(-expectedCount) * Math.log(fpRate) / (Math.log(2.0) * Math.log(2.0) * 8.0));
    }

    private static int optimalExpectedCount(int nBytes, double fpRate) {
        int nBits = nBytes * 8;
        return (int)Math.ceil((double)(-nBits) * kNaturalLog2 * kNaturalLog2 / Math.log(fpRate));
    }

    private static int computeOptimalHashCount(int nBits, int elems) {
        int nHashes = (int)((double)nBits * kNaturalLog2 / (double)elems);
        if (nHashes < 1) {
            nHashes = 1;
        }
        return nHashes;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("BloomFilter(nBits=");
        sb.append(this.bitSet.size());
        sb.append(", nHashes=");
        sb.append(this.nHashes);
        sb.append(", hashing=");
        sb.append(this.hashFunction);
        sb.append(")");
        return sb.toString();
    }

    public static enum HashFunctions implements HashFunction
    {
        MURMUR2{

            @Override
            public long hash(byte[] data, int length, long seed) {
                return Murmur2.hash(data, length, seed);
            }

            public String toString() {
                return "Murmur2";
            }
        };

    }

    private static interface HashFunction {
        public long hash(byte[] var1, int var2, long var3);
    }
}

