/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kudu.client;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.TreeSet;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.ColumnTypeAttributes;
import org.apache.kudu.Common;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.Bytes;
import org.apache.kudu.shaded.com.google.common.base.Joiner;
import org.apache.kudu.shaded.com.google.common.base.Objects;
import org.apache.kudu.shaded.com.google.common.base.Preconditions;
import org.apache.kudu.shaded.com.google.common.primitives.UnsignedBytes;
import org.apache.kudu.shaded.com.google.protobuf.ByteString;
import org.apache.kudu.util.DateUtil;
import org.apache.kudu.util.DecimalUtil;
import org.apache.kudu.util.TimestampUtil;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.yetus.audience.InterfaceStability;

@InterfaceAudience.Public
@InterfaceStability.Evolving
public class KuduPredicate {
    private final PredicateType type;
    private final ColumnSchema column;
    private final byte[] lower;
    private final byte[] upper;
    private final byte[][] inListValues;

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, boolean value) {
        KuduPredicate.checkColumn(column, Type.BOOL);
        switch (op) {
            case GREATER: {
                if (value) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.EQUALITY, column, Bytes.fromBoolean(true), null);
            }
            case GREATER_EQUAL: {
                if (value) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, Bytes.fromBoolean(true), null);
                }
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, Bytes.fromBoolean(value), null);
            }
            case LESS: {
                if (value) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, Bytes.fromBoolean(false), null);
                }
                return KuduPredicate.none(column);
            }
            case LESS_EQUAL: {
                if (value) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                return new KuduPredicate(PredicateType.EQUALITY, column, Bytes.fromBoolean(false), null);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, long value) {
        byte[] bytes;
        KuduPredicate.checkColumn(column, Type.INT8, Type.INT16, Type.INT32, Type.INT64, Type.UNIXTIME_MICROS, Type.DATE);
        long minValue = KuduPredicate.minIntValue(column.getType());
        long maxValue = KuduPredicate.maxIntValue(column.getType());
        Preconditions.checkArgument(value <= maxValue && value >= minValue, "integer value out of range for %s column: %s", (Object)column.getType(), value);
        if (op == ComparisonOp.LESS_EQUAL) {
            if (value == maxValue) {
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            ++value;
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            if (value == maxValue) {
                return KuduPredicate.none(column);
            }
            ++value;
            op = ComparisonOp.GREATER_EQUAL;
        }
        switch (column.getType()) {
            case INT8: {
                bytes = new byte[]{(byte)value};
                break;
            }
            case INT16: {
                bytes = Bytes.fromShort((short)value);
                break;
            }
            case DATE: 
            case INT32: {
                bytes = Bytes.fromInt((int)value);
                break;
            }
            case INT64: 
            case UNIXTIME_MICROS: {
                bytes = Bytes.fromLong(value);
                break;
            }
            default: {
                throw new RuntimeException("already checked");
            }
        }
        switch (op) {
            case GREATER_EQUAL: {
                if (value == minValue) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                if (value == maxValue) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
                }
                return new KuduPredicate(PredicateType.RANGE, column, bytes, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
            }
            case LESS: {
                if (value == minValue) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, bytes);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, BigDecimal value) {
        KuduPredicate.checkColumn(column, Type.DECIMAL);
        ColumnTypeAttributes typeAttributes = column.getTypeAttributes();
        int precision = typeAttributes.getPrecision();
        int scale = typeAttributes.getScale();
        value = DecimalUtil.coerce(value, precision, scale);
        BigDecimal minValue = DecimalUtil.minValue(precision, scale);
        BigDecimal maxValue = DecimalUtil.maxValue(precision, scale);
        Preconditions.checkArgument(value.compareTo(maxValue) <= 0 && value.compareTo(minValue) >= 0, "Decimal value out of range for %s column: %s", (Object)column.getType(), (Object)value);
        BigDecimal smallestValue = DecimalUtil.smallestValue(scale);
        if (op == ComparisonOp.LESS_EQUAL) {
            if (value.equals(maxValue)) {
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            value = value.add(smallestValue);
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            if (value.equals(maxValue)) {
                return KuduPredicate.none(column);
            }
            value = value.add(smallestValue);
            op = ComparisonOp.GREATER_EQUAL;
        }
        byte[] bytes = Bytes.fromBigDecimal(value, precision);
        switch (op) {
            case GREATER_EQUAL: {
                if (value.equals(minValue)) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                if (value.equals(maxValue)) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
                }
                return new KuduPredicate(PredicateType.RANGE, column, bytes, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
            }
            case LESS: {
                if (value.equals(minValue)) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, bytes);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, Timestamp value) {
        KuduPredicate.checkColumn(column, Type.UNIXTIME_MICROS);
        long micros = TimestampUtil.timestampToMicros(value);
        return KuduPredicate.newComparisonPredicate(column, op, micros);
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, Date value) {
        KuduPredicate.checkColumn(column, Type.DATE);
        int days = DateUtil.sqlDateToEpochDays(value);
        return KuduPredicate.newComparisonPredicate(column, op, days);
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, float value) {
        KuduPredicate.checkColumn(column, Type.FLOAT);
        if (op == ComparisonOp.LESS_EQUAL) {
            if (value == Float.POSITIVE_INFINITY) {
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            value = Math.nextAfter(value, Double.POSITIVE_INFINITY);
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            if (value == Float.POSITIVE_INFINITY) {
                return KuduPredicate.none(column);
            }
            value = Math.nextAfter(value, Double.POSITIVE_INFINITY);
            op = ComparisonOp.GREATER_EQUAL;
        }
        byte[] bytes = Bytes.fromFloat(value);
        switch (op) {
            case GREATER_EQUAL: {
                if (value == Float.NEGATIVE_INFINITY) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                if (value == Float.POSITIVE_INFINITY) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
                }
                return new KuduPredicate(PredicateType.RANGE, column, bytes, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
            }
            case LESS: {
                if (value == Float.NEGATIVE_INFINITY) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, bytes);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, double value) {
        KuduPredicate.checkColumn(column, Type.DOUBLE);
        if (op == ComparisonOp.LESS_EQUAL) {
            if (value == Double.POSITIVE_INFINITY) {
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            value = Math.nextAfter(value, Double.POSITIVE_INFINITY);
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            if (value == Double.POSITIVE_INFINITY) {
                return KuduPredicate.none(column);
            }
            value = Math.nextAfter(value, Double.POSITIVE_INFINITY);
            op = ComparisonOp.GREATER_EQUAL;
        }
        byte[] bytes = Bytes.fromDouble(value);
        switch (op) {
            case GREATER_EQUAL: {
                if (value == Double.NEGATIVE_INFINITY) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                if (value == Double.POSITIVE_INFINITY) {
                    return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
                }
                return new KuduPredicate(PredicateType.RANGE, column, bytes, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
            }
            case LESS: {
                if (value == Double.NEGATIVE_INFINITY) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, bytes);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, String value) {
        KuduPredicate.checkColumn(column, Type.STRING, Type.VARCHAR);
        byte[] bytes = Bytes.fromString(value);
        if (op == ComparisonOp.LESS_EQUAL) {
            bytes = Arrays.copyOf(bytes, bytes.length + 1);
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            bytes = Arrays.copyOf(bytes, bytes.length + 1);
            op = ComparisonOp.GREATER_EQUAL;
        }
        switch (op) {
            case GREATER_EQUAL: {
                if (bytes.length == 0) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, bytes, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, bytes, null);
            }
            case LESS: {
                if (bytes.length == 0) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, bytes);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, byte[] value) {
        KuduPredicate.checkColumn(column, Type.BINARY);
        if (op == ComparisonOp.LESS_EQUAL) {
            value = Arrays.copyOf(value, value.length + 1);
            op = ComparisonOp.LESS;
        } else if (op == ComparisonOp.GREATER) {
            value = Arrays.copyOf(value, value.length + 1);
            op = ComparisonOp.GREATER_EQUAL;
        }
        switch (op) {
            case GREATER_EQUAL: {
                if (value.length == 0) {
                    return KuduPredicate.newIsNotNullPredicate(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, value, null);
            }
            case EQUAL: {
                return new KuduPredicate(PredicateType.EQUALITY, column, value, null);
            }
            case LESS: {
                if (value.length == 0) {
                    return KuduPredicate.none(column);
                }
                return new KuduPredicate(PredicateType.RANGE, column, null, value);
            }
        }
        throw new RuntimeException("unknown comparison op");
    }

    public static KuduPredicate newComparisonPredicate(ColumnSchema column, ComparisonOp op, Object value) {
        if (value instanceof Boolean) {
            return KuduPredicate.newComparisonPredicate(column, op, (Boolean)value);
        }
        if (value instanceof Byte) {
            return KuduPredicate.newComparisonPredicate(column, op, ((Byte)value).byteValue());
        }
        if (value instanceof Short) {
            return KuduPredicate.newComparisonPredicate(column, op, ((Short)value).shortValue());
        }
        if (value instanceof Integer) {
            return KuduPredicate.newComparisonPredicate(column, op, ((Integer)value).intValue());
        }
        if (value instanceof Long) {
            return KuduPredicate.newComparisonPredicate(column, op, (Long)value);
        }
        if (value instanceof Timestamp) {
            return KuduPredicate.newComparisonPredicate(column, op, (Timestamp)value);
        }
        if (value instanceof Float) {
            return KuduPredicate.newComparisonPredicate(column, op, ((Float)value).floatValue());
        }
        if (value instanceof Double) {
            return KuduPredicate.newComparisonPredicate(column, op, (Double)value);
        }
        if (value instanceof BigDecimal) {
            return KuduPredicate.newComparisonPredicate(column, op, (BigDecimal)value);
        }
        if (value instanceof String) {
            return KuduPredicate.newComparisonPredicate(column, op, (String)value);
        }
        if (value instanceof byte[]) {
            return KuduPredicate.newComparisonPredicate(column, op, (byte[])value);
        }
        if (value instanceof Date) {
            return KuduPredicate.newComparisonPredicate(column, op, (Date)value);
        }
        throw new IllegalArgumentException(String.format("illegal type for %s predicate: %s", new Object[]{op, value.getClass().getName()}));
    }

    public static <T> KuduPredicate newInListPredicate(final ColumnSchema column, List<T> values) {
        if (values.isEmpty()) {
            return KuduPredicate.none(column);
        }
        T t = values.get(0);
        TreeSet<byte[]> vals = new TreeSet<byte[]>(new Comparator<byte[]>(){

            @Override
            public int compare(byte[] a, byte[] b) {
                return KuduPredicate.compare(column, a, b);
            }
        });
        if (t instanceof Boolean) {
            KuduPredicate.checkColumn(column, Type.BOOL);
            for (T value : values) {
                vals.add(Bytes.fromBoolean((Boolean)value));
            }
        } else if (t instanceof Byte) {
            KuduPredicate.checkColumn(column, Type.INT8);
            for (T value : values) {
                vals.add(new byte[]{(Byte)value});
            }
        } else if (t instanceof Short) {
            KuduPredicate.checkColumn(column, Type.INT16);
            for (T value : values) {
                vals.add(Bytes.fromShort((Short)value));
            }
        } else if (t instanceof Integer) {
            KuduPredicate.checkColumn(column, Type.INT32, Type.DATE);
            for (T value : values) {
                vals.add(Bytes.fromInt((Integer)value));
            }
        } else if (t instanceof Long) {
            KuduPredicate.checkColumn(column, Type.INT64, Type.UNIXTIME_MICROS);
            for (T value : values) {
                vals.add(Bytes.fromLong((Long)value));
            }
        } else if (t instanceof Float) {
            KuduPredicate.checkColumn(column, Type.FLOAT);
            for (T value : values) {
                vals.add(Bytes.fromFloat(((Float)value).floatValue()));
            }
        } else if (t instanceof Double) {
            KuduPredicate.checkColumn(column, Type.DOUBLE);
            for (T value : values) {
                vals.add(Bytes.fromDouble((Double)value));
            }
        } else if (t instanceof BigDecimal) {
            KuduPredicate.checkColumn(column, Type.DECIMAL);
            for (T value : values) {
                vals.add(Bytes.fromBigDecimal((BigDecimal)value, column.getTypeAttributes().getPrecision()));
            }
        } else if (t instanceof String) {
            KuduPredicate.checkColumn(column, Type.STRING, Type.VARCHAR);
            for (T value : values) {
                vals.add(Bytes.fromString((String)value));
            }
        } else if (t instanceof byte[]) {
            KuduPredicate.checkColumn(column, Type.BINARY);
            for (T value : values) {
                vals.add((byte[])value);
            }
        } else if (t instanceof Date) {
            KuduPredicate.checkColumn(column, Type.DATE);
            for (T value : values) {
                vals.add(Bytes.fromInt(DateUtil.sqlDateToEpochDays((Date)value)));
            }
        } else {
            throw new IllegalArgumentException(String.format("illegal type for IN list values: %s", t.getClass().getName()));
        }
        return KuduPredicate.buildInList(column, vals);
    }

    public static KuduPredicate newIsNotNullPredicate(ColumnSchema column) {
        return new KuduPredicate(PredicateType.IS_NOT_NULL, column, null, null);
    }

    public static KuduPredicate newIsNullPredicate(ColumnSchema column) {
        if (!column.isNullable()) {
            return KuduPredicate.none(column);
        }
        return new KuduPredicate(PredicateType.IS_NULL, column, null, null);
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    KuduPredicate(PredicateType type, ColumnSchema column, byte[] lower, byte[] upper) {
        this.type = type;
        this.column = column;
        this.lower = lower;
        this.upper = upper;
        this.inListValues = null;
    }

    private KuduPredicate(ColumnSchema column, byte[][] inListValues) {
        this.column = column;
        this.type = PredicateType.IN_LIST;
        this.lower = null;
        this.upper = null;
        this.inListValues = inListValues;
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    static KuduPredicate none(ColumnSchema column) {
        return new KuduPredicate(PredicateType.NONE, column, null, null);
    }

    PredicateType getType() {
        return this.type;
    }

    KuduPredicate merge(KuduPredicate other) {
        Preconditions.checkArgument(this.column.equals(other.column), "predicates from different columns may not be merged");
        if (other.type == PredicateType.NONE) {
            return other;
        }
        if (other.type == PredicateType.IS_NOT_NULL) {
            return this.type == PredicateType.IS_NULL ? KuduPredicate.none(this.column) : this;
        }
        if (other.type == PredicateType.IS_NULL) {
            return this.type == PredicateType.IS_NULL ? this : KuduPredicate.none(this.column);
        }
        switch (this.type) {
            case NONE: {
                return this;
            }
            case IS_NOT_NULL: {
                return other;
            }
            case IS_NULL: {
                return KuduPredicate.none(this.column);
            }
            case EQUALITY: {
                if (other.type == PredicateType.EQUALITY) {
                    if (KuduPredicate.compare(this.column, this.lower, other.lower) != 0) {
                        return KuduPredicate.none(this.column);
                    }
                    return this;
                }
                if (other.type == PredicateType.RANGE) {
                    if (other.rangeContains(this.lower)) {
                        return this;
                    }
                    return KuduPredicate.none(this.column);
                }
                Preconditions.checkState(other.type == PredicateType.IN_LIST);
                return other.merge(this);
            }
            case RANGE: {
                byte[] newUpper;
                if (other.type == PredicateType.EQUALITY || other.type == PredicateType.IN_LIST) {
                    return other.merge(this);
                }
                Preconditions.checkState(other.type == PredicateType.RANGE);
                byte[] newLower = other.lower == null || this.lower != null && KuduPredicate.compare(this.column, this.lower, other.lower) >= 0 ? this.lower : other.lower;
                byte[] byArray = newUpper = other.upper == null || this.upper != null && KuduPredicate.compare(this.column, this.upper, other.upper) <= 0 ? this.upper : other.upper;
                if (newLower != null && newUpper != null && KuduPredicate.compare(this.column, newLower, newUpper) >= 0) {
                    return KuduPredicate.none(this.column);
                }
                if (newLower != null && newUpper != null && this.areConsecutive(newLower, newUpper)) {
                    return new KuduPredicate(PredicateType.EQUALITY, this.column, newLower, null);
                }
                return new KuduPredicate(PredicateType.RANGE, this.column, newLower, newUpper);
            }
            case IN_LIST: {
                if (other.type == PredicateType.EQUALITY) {
                    if (this.inListContains(other.lower)) {
                        return other;
                    }
                    return KuduPredicate.none(this.column);
                }
                if (other.type == PredicateType.RANGE) {
                    ArrayList<byte[]> values = new ArrayList<byte[]>();
                    for (byte[] value : this.inListValues) {
                        if (!other.rangeContains(value)) continue;
                        values.add(value);
                    }
                    return KuduPredicate.buildInList(this.column, values);
                }
                Preconditions.checkState(other.type == PredicateType.IN_LIST);
                ArrayList<byte[]> values = new ArrayList<byte[]>();
                for (byte[] value : this.inListValues) {
                    if (!other.inListContains(value)) continue;
                    values.add(value);
                }
                return KuduPredicate.buildInList(this.column, values);
            }
        }
        throw new IllegalStateException(String.format("unknown predicate type %s", this));
    }

    private static KuduPredicate buildInList(ColumnSchema column, Collection<byte[]> values) {
        if (column.getType().getDataType(column.getTypeAttributes()) == Common.DataType.BOOL && values.size() > 1) {
            return KuduPredicate.newIsNotNullPredicate(column);
        }
        switch (values.size()) {
            case 0: {
                return KuduPredicate.none(column);
            }
            case 1: {
                return new KuduPredicate(PredicateType.EQUALITY, column, values.iterator().next(), null);
            }
        }
        return new KuduPredicate(column, (byte[][])values.toArray((T[])new byte[values.size()][]));
    }

    boolean inListContains(byte[] value) {
        Comparator<byte[]> comparator = new Comparator<byte[]>(){

            @Override
            public int compare(byte[] a, byte[] b) {
                return KuduPredicate.compare(KuduPredicate.this.column, a, b);
            }
        };
        return Arrays.binarySearch(this.inListValues, value, comparator) >= 0;
    }

    boolean rangeContains(byte[] value) {
        return !(this.lower != null && KuduPredicate.compare(this.column, value, this.lower) < 0 || this.upper != null && KuduPredicate.compare(this.column, value, this.upper) >= 0);
    }

    ColumnSchema getColumn() {
        return this.column;
    }

    @InterfaceAudience.LimitedPrivate(value={"kudu-mapreduce"})
    public static byte[] serialize(List<KuduPredicate> predicates) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        for (KuduPredicate predicate : predicates) {
            Common.ColumnPredicatePB message = predicate.toPB();
            message.writeDelimitedTo(baos);
        }
        return baos.toByteArray();
    }

    @InterfaceAudience.LimitedPrivate(value={"kudu-mapreduce"})
    public static List<KuduPredicate> deserialize(Schema schema, byte[] bytes) throws IOException {
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        ArrayList<KuduPredicate> predicates = new ArrayList<KuduPredicate>();
        while (bais.available() > 0) {
            Common.ColumnPredicatePB message = Common.ColumnPredicatePB.parseDelimitedFrom(bais);
            predicates.add(KuduPredicate.fromPB(schema, message));
        }
        return predicates;
    }

    @InterfaceAudience.Private
    public Common.ColumnPredicatePB toPB() {
        Common.ColumnPredicatePB.Builder builder = Common.ColumnPredicatePB.newBuilder();
        builder.setColumn(this.column.getName());
        switch (this.type) {
            case EQUALITY: {
                builder.getEqualityBuilder().setValue(ByteString.copyFrom(this.lower));
                break;
            }
            case RANGE: {
                Common.ColumnPredicatePB.Range.Builder b = builder.getRangeBuilder();
                if (this.lower != null) {
                    b.setLower(ByteString.copyFrom(this.lower));
                }
                if (this.upper == null) break;
                b.setUpper(ByteString.copyFrom(this.upper));
                break;
            }
            case IS_NOT_NULL: {
                builder.setIsNotNull(builder.getIsNotNullBuilder());
                break;
            }
            case IS_NULL: {
                builder.setIsNull(builder.getIsNullBuilder());
                break;
            }
            case IN_LIST: {
                Common.ColumnPredicatePB.InList.Builder inListBuilder = builder.getInListBuilder();
                for (byte[] value : this.inListValues) {
                    inListBuilder.addValues(ByteString.copyFrom(value));
                }
                break;
            }
            case NONE: {
                throw new IllegalStateException("can not convert None predicate to protobuf message");
            }
            default: {
                throw new IllegalArgumentException(String.format("unknown predicate type: %s", new Object[]{this.type}));
            }
        }
        return builder.build();
    }

    @InterfaceAudience.Private
    public static KuduPredicate fromPB(Schema schema, Common.ColumnPredicatePB pb) {
        final ColumnSchema column = schema.getColumn(pb.getColumn());
        switch (pb.getPredicateCase()) {
            case EQUALITY: {
                return new KuduPredicate(PredicateType.EQUALITY, column, pb.getEquality().getValue().toByteArray(), null);
            }
            case RANGE: {
                Common.ColumnPredicatePB.Range range = pb.getRange();
                return new KuduPredicate(PredicateType.RANGE, column, range.hasLower() ? range.getLower().toByteArray() : null, range.hasUpper() ? range.getUpper().toByteArray() : null);
            }
            case IS_NOT_NULL: {
                return KuduPredicate.newIsNotNullPredicate(column);
            }
            case IS_NULL: {
                return KuduPredicate.newIsNullPredicate(column);
            }
            case IN_LIST: {
                Common.ColumnPredicatePB.InList inList = pb.getInList();
                TreeSet<byte[]> values = new TreeSet<byte[]>(new Comparator<byte[]>(){

                    @Override
                    public int compare(byte[] a, byte[] b) {
                        return KuduPredicate.compare(column, a, b);
                    }
                });
                for (ByteString value : inList.getValuesList()) {
                    values.add(value.toByteArray());
                }
                return KuduPredicate.buildInList(column, values);
            }
        }
        throw new IllegalArgumentException("unknown predicate type");
    }

    private static int compare(ColumnSchema column, byte[] a, byte[] b) {
        switch (column.getType().getDataType(column.getTypeAttributes())) {
            case BOOL: {
                return Boolean.compare(Bytes.getBoolean(a), Bytes.getBoolean(b));
            }
            case INT8: {
                return Byte.compare(Bytes.getByte(a), Bytes.getByte(b));
            }
            case INT16: {
                return Short.compare(Bytes.getShort(a), Bytes.getShort(b));
            }
            case INT32: 
            case DATE: 
            case DECIMAL32: {
                return Integer.compare(Bytes.getInt(a), Bytes.getInt(b));
            }
            case INT64: 
            case UNIXTIME_MICROS: 
            case DECIMAL64: {
                return Long.compare(Bytes.getLong(a), Bytes.getLong(b));
            }
            case FLOAT: {
                return Float.compare(Bytes.getFloat(a), Bytes.getFloat(b));
            }
            case DOUBLE: {
                return Double.compare(Bytes.getDouble(a), Bytes.getDouble(b));
            }
            case STRING: 
            case VARCHAR: 
            case BINARY: {
                return UnsignedBytes.lexicographicalComparator().compare(a, b);
            }
            case DECIMAL128: {
                return Bytes.getBigInteger(a).compareTo(Bytes.getBigInteger(b));
            }
        }
        throw new IllegalStateException(String.format("unknown column type %s", new Object[]{column.getType()}));
    }

    private boolean areConsecutive(byte[] a, byte[] b) {
        switch (this.column.getType().getDataType(this.column.getTypeAttributes())) {
            case BOOL: {
                return false;
            }
            case INT8: {
                byte m = Bytes.getByte(a);
                byte n = Bytes.getByte(b);
                return m < n && m + 1 == n;
            }
            case INT16: {
                short m = Bytes.getShort(a);
                short n = Bytes.getShort(b);
                return m < n && m + 1 == n;
            }
            case INT32: 
            case DATE: 
            case DECIMAL32: {
                int m = Bytes.getInt(a);
                int n = Bytes.getInt(b);
                return m < n && m + 1 == n;
            }
            case INT64: 
            case UNIXTIME_MICROS: 
            case DECIMAL64: {
                long m = Bytes.getLong(a);
                long n = Bytes.getLong(b);
                return m < n && m + 1L == n;
            }
            case FLOAT: {
                float m = Bytes.getFloat(a);
                float n = Bytes.getFloat(b);
                return m < n && Math.nextAfter(m, Double.POSITIVE_INFINITY) == n;
            }
            case DOUBLE: {
                double m = Bytes.getDouble(a);
                double n = Bytes.getDouble(b);
                return m < n && Math.nextAfter(m, Double.POSITIVE_INFINITY) == n;
            }
            case STRING: 
            case VARCHAR: 
            case BINARY: {
                if (a.length + 1 != b.length || b[a.length] != 0) {
                    return false;
                }
                for (int i = 0; i < a.length; ++i) {
                    if (a[i] == b[i]) continue;
                    return false;
                }
                return true;
            }
            case DECIMAL128: {
                BigInteger m = Bytes.getBigInteger(a);
                BigInteger n = Bytes.getBigInteger(b);
                return m.compareTo(n) < 0 && m.add(BigInteger.ONE).equals(n);
            }
        }
        throw new IllegalStateException(String.format("unknown column type %s", new Object[]{this.column.getType()}));
    }

    byte[] getLower() {
        return this.lower;
    }

    byte[] getUpper() {
        return this.upper;
    }

    byte[][] getInListValues() {
        return this.inListValues;
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    static long maxIntValue(Type type) {
        switch (type) {
            case INT8: {
                return 127L;
            }
            case INT16: {
                return 32767L;
            }
            case INT32: {
                return Integer.MAX_VALUE;
            }
            case INT64: 
            case UNIXTIME_MICROS: {
                return Long.MAX_VALUE;
            }
            case DATE: {
                return DateUtil.MAX_DATE_VALUE;
            }
        }
        throw new IllegalArgumentException("type must be an integer type");
    }

    @InterfaceAudience.LimitedPrivate(value={"Test"})
    static long minIntValue(Type type) {
        switch (type) {
            case INT8: {
                return -128L;
            }
            case INT16: {
                return -32768L;
            }
            case INT32: {
                return Integer.MIN_VALUE;
            }
            case INT64: 
            case UNIXTIME_MICROS: {
                return Long.MIN_VALUE;
            }
            case DATE: {
                return DateUtil.MIN_DATE_VALUE;
            }
        }
        throw new IllegalArgumentException("type must be an integer type");
    }

    private static void checkColumn(ColumnSchema column, Type ... passedTypes) {
        for (Type type : passedTypes) {
            if (!column.getType().equals((Object)type)) continue;
            return;
        }
        throw new IllegalArgumentException(String.format("%s's type isn't %s, it's %s", column.getName(), Arrays.toString((Object[])passedTypes), column.getType().getName()));
    }

    private String valueToString(byte[] value) {
        switch (this.column.getType().getDataType(this.column.getTypeAttributes())) {
            case BOOL: {
                return Boolean.toString(Bytes.getBoolean(value));
            }
            case INT8: {
                return Byte.toString(Bytes.getByte(value));
            }
            case INT16: {
                return Short.toString(Bytes.getShort(value));
            }
            case INT32: {
                return Integer.toString(Bytes.getInt(value));
            }
            case INT64: {
                return Long.toString(Bytes.getLong(value));
            }
            case DATE: {
                return DateUtil.epochDaysToDateString(Bytes.getInt(value));
            }
            case UNIXTIME_MICROS: {
                return TimestampUtil.timestampToString(Bytes.getLong(value));
            }
            case FLOAT: {
                return Float.toString(Bytes.getFloat(value));
            }
            case DOUBLE: {
                return Double.toString(Bytes.getDouble(value));
            }
            case STRING: 
            case VARCHAR: {
                String v = Bytes.getString(value);
                StringBuilder sb = new StringBuilder(2 + v.length());
                sb.append('\"');
                sb.append(v);
                sb.append('\"');
                return sb.toString();
            }
            case BINARY: {
                return Bytes.hex(value);
            }
            case DECIMAL32: 
            case DECIMAL64: 
            case DECIMAL128: {
                ColumnTypeAttributes typeAttributes = this.column.getTypeAttributes();
                return Bytes.getDecimal(value, typeAttributes.getPrecision(), typeAttributes.getScale()).toString();
            }
        }
        throw new IllegalStateException(String.format("unknown column type %s", new Object[]{this.column.getType()}));
    }

    public String toString() {
        switch (this.type) {
            case EQUALITY: {
                return String.format("`%s` = %s", this.column.getName(), this.valueToString(this.lower));
            }
            case RANGE: {
                if (this.lower == null) {
                    return String.format("`%s` < %s", this.column.getName(), this.valueToString(this.upper));
                }
                if (this.upper == null) {
                    return String.format("`%s` >= %s", this.column.getName(), this.valueToString(this.lower));
                }
                return String.format("`%s` >= %s AND `%s` < %s", this.column.getName(), this.valueToString(this.lower), this.column.getName(), this.valueToString(this.upper));
            }
            case IN_LIST: {
                ArrayList<String> strings = new ArrayList<String>(this.inListValues.length);
                for (byte[] value : this.inListValues) {
                    strings.add(this.valueToString(value));
                }
                return String.format("`%s` IN (%s)", this.column.getName(), Joiner.on(", ").join(strings));
            }
            case IS_NOT_NULL: {
                return String.format("`%s` IS NOT NULL", this.column.getName());
            }
            case IS_NULL: {
                return String.format("`%s` IS NULL", this.column.getName());
            }
            case NONE: {
                return String.format("`%s` NONE", this.column.getName());
            }
        }
        throw new IllegalArgumentException(String.format("unknown predicate type %s", new Object[]{this.type}));
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof KuduPredicate)) {
            return false;
        }
        KuduPredicate that = (KuduPredicate)o;
        return this.type == that.type && this.column.equals(that.column) && Arrays.equals(this.lower, that.lower) && Arrays.equals(this.upper, that.upper) && Arrays.deepEquals((Object[])this.inListValues, (Object[])that.inListValues);
    }

    public int hashCode() {
        return Objects.hashCode(new Object[]{this.type, this.column, Arrays.hashCode(this.lower), Arrays.hashCode(this.upper), Arrays.deepHashCode((Object[])this.inListValues)});
    }

    @InterfaceAudience.Public
    @InterfaceStability.Evolving
    public static enum ComparisonOp {
        GREATER,
        GREATER_EQUAL,
        EQUAL,
        LESS,
        LESS_EQUAL;

    }

    @InterfaceAudience.Private
    static enum PredicateType {
        NONE,
        EQUALITY,
        RANGE,
        IS_NOT_NULL,
        IS_NULL,
        IN_LIST;

    }
}

