/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.integration;

import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.serialization.IntegerDeserializer;
import org.apache.kafka.common.serialization.IntegerSerializer;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.common.utils.Bytes;
import org.apache.kafka.common.utils.MockTime;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.Topology;
import org.apache.kafka.streams.integration.utils.EmbeddedKafkaCluster;
import org.apache.kafka.streams.integration.utils.IntegrationTestUtils;
import org.apache.kafka.streams.kstream.Consumed;
import org.apache.kafka.streams.kstream.KStream;
import org.apache.kafka.streams.kstream.Materialized;
import org.apache.kafka.streams.processor.Processor;
import org.apache.kafka.streams.processor.ProcessorContext;
import org.apache.kafka.streams.processor.StateRestoreListener;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.StateDirectory;
import org.apache.kafka.streams.state.KeyValueBytesStoreSupplier;
import org.apache.kafka.streams.state.KeyValueStore;
import org.apache.kafka.streams.state.StoreBuilder;
import org.apache.kafka.streams.state.Stores;
import org.apache.kafka.streams.state.internals.InMemoryKeyValueStore;
import org.apache.kafka.streams.state.internals.KeyValueStoreBuilder;
import org.apache.kafka.streams.state.internals.OffsetCheckpoint;
import org.apache.kafka.test.IntegrationTest;
import org.apache.kafka.test.TestUtils;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.core.IsEqual;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.rules.TestName;

@Category(value={IntegrationTest.class})
public class RestoreIntegrationTest {
    private static final int NUM_BROKERS = 1;
    public static final EmbeddedKafkaCluster CLUSTER = new EmbeddedKafkaCluster(1);
    @Rule
    public final TestName testName = new TestName();
    private String appId;
    private String inputStream;
    private final int numberOfKeys = 10000;
    private KafkaStreams kafkaStreams;

    @BeforeClass
    public static void startCluster() throws IOException {
        CLUSTER.start();
    }

    @AfterClass
    public static void closeCluster() {
        CLUSTER.stop();
    }

    @Before
    public void createTopics() throws InterruptedException {
        this.appId = IntegrationTestUtils.safeUniqueTestName(RestoreIntegrationTest.class, this.testName);
        this.inputStream = this.appId + "-input-stream";
        CLUSTER.createTopic(this.inputStream, 2, 1);
    }

    private Properties props() {
        Properties streamsConfiguration = new Properties();
        streamsConfiguration.put("application.id", this.appId);
        streamsConfiguration.put("bootstrap.servers", CLUSTER.bootstrapServers());
        streamsConfiguration.put("cache.max.bytes.buffering", (Object)0);
        streamsConfiguration.put("state.dir", TestUtils.tempDirectory((String)this.appId).getPath());
        streamsConfiguration.put("default.key.serde", Serdes.Integer().getClass());
        streamsConfiguration.put("default.value.serde", Serdes.Integer().getClass());
        streamsConfiguration.put("commit.interval.ms", (Object)1000L);
        streamsConfiguration.put("auto.offset.reset", "earliest");
        return streamsConfiguration;
    }

    @After
    public void shutdown() {
        if (this.kafkaStreams != null) {
            this.kafkaStreams.close(Duration.ofSeconds(30L));
        }
    }

    @Test
    public void shouldRestoreStateFromSourceTopic() throws Exception {
        AtomicInteger numReceived = new AtomicInteger(0);
        StreamsBuilder builder = new StreamsBuilder();
        Properties props = this.props();
        props.put("topology.optimization", "all");
        int offsetLimitDelta = 1000;
        int offsetCheckpointed = 1000;
        this.createStateForRestoration(this.inputStream, 0);
        this.setCommittedOffset(this.inputStream, 1000);
        StateDirectory stateDirectory = new StateDirectory(new StreamsConfig((Map)props), (Time)new MockTime(), true);
        new OffsetCheckpoint(new File(stateDirectory.getOrCreateDirectoryForTask(new TaskId(0, 0)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(this.inputStream, 0), 999L));
        new OffsetCheckpoint(new File(stateDirectory.getOrCreateDirectoryForTask(new TaskId(0, 1)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(this.inputStream, 1), 999L));
        CountDownLatch startupLatch = new CountDownLatch(1);
        CountDownLatch shutdownLatch = new CountDownLatch(1);
        builder.table(this.inputStream, Materialized.as((String)"store").withKeySerde(Serdes.Integer()).withValueSerde(Serdes.Integer())).toStream().foreach((key, value) -> {
            if (numReceived.incrementAndGet() == 2000) {
                shutdownLatch.countDown();
            }
        });
        this.kafkaStreams = new KafkaStreams(builder.build(props), props);
        this.kafkaStreams.setStateListener((newState, oldState) -> {
            if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                startupLatch.countDown();
            }
        });
        final AtomicLong restored = new AtomicLong(0L);
        this.kafkaStreams.setGlobalStateRestoreListener(new StateRestoreListener(){

            public void onRestoreStart(TopicPartition topicPartition, String storeName, long startingOffset, long endingOffset) {
            }

            public void onBatchRestored(TopicPartition topicPartition, String storeName, long batchEndOffset, long numRestored) {
            }

            public void onRestoreEnd(TopicPartition topicPartition, String storeName, long totalRestored) {
                restored.addAndGet(totalRestored);
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)restored.get(), (Matcher)IsEqual.equalTo((Object)6000L));
        Assert.assertTrue((boolean)shutdownLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)numReceived.get(), (Matcher)IsEqual.equalTo((Object)2000));
    }

    @Test
    public void shouldRestoreStateFromChangelogTopic() throws Exception {
        String changelog = this.appId + "-store-changelog";
        CLUSTER.createTopic(changelog, 2, 1);
        AtomicInteger numReceived = new AtomicInteger(0);
        StreamsBuilder builder = new StreamsBuilder();
        Properties props = this.props();
        int offsetCheckpointed = 1000;
        this.createStateForRestoration(changelog, 0);
        this.createStateForRestoration(this.inputStream, 10000);
        StateDirectory stateDirectory = new StateDirectory(new StreamsConfig((Map)props), (Time)new MockTime(), true);
        new OffsetCheckpoint(new File(stateDirectory.getOrCreateDirectoryForTask(new TaskId(0, 0)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(changelog, 0), 999L));
        new OffsetCheckpoint(new File(stateDirectory.getOrCreateDirectoryForTask(new TaskId(0, 1)), ".checkpoint")).write(Collections.singletonMap(new TopicPartition(changelog, 1), 999L));
        CountDownLatch startupLatch = new CountDownLatch(1);
        CountDownLatch shutdownLatch = new CountDownLatch(1);
        builder.table(this.inputStream, Consumed.with((Serde)Serdes.Integer(), (Serde)Serdes.Integer()), Materialized.as((String)"store")).toStream().foreach((key, value) -> {
            if (numReceived.incrementAndGet() == 10000) {
                shutdownLatch.countDown();
            }
        });
        this.kafkaStreams = new KafkaStreams(builder.build(), props);
        this.kafkaStreams.setStateListener((newState, oldState) -> {
            if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                startupLatch.countDown();
            }
        });
        final AtomicLong restored = new AtomicLong(0L);
        this.kafkaStreams.setGlobalStateRestoreListener(new StateRestoreListener(){

            public void onRestoreStart(TopicPartition topicPartition, String storeName, long startingOffset, long endingOffset) {
            }

            public void onBatchRestored(TopicPartition topicPartition, String storeName, long batchEndOffset, long numRestored) {
            }

            public void onRestoreEnd(TopicPartition topicPartition, String storeName, long totalRestored) {
                restored.addAndGet(totalRestored);
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)restored.get(), (Matcher)IsEqual.equalTo((Object)8000L));
        Assert.assertTrue((boolean)shutdownLatch.await(30L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)numReceived.get(), (Matcher)IsEqual.equalTo((Object)10000));
    }

    @Test
    public void shouldSuccessfullyStartWhenLoggingDisabled() throws InterruptedException {
        StreamsBuilder builder = new StreamsBuilder();
        KStream stream = builder.stream(this.inputStream);
        stream.groupByKey().reduce((value1, value2) -> value1 + value2, Materialized.as((String)"reduce-store").withLoggingDisabled());
        CountDownLatch startupLatch = new CountDownLatch(1);
        this.kafkaStreams = new KafkaStreams(builder.build(), this.props());
        this.kafkaStreams.setStateListener((newState, oldState) -> {
            if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                startupLatch.countDown();
            }
        });
        this.kafkaStreams.start();
        Assert.assertTrue((boolean)startupLatch.await(30L, TimeUnit.SECONDS));
    }

    @Test
    public void shouldProcessDataFromStoresWithLoggingDisabled() throws InterruptedException {
        IntegrationTestUtils.produceKeyValuesSynchronously(this.inputStream, Arrays.asList(KeyValue.pair((Object)1, (Object)1), KeyValue.pair((Object)2, (Object)2), KeyValue.pair((Object)3, (Object)3)), TestUtils.producerConfig((String)CLUSTER.bootstrapServers(), IntegerSerializer.class, IntegerSerializer.class), (Time)RestoreIntegrationTest.CLUSTER.time);
        KeyValueBytesStoreSupplier lruMapSupplier = Stores.lruMap((String)this.inputStream, (int)10);
        StoreBuilder storeBuilder = new KeyValueStoreBuilder(lruMapSupplier, Serdes.Integer(), Serdes.Integer(), (Time)RestoreIntegrationTest.CLUSTER.time).withLoggingDisabled();
        StreamsBuilder streamsBuilder = new StreamsBuilder();
        streamsBuilder.addStateStore(storeBuilder);
        KStream stream = streamsBuilder.stream(this.inputStream);
        CountDownLatch processorLatch = new CountDownLatch(3);
        stream.process(() -> new KeyValueStoreProcessor(this.inputStream, processorLatch), new String[]{this.inputStream});
        Topology topology = streamsBuilder.build();
        this.kafkaStreams = new KafkaStreams(topology, this.props());
        CountDownLatch latch = new CountDownLatch(1);
        this.kafkaStreams.setStateListener((newState, oldState) -> {
            if (newState == KafkaStreams.State.RUNNING && oldState == KafkaStreams.State.REBALANCING) {
                latch.countDown();
            }
        });
        this.kafkaStreams.start();
        latch.await(30L, TimeUnit.SECONDS);
        Assert.assertTrue((boolean)processorLatch.await(30L, TimeUnit.SECONDS));
    }

    @Test
    public void shouldRecycleStateFromStandbyTaskPromotedToActiveTaskAndNotRestore() throws Exception {
        StreamsBuilder builder = new StreamsBuilder();
        builder.table(this.inputStream, Consumed.with((Serde)Serdes.Integer(), (Serde)Serdes.Integer()), Materialized.as((KeyValueBytesStoreSupplier)RestoreIntegrationTest.getCloseCountingStore("store")));
        this.createStateForRestoration(this.inputStream, 0);
        Properties props1 = this.props();
        props1.put("num.standby.replicas", (Object)1);
        props1.put("state.dir", TestUtils.tempDirectory((String)(this.appId + "-1")).getPath());
        IntegrationTestUtils.purgeLocalStreamsState(props1);
        KafkaStreams client1 = new KafkaStreams(builder.build(), props1);
        Properties props2 = this.props();
        props2.put("num.standby.replicas", (Object)1);
        props2.put("state.dir", TestUtils.tempDirectory((String)(this.appId + "-2")).getPath());
        IntegrationTestUtils.purgeLocalStreamsState(props2);
        KafkaStreams client2 = new KafkaStreams(builder.build(), props2);
        IntegrationTestUtils.TrackingStateRestoreListener restoreListener = new IntegrationTestUtils.TrackingStateRestoreListener();
        client1.setGlobalStateRestoreListener((StateRestoreListener)restoreListener);
        IntegrationTestUtils.startApplicationAndWaitUntilRunning(Arrays.asList(client1, client2), Duration.ofSeconds(60L));
        IntegrationTestUtils.waitForCompletion(client1, 1, 30000L);
        IntegrationTestUtils.waitForCompletion(client2, 1, 30000L);
        IntegrationTestUtils.waitForStandbyCompletion(client1, 1, 30000L);
        IntegrationTestUtils.waitForStandbyCompletion(client2, 1, 30000L);
        int initialStoreCloseCount = CloseCountingInMemoryStore.numStoresClosed();
        MatcherAssert.assertThat((Object)restoreListener.totalNumRestored(), (Matcher)CoreMatchers.equalTo((Object)0L));
        client2.close();
        IntegrationTestUtils.waitForApplicationState(Collections.singletonList(client2), KafkaStreams.State.NOT_RUNNING, Duration.ofSeconds(60L));
        IntegrationTestUtils.waitForApplicationState(Collections.singletonList(client1), KafkaStreams.State.REBALANCING, Duration.ofSeconds(60L));
        IntegrationTestUtils.waitForApplicationState(Collections.singletonList(client1), KafkaStreams.State.RUNNING, Duration.ofSeconds(60L));
        IntegrationTestUtils.waitForCompletion(client1, 1, 30000L);
        IntegrationTestUtils.waitForStandbyCompletion(client1, 1, 30000L);
        MatcherAssert.assertThat((Object)restoreListener.totalNumRestored(), (Matcher)CoreMatchers.equalTo((Object)0L));
        MatcherAssert.assertThat((Object)CloseCountingInMemoryStore.numStoresClosed(), (Matcher)IsEqual.equalTo((Object)(initialStoreCloseCount + 2)));
        client1.close();
        IntegrationTestUtils.waitForApplicationState(Collections.singletonList(client2), KafkaStreams.State.NOT_RUNNING, Duration.ofSeconds(60L));
        MatcherAssert.assertThat((Object)CloseCountingInMemoryStore.numStoresClosed(), (Matcher)CoreMatchers.equalTo((Object)(initialStoreCloseCount + 4)));
    }

    private static KeyValueBytesStoreSupplier getCloseCountingStore(final String name) {
        return new KeyValueBytesStoreSupplier(){

            public String name() {
                return name;
            }

            public KeyValueStore<Bytes, byte[]> get() {
                return new CloseCountingInMemoryStore(name);
            }

            public String metricsScope() {
                return "close-counting";
            }
        };
    }

    private void createStateForRestoration(String changelogTopic, int startingOffset) {
        Properties producerConfig = new Properties();
        producerConfig.put("bootstrap.servers", CLUSTER.bootstrapServers());
        try (KafkaProducer producer = new KafkaProducer(producerConfig, (Serializer)new IntegerSerializer(), (Serializer)new IntegerSerializer());){
            for (int i = 0; i < 10000; ++i) {
                int offset = startingOffset + i;
                producer.send(new ProducerRecord(changelogTopic, (Object)offset, (Object)offset));
            }
        }
    }

    private void setCommittedOffset(String topic, int limitDelta) {
        Properties consumerConfig = new Properties();
        consumerConfig.put("bootstrap.servers", CLUSTER.bootstrapServers());
        consumerConfig.put("group.id", this.appId);
        consumerConfig.put("client.id", "commit-consumer");
        consumerConfig.put("key.deserializer", IntegerDeserializer.class);
        consumerConfig.put("value.deserializer", IntegerDeserializer.class);
        KafkaConsumer consumer = new KafkaConsumer(consumerConfig);
        List<TopicPartition> partitions = Arrays.asList(new TopicPartition(topic, 0), new TopicPartition(topic, 1));
        consumer.assign(partitions);
        consumer.seekToEnd(partitions);
        for (TopicPartition partition : partitions) {
            long position = consumer.position(partition);
            consumer.seek(partition, position - (long)limitDelta);
        }
        consumer.commitSync();
        consumer.close();
    }

    public static class KeyValueStoreProcessor
    implements Processor<Integer, Integer> {
        private final String topic;
        private final CountDownLatch processorLatch;
        private KeyValueStore<Integer, Integer> store;

        KeyValueStoreProcessor(String topic, CountDownLatch processorLatch) {
            this.topic = topic;
            this.processorLatch = processorLatch;
        }

        public void init(ProcessorContext context) {
            this.store = (KeyValueStore)context.getStateStore(this.topic);
        }

        public void process(Integer key, Integer value) {
            if (key != null) {
                this.store.put((Object)key, (Object)value);
                this.processorLatch.countDown();
            }
        }

        public void close() {
        }
    }

    static class CloseCountingInMemoryStore
    extends InMemoryKeyValueStore {
        static AtomicInteger numStoresClosed = new AtomicInteger(0);

        CloseCountingInMemoryStore(String name) {
            super(name);
        }

        public void close() {
            numStoresClosed.incrementAndGet();
            super.close();
        }

        static int numStoresClosed() {
            return numStoresClosed.get();
        }
    }
}

