/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.connect.runtime;

import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import org.apache.kafka.connect.connector.Connector;
import org.apache.kafka.connect.connector.ConnectorContext;
import org.apache.kafka.connect.runtime.AbstractStatus;
import org.apache.kafka.connect.runtime.ConnectMetrics;
import org.apache.kafka.connect.runtime.ConnectMetricsRegistry;
import org.apache.kafka.connect.runtime.ConnectorConfig;
import org.apache.kafka.connect.runtime.ConnectorStatus;
import org.apache.kafka.connect.runtime.SinkConnectorConfig;
import org.apache.kafka.connect.runtime.TargetState;
import org.apache.kafka.connect.sink.SinkConnector;
import org.apache.kafka.connect.source.SourceConnector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkerConnector {
    private static final Logger log = LoggerFactory.getLogger(WorkerConnector.class);
    private final String connName;
    private final ConnectorStatus.Listener statusListener;
    private final ConnectorContext ctx;
    private final Connector connector;
    private final ConnectorMetricsGroup metrics;
    private Map<String, String> config;
    private State state;

    public WorkerConnector(String connName, Connector connector, ConnectorContext ctx, ConnectMetrics metrics, ConnectorStatus.Listener statusListener) {
        this.connName = connName;
        this.ctx = ctx;
        this.connector = connector;
        this.state = State.INIT;
        this.metrics = new ConnectorMetricsGroup(metrics, AbstractStatus.State.UNASSIGNED, statusListener);
        this.statusListener = this.metrics;
    }

    public void initialize(ConnectorConfig connectorConfig) {
        try {
            this.config = connectorConfig.originalsStrings();
            log.debug("{} Initializing connector {} with config {}", new Object[]{this, this.connName, this.config});
            if (this.isSinkConnector()) {
                SinkConnectorConfig.validate(this.config);
            }
            this.connector.initialize(new ConnectorContext(){

                public void requestTaskReconfiguration() {
                    WorkerConnector.this.ctx.requestTaskReconfiguration();
                }

                public void raiseError(Exception e) {
                    log.error("{} Connector raised an error", (Object)WorkerConnector.this, (Object)e);
                    WorkerConnector.this.onFailure(e);
                    WorkerConnector.this.ctx.raiseError(e);
                }
            });
        }
        catch (Throwable t) {
            log.error("{} Error initializing connector", (Object)this, (Object)t);
            this.onFailure(t);
        }
    }

    private boolean doStart() {
        try {
            switch (this.state) {
                case STARTED: {
                    return false;
                }
                case INIT: 
                case STOPPED: {
                    this.connector.start(this.config);
                    this.state = State.STARTED;
                    return true;
                }
            }
            throw new IllegalArgumentException("Cannot start connector in state " + (Object)((Object)this.state));
        }
        catch (Throwable t) {
            log.error("{} Error while starting connector", (Object)this, (Object)t);
            this.onFailure(t);
            return false;
        }
    }

    private void onFailure(Throwable t) {
        this.statusListener.onFailure(this.connName, t);
        this.state = State.FAILED;
    }

    private void resume() {
        if (this.doStart()) {
            this.statusListener.onResume(this.connName);
        }
    }

    private void start() {
        if (this.doStart()) {
            this.statusListener.onStartup(this.connName);
        }
    }

    public boolean isRunning() {
        return this.state == State.STARTED;
    }

    private void pause() {
        try {
            switch (this.state) {
                case STOPPED: {
                    return;
                }
                case STARTED: {
                    this.connector.stop();
                }
                case INIT: {
                    this.statusListener.onPause(this.connName);
                    this.state = State.STOPPED;
                    break;
                }
                default: {
                    throw new IllegalArgumentException("Cannot pause connector in state " + (Object)((Object)this.state));
                }
            }
        }
        catch (Throwable t) {
            log.error("{} Error while shutting down connector", (Object)this, (Object)t);
            this.statusListener.onFailure(this.connName, t);
            this.state = State.FAILED;
        }
    }

    public void shutdown() {
        try {
            if (this.state == State.STARTED) {
                this.connector.stop();
            }
            this.state = State.STOPPED;
            this.statusListener.onShutdown(this.connName);
        }
        catch (Throwable t) {
            log.error("{} Error while shutting down connector", (Object)this, (Object)t);
            this.state = State.FAILED;
            this.statusListener.onFailure(this.connName, t);
        }
        finally {
            this.metrics.close();
        }
    }

    public void transitionTo(TargetState targetState) {
        if (this.state == State.FAILED) {
            log.warn("{} Cannot transition connector to {} since it has failed", (Object)this, (Object)targetState);
            return;
        }
        log.debug("{} Transition connector to {}", (Object)this, (Object)targetState);
        if (targetState == TargetState.PAUSED) {
            this.pause();
        } else if (targetState == TargetState.STARTED) {
            if (this.state == State.INIT) {
                this.start();
            } else {
                this.resume();
            }
        } else {
            throw new IllegalArgumentException("Unhandled target state " + (Object)((Object)targetState));
        }
    }

    public boolean isSinkConnector() {
        return SinkConnector.class.isAssignableFrom(this.connector.getClass());
    }

    public boolean isSourceConnector() {
        return SourceConnector.class.isAssignableFrom(this.connector.getClass());
    }

    protected String connectorType() {
        if (this.isSinkConnector()) {
            return "sink";
        }
        if (this.isSourceConnector()) {
            return "source";
        }
        return "unknown";
    }

    public Connector connector() {
        return this.connector;
    }

    ConnectorMetricsGroup metrics() {
        return this.metrics;
    }

    public String toString() {
        return "WorkerConnector{id=" + this.connName + '}';
    }

    class ConnectorMetricsGroup
    implements ConnectorStatus.Listener,
    AutoCloseable {
        private volatile AbstractStatus.State state;
        private final ConnectMetrics.MetricGroup metricGroup;
        private final ConnectorStatus.Listener delegate;

        public ConnectorMetricsGroup(ConnectMetrics connectMetrics, AbstractStatus.State initialState, ConnectorStatus.Listener delegate) {
            Objects.requireNonNull(connectMetrics);
            Objects.requireNonNull(WorkerConnector.this.connector);
            Objects.requireNonNull(initialState);
            Objects.requireNonNull(delegate);
            this.delegate = delegate;
            this.state = initialState;
            ConnectMetricsRegistry registry = connectMetrics.registry();
            this.metricGroup = connectMetrics.group(registry.connectorGroupName(), registry.connectorTagName(), WorkerConnector.this.connName);
            this.metricGroup.close();
            this.metricGroup.addImmutableValueMetric(registry.connectorType, WorkerConnector.this.connectorType());
            this.metricGroup.addImmutableValueMetric(registry.connectorClass, WorkerConnector.this.connector.getClass().getName());
            this.metricGroup.addImmutableValueMetric(registry.connectorVersion, WorkerConnector.this.connector.version());
            this.metricGroup.addValueMetric(registry.connectorStatus, new ConnectMetrics.LiteralSupplier<String>(){

                @Override
                public String metricValue(long now) {
                    return ConnectorMetricsGroup.this.state.toString().toLowerCase(Locale.getDefault());
                }
            });
        }

        @Override
        public void close() {
            this.metricGroup.close();
        }

        @Override
        public void onStartup(String connector) {
            this.state = AbstractStatus.State.RUNNING;
            this.delegate.onStartup(connector);
        }

        @Override
        public void onShutdown(String connector) {
            this.state = AbstractStatus.State.UNASSIGNED;
            this.delegate.onShutdown(connector);
        }

        @Override
        public void onPause(String connector) {
            this.state = AbstractStatus.State.PAUSED;
            this.delegate.onPause(connector);
        }

        @Override
        public void onResume(String connector) {
            this.state = AbstractStatus.State.RUNNING;
            this.delegate.onResume(connector);
        }

        @Override
        public void onFailure(String connector, Throwable cause) {
            this.state = AbstractStatus.State.FAILED;
            this.delegate.onFailure(connector, cause);
        }

        @Override
        public void onDeletion(String connector) {
            this.state = AbstractStatus.State.DESTROYED;
            this.delegate.onDeletion(connector);
        }

        boolean isUnassigned() {
            return this.state == AbstractStatus.State.UNASSIGNED;
        }

        boolean isRunning() {
            return this.state == AbstractStatus.State.RUNNING;
        }

        boolean isPaused() {
            return this.state == AbstractStatus.State.PAUSED;
        }

        boolean isFailed() {
            return this.state == AbstractStatus.State.FAILED;
        }

        protected ConnectMetrics.MetricGroup metricGroup() {
            return this.metricGroup;
        }
    }

    private static enum State {
        INIT,
        STOPPED,
        STARTED,
        FAILED;

    }
}

