/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document.persistentCache;

import com.google.common.base.Function;
import com.google.common.cache.Cache;
import java.io.File;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeSet;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.jackrabbit.oak.cache.CacheValue;
import org.apache.jackrabbit.oak.plugins.document.DocumentNodeStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.CacheMap;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.CacheType;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.NodeCache;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.PersistentCacheStats;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.async.CacheActionDispatcher;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.broadcast.Broadcaster;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.broadcast.DynamicBroadcastConfig;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.broadcast.InMemoryBroadcaster;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.broadcast.TCPBroadcaster;
import org.apache.jackrabbit.oak.plugins.document.persistentCache.broadcast.UDPBroadcaster;
import org.apache.jackrabbit.oak.stats.StatisticsProvider;
import org.h2.mvstore.FileStore;
import org.h2.mvstore.MVMap;
import org.h2.mvstore.MVStore;
import org.h2.mvstore.MVStoreTool;
import org.h2.mvstore.WriteBuffer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PersistentCache
implements Broadcaster.Listener {
    static final Logger LOG = LoggerFactory.getLogger(PersistentCache.class);
    private static final String FILE_PREFIX = "cache-";
    private static final String FILE_SUFFIX = ".data";
    private static final AtomicInteger COUNTER = new AtomicInteger();
    private boolean cacheNodes = true;
    private boolean cacheChildren = true;
    private boolean cacheDiff = true;
    private boolean cacheLocalDiff = true;
    private boolean cachePrevDocs = true;
    private boolean compactOnClose;
    private boolean compress = true;
    private boolean asyncCache = true;
    private boolean asyncDiffCache = false;
    private HashMap<CacheType, GenerationCache> caches = new HashMap();
    private final String directory;
    private MapFactory writeStore;
    private MapFactory readStore;
    private int maxSizeMB = 1024;
    private int memCache = -1;
    private int readGeneration = -1;
    private int writeGeneration;
    private int autoCompact = 0;
    private boolean appendOnly;
    private boolean manualCommit;
    private Broadcaster broadcaster;
    private ThreadLocal<WriteBuffer> writeBuffer = new ThreadLocal();
    private final byte[] broadcastId;
    private DynamicBroadcastConfig broadcastConfig;
    private CacheActionDispatcher writeDispatcher;
    private Thread writeDispatcherThread;
    private int exceptionCount;

    public PersistentCache(String url) {
        ByteBuffer bb = ByteBuffer.wrap(new byte[16]);
        UUID uuid = UUID.randomUUID();
        bb.putLong(uuid.getMostSignificantBits());
        bb.putLong(uuid.getLeastSignificantBits());
        this.broadcastId = bb.array();
        LOG.info("start, url={}", (Object)url);
        String[] parts = url.split(",");
        String dir = parts[0];
        String broadcast = "disabled";
        for (String string : parts) {
            if (string.equals("+docs")) {
                this.logUnsupportedWarning("docs");
                continue;
            }
            if (string.equals("-prevDocs")) {
                this.cachePrevDocs = false;
                continue;
            }
            if (string.equals("+docChildren")) {
                this.logUnsupportedWarning("docChildren");
                continue;
            }
            if (string.equals("-nodes")) {
                this.cacheNodes = false;
                continue;
            }
            if (string.equals("-children")) {
                this.cacheChildren = false;
                continue;
            }
            if (string.equals("-diff")) {
                this.cacheDiff = false;
                continue;
            }
            if (string.equals("-localDiff")) {
                this.cacheLocalDiff = false;
                continue;
            }
            if (string.equals("+all")) {
                this.logUnsupportedWarning("all");
                continue;
            }
            if (string.equals("-compact")) {
                this.compactOnClose = false;
                continue;
            }
            if (string.equals("+compact")) {
                this.compactOnClose = true;
                continue;
            }
            if (string.equals("-compress")) {
                this.compress = false;
                continue;
            }
            if (string.endsWith("time")) {
                dir = dir + "-" + System.currentTimeMillis() + "-" + COUNTER.getAndIncrement();
                continue;
            }
            if (string.startsWith("size=")) {
                this.maxSizeMB = Integer.parseInt(string.split("=")[1]);
                continue;
            }
            if (string.startsWith("memCache=")) {
                this.memCache = Integer.parseInt(string.split("=")[1]);
                continue;
            }
            if (string.startsWith("binary=")) {
                this.logUnsupportedWarning("binary");
                continue;
            }
            if (string.startsWith("autoCompact=")) {
                this.autoCompact = Integer.parseInt(string.split("=")[1]);
                continue;
            }
            if (string.equals("appendOnly")) {
                this.appendOnly = true;
                continue;
            }
            if (string.equals("manualCommit")) {
                this.manualCommit = true;
                continue;
            }
            if (string.startsWith("broadcast=")) {
                broadcast = string.split("=")[1];
                continue;
            }
            if (string.equals("-async")) {
                this.asyncCache = false;
                continue;
            }
            if (!string.equals("+asyncDiff")) continue;
            this.asyncDiffCache = true;
        }
        this.directory = dir;
        if (dir.length() == 0) {
            this.readGeneration = -1;
            this.writeGeneration = 0;
            this.writeStore = this.createMapFactory(this.writeGeneration, false);
            return;
        }
        File dr = new File(dir);
        if (!dr.exists()) {
            dr.mkdirs();
        }
        if (dr.exists() && !dr.isDirectory()) {
            throw new IllegalArgumentException("A file exists at cache directory " + dir);
        }
        File[] list = dr.listFiles();
        TreeSet<Integer> generations = new TreeSet<Integer>();
        if (list != null) {
            for (File f : list) {
                String fn = f.getName();
                if (!fn.startsWith(FILE_PREFIX) || !fn.endsWith(FILE_SUFFIX)) continue;
                String g = fn.substring(FILE_PREFIX.length(), fn.indexOf(FILE_SUFFIX));
                try {
                    File f2;
                    int gen = Integer.parseInt(g);
                    if (gen < 0 || !fn.equals((f2 = new File(this.getFileName(gen))).getName())) continue;
                    generations.add(gen);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        while (generations.size() > 2) {
            Integer n = (Integer)generations.first();
            File oldFile = new File(this.getFileName(n));
            if (!oldFile.canWrite()) {
                LOG.info("Ignoring old, read-only generation " + oldFile.getAbsolutePath());
            } else {
                LOG.info("Removing old generation " + oldFile.getAbsolutePath());
                oldFile.delete();
            }
            generations.remove(n);
        }
        this.readGeneration = generations.size() > 1 ? (Integer)generations.first() : -1;
        int n = this.writeGeneration = generations.size() > 0 ? (Integer)generations.last() : 0;
        if (this.readGeneration >= 0) {
            this.readStore = this.createMapFactory(this.readGeneration, true);
        }
        this.writeStore = this.createMapFactory(this.writeGeneration, false);
        this.initBroadcast(broadcast);
        this.writeDispatcher = new CacheActionDispatcher();
        this.writeDispatcherThread = new Thread((Runnable)this.writeDispatcher, "Oak CacheWriteQueue");
        this.writeDispatcherThread.setDaemon(true);
        this.writeDispatcherThread.start();
    }

    private void logUnsupportedWarning(String configKey) {
        LOG.warn("Support for '{}' has been removed from persistent cache. Please update the configuration.", (Object)configKey);
    }

    private void initBroadcast(String broadcast) {
        if (broadcast == null) {
            return;
        }
        if (broadcast.equals("disabled")) {
            return;
        }
        if (broadcast.equals("inMemory")) {
            this.broadcaster = InMemoryBroadcaster.INSTANCE;
        } else if (broadcast.startsWith("udp:")) {
            String config = broadcast.substring("udp:".length(), broadcast.length());
            this.broadcaster = new UDPBroadcaster(config);
        } else if (broadcast.startsWith("tcp:")) {
            String config = broadcast.substring("tcp:".length(), broadcast.length());
            this.broadcaster = new TCPBroadcaster(config);
        } else {
            throw new IllegalArgumentException("Unknown broadcaster type " + broadcast);
        }
        this.broadcaster.addListener(this);
    }

    private String getFileName(int generation) {
        if (this.directory.length() == 0) {
            return null;
        }
        return this.directory + "/" + FILE_PREFIX + generation + FILE_SUFFIX;
    }

    private MapFactory createMapFactory(final int generation, final boolean readOnly) {
        MapFactory f = new MapFactory(){
            final String fileName;
            MVStore store;
            {
                this.fileName = PersistentCache.this.getFileName(generation);
            }

            @Override
            void openStore() {
                if (this.store != null) {
                    return;
                }
                MVStore.Builder builder = new MVStore.Builder();
                try {
                    if (PersistentCache.this.compress) {
                        builder.compress();
                    }
                    if (PersistentCache.this.manualCommit) {
                        builder.autoCommitDisabled();
                    }
                    if (this.fileName != null) {
                        builder.fileName(this.fileName);
                    }
                    if (PersistentCache.this.memCache >= 0) {
                        builder.cacheSize(PersistentCache.this.memCache);
                    }
                    if (readOnly) {
                        builder.readOnly();
                    }
                    if (PersistentCache.this.maxSizeMB < 10) {
                        builder.cacheSize(PersistentCache.this.maxSizeMB);
                    }
                    if (PersistentCache.this.autoCompact >= 0) {
                        builder.autoCompactFillRate(PersistentCache.this.autoCompact);
                    }
                    builder.backgroundExceptionHandler(new Thread.UncaughtExceptionHandler(){

                        @Override
                        public void uncaughtException(Thread t, Throwable e) {
                            PersistentCache.this.exceptionCount++;
                            MapFactory.LOG.debug("Error in the background thread of the persistent cache", e);
                            MapFactory.LOG.warn("Error in the background thread of the persistent cache: " + e);
                        }
                    });
                    this.store = builder.open();
                    if (PersistentCache.this.appendOnly) {
                        this.store.setReuseSpace(false);
                    }
                }
                catch (Exception e) {
                    PersistentCache.this.exceptionCount++;
                    LOG.warn("Could not open the store " + this.fileName, (Throwable)e);
                }
            }

            @Override
            synchronized void closeStore() {
                if (this.store == null) {
                    return;
                }
                boolean compact = PersistentCache.this.compactOnClose;
                try {
                    if (this.store.getFileStore().isReadOnly()) {
                        compact = false;
                    }
                    Thread.interrupted();
                    this.store.close();
                }
                catch (Exception e) {
                    PersistentCache.this.exceptionCount++;
                    LOG.debug("Could not close the store", (Throwable)e);
                    LOG.warn("Could not close the store: " + e);
                    this.store.closeImmediately();
                }
                if (compact) {
                    try {
                        MVStoreTool.compact((String)this.fileName, (boolean)true);
                    }
                    catch (Exception e) {
                        PersistentCache.this.exceptionCount++;
                        LOG.debug("Could not compact the store", (Throwable)e);
                        LOG.warn("Could not compact the store: " + e);
                    }
                }
                this.store = null;
            }

            @Override
            <K, V> Map<K, V> openMap(String name, MVMap.Builder<K, V> builder) {
                try {
                    if (builder == null) {
                        return this.store.openMap(name);
                    }
                    return this.store.openMap(name, builder);
                }
                catch (Exception e) {
                    PersistentCache.this.exceptionCount++;
                    LOG.warn("Could not open the map", (Throwable)e);
                    return null;
                }
            }

            @Override
            long getFileSize() {
                try {
                    if (this.store == null) {
                        return 0L;
                    }
                    FileStore fs = this.store.getFileStore();
                    if (fs == null) {
                        return 0L;
                    }
                    return fs.size();
                }
                catch (Exception e) {
                    PersistentCache.this.exceptionCount++;
                    LOG.warn("Could not retrieve the map size", (Throwable)e);
                    return 0L;
                }
            }
        };
        f.openStore();
        return f;
    }

    public void close() {
        this.writeDispatcher.stop();
        try {
            this.writeDispatcherThread.join();
        }
        catch (InterruptedException e) {
            LOG.error("Can't join the {}", (Object)this.writeDispatcherThread.getName(), (Object)e);
        }
        if (this.writeStore != null) {
            this.writeStore.closeStore();
        }
        if (this.readStore != null) {
            this.readStore.closeStore();
        }
        if (this.broadcaster != null) {
            this.broadcaster.removeListener(this);
            this.broadcaster.close();
            this.broadcaster = null;
        }
        this.writeBuffer.remove();
    }

    public synchronized <K extends CacheValue, V extends CacheValue> Cache<K, V> wrap(DocumentNodeStore docNodeStore, DocumentStore docStore, Cache<K, V> base, CacheType type) {
        return this.wrap(docNodeStore, docStore, base, type, StatisticsProvider.NOOP);
    }

    public synchronized <K extends CacheValue, V extends CacheValue> Cache<K, V> wrap(DocumentNodeStore docNodeStore, DocumentStore docStore, Cache<K, V> base, CacheType type, StatisticsProvider statisticsProvider) {
        boolean wrap;
        boolean async = this.asyncCache;
        switch (type) {
            case NODE: {
                wrap = this.cacheNodes;
                break;
            }
            case CHILDREN: {
                wrap = this.cacheChildren;
                break;
            }
            case DIFF: {
                wrap = this.cacheDiff;
                async = this.asyncDiffCache;
                break;
            }
            case LOCAL_DIFF: {
                wrap = this.cacheLocalDiff;
                async = this.asyncDiffCache;
                break;
            }
            case PREV_DOCUMENT: {
                wrap = this.cachePrevDocs;
                break;
            }
            default: {
                wrap = false;
            }
        }
        if (wrap) {
            NodeCache<K, V> c = new NodeCache<K, V>(this, base, docNodeStore, docStore, type, this.writeDispatcher, statisticsProvider, async);
            this.initGenerationCache(c);
            return c;
        }
        return base;
    }

    private void initGenerationCache(GenerationCache c) {
        this.caches.put(c.getType(), c);
        if (this.readGeneration >= 0) {
            c.addGeneration(this.readGeneration, true);
        }
        c.addGeneration(this.writeGeneration, false);
    }

    public synchronized <K, V> CacheMap<K, V> openMap(int generation, String name, MVMap.Builder<K, V> builder) {
        MapFactory s;
        if (generation == this.readGeneration) {
            s = this.readStore;
        } else if (generation == this.writeGeneration) {
            s = this.writeStore;
        } else {
            ++this.exceptionCount;
            throw new IllegalArgumentException("Unknown generation: " + generation);
        }
        return new CacheMap<K, V>(s, name, builder);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void switchGenerationIfNeeded() {
        if (!this.needSwitch()) {
            return;
        }
        PersistentCache persistentCache = this;
        synchronized (persistentCache) {
            MapFactory w;
            if (!this.needSwitch()) {
                return;
            }
            int oldReadGeneration = this.readGeneration;
            MapFactory oldRead = this.readStore;
            this.readStore = this.writeStore;
            this.readGeneration = this.writeGeneration;
            this.writeStore = w = this.createMapFactory(this.writeGeneration + 1, false);
            ++this.writeGeneration;
            for (GenerationCache c : this.caches.values()) {
                c.addGeneration(this.writeGeneration, false);
                if (oldReadGeneration < 0) continue;
                c.removeGeneration(oldReadGeneration);
            }
            if (oldRead != null) {
                oldRead.closeStore();
                new File(this.getFileName(oldReadGeneration)).delete();
            }
        }
    }

    boolean needSwitch() {
        long size = this.writeStore.getFileSize();
        return size / 1024L / 1024L > (long)this.maxSizeMB;
    }

    public int getMaxSize() {
        return this.maxSizeMB;
    }

    public int getOpenCount() {
        return this.writeStore.getOpenCount();
    }

    public int getExceptionCount() {
        return this.exceptionCount;
    }

    void broadcast(CacheType type, Function<WriteBuffer, Void> writer) {
        Broadcaster b = this.broadcaster;
        if (b == null) {
            return;
        }
        WriteBuffer buff = this.writeBuffer.get();
        if (buff == null) {
            buff = new WriteBuffer();
            this.writeBuffer.set(buff);
        }
        buff.clear();
        buff.putInt(0);
        buff.put(this.broadcastId);
        buff.put((byte)type.ordinal());
        writer.apply((Object)buff);
        ByteBuffer byteBuff = buff.getBuffer();
        int length = byteBuff.position();
        byteBuff.limit(length);
        byteBuff.putInt(0, length);
        byteBuff.position(0);
        b.send(byteBuff);
    }

    @Override
    public void receive(ByteBuffer buff) {
        int end = buff.position() + buff.getInt();
        byte[] id = new byte[this.broadcastId.length];
        buff.get(id);
        if (!Arrays.equals(id, this.broadcastId)) {
            this.receiveMessage(buff);
        }
        buff.position(end);
    }

    public static PersistentCacheStats getPersistentCacheStats(Cache<?, ?> cache) {
        if (cache instanceof NodeCache) {
            return ((NodeCache)cache).getPersistentCacheStats();
        }
        return null;
    }

    private void receiveMessage(ByteBuffer buff) {
        CacheType type = CacheType.VALUES[buff.get()];
        GenerationCache cache = this.caches.get((Object)type);
        if (cache == null) {
            return;
        }
        cache.receive(buff);
    }

    public DynamicBroadcastConfig getBroadcastConfig() {
        return this.broadcastConfig;
    }

    public void setBroadcastConfig(DynamicBroadcastConfig broadcastConfig) {
        this.broadcastConfig = broadcastConfig;
        if (this.broadcaster != null) {
            this.broadcaster.setBroadcastConfig(broadcastConfig);
        }
    }

    static interface GenerationCache {
        public void addGeneration(int var1, boolean var2);

        public CacheType getType();

        public void receive(ByteBuffer var1);

        public void removeGeneration(int var1);
    }
}

