/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.document;

import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.Collections;
import java.util.Map;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.plugins.document.ClusterNodeInfoDocument;
import org.apache.jackrabbit.oak.plugins.document.Collection;
import org.apache.jackrabbit.oak.plugins.document.DocumentStore;
import org.apache.jackrabbit.oak.plugins.document.DocumentStoreException;
import org.apache.jackrabbit.oak.plugins.document.JournalEntry;
import org.apache.jackrabbit.oak.plugins.document.MissingLastRevSeeker;
import org.apache.jackrabbit.oak.plugins.document.NodeDocument;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentSweepListener;
import org.apache.jackrabbit.oak.plugins.document.NodeDocumentSweeper;
import org.apache.jackrabbit.oak.plugins.document.RecoveryContext;
import org.apache.jackrabbit.oak.plugins.document.Revision;
import org.apache.jackrabbit.oak.plugins.document.RevisionContext;
import org.apache.jackrabbit.oak.plugins.document.UnsavedModifications;
import org.apache.jackrabbit.oak.plugins.document.UpdateOp;
import org.apache.jackrabbit.oak.plugins.document.util.MapFactory;
import org.apache.jackrabbit.oak.plugins.document.util.Utils;
import org.apache.jackrabbit.oak.stats.Clock;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LastRevRecoveryAgent {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final DocumentStore store;
    private final RevisionContext revisionContext;
    private final MissingLastRevSeeker missingLastRevUtil;
    private final Consumer<Integer> afterRecovery;

    public LastRevRecoveryAgent(DocumentStore store, RevisionContext revisionContext, MissingLastRevSeeker seeker, Consumer<Integer> afterRecovery) {
        this.store = store;
        this.revisionContext = revisionContext;
        this.missingLastRevUtil = seeker;
        this.afterRecovery = afterRecovery;
    }

    public LastRevRecoveryAgent(DocumentStore store, RevisionContext context) {
        this(store, context, new MissingLastRevSeeker(store, context.getClock()), i -> {});
    }

    public int recover(int clusterId, long waitUntil) throws DocumentStoreException {
        ClusterNodeInfoDocument nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
        if (nodeInfo != null && nodeInfo.isRecoveryNeeded(this.revisionContext.getClock().getTime())) {
            String reason;
            long startTime;
            NodeDocument root = this.missingLastRevUtil.getRoot();
            Revision lastRev = root.getLastRev().get(clusterId);
            Revision sweepRev = root.getSweepRevisions().getRevision(clusterId);
            if (lastRev != null) {
                startTime = lastRev.getTimestamp();
                reason = "lastRev: " + lastRev.toString();
            } else {
                startTime = nodeInfo.getStartTime();
                reason = String.format("no lastRev for root, using startTime %d", startTime);
            }
            if (sweepRev != null && sweepRev.getTimestamp() < startTime) {
                startTime = sweepRev.getTimestamp();
                reason = "sweepRev: " + sweepRev.toString();
            }
            return this.recoverCandidates(nodeInfo, startTime, waitUntil, reason);
        }
        this.log.debug("No recovery needed for clusterId {}", (Object)clusterId);
        return 0;
    }

    public int recover(int clusterId) throws DocumentStoreException {
        return this.recover(clusterId, 0L);
    }

    public int recover(Iterable<NodeDocument> suspects, int clusterId) throws DocumentStoreException {
        return this.recover(suspects, clusterId, false);
    }

    public int recover(Iterable<NodeDocument> suspects, final int clusterId, final boolean dryRun) throws DocumentStoreException {
        ClusterNodeInfoDocument nodeInfo;
        long deadline = Long.MAX_VALUE;
        if (clusterId == this.revisionContext.getClusterId() && (nodeInfo = this.missingLastRevUtil.getClusterNodeInfo(clusterId)) != null && nodeInfo.isActive()) {
            deadline = nodeInfo.getLeaseEndTime() - 20000L;
        }
        NodeDocument rootDoc = Utils.getRootDocument(this.store);
        final AtomicReference sweepRev = new AtomicReference();
        if (rootDoc.getSweepRevisions().getRevision(clusterId) != null) {
            final RecoveryContext context = new RecoveryContext(rootDoc, this.revisionContext.getClock(), clusterId, this.revisionContext::getCommitValue);
            NodeDocumentSweeper sweeper = new NodeDocumentSweeper(context, true);
            sweeper.sweep(suspects, new NodeDocumentSweepListener(){

                @Override
                public void sweepUpdate(Map<String, UpdateOp> updates) throws DocumentStoreException {
                    if (dryRun) {
                        LastRevRecoveryAgent.this.log.info("Dry run of sweeper identified [{}] documents for cluster node [{}]: {}", new Object[]{updates.size(), clusterId, updates.values()});
                        return;
                    }
                    JournalEntry inv = Collection.JOURNAL.newDocument(LastRevRecoveryAgent.this.store);
                    inv.modified(updates.keySet());
                    Revision r = context.newRevision().asBranchRevision();
                    UpdateOp invOp = inv.asUpdateOp(r);
                    JournalEntry entry = Collection.JOURNAL.newDocument(LastRevRecoveryAgent.this.store);
                    entry.invalidate(Collections.singleton(r));
                    Revision jRev = context.newRevision();
                    UpdateOp jOp = entry.asUpdateOp(jRev);
                    if (!LastRevRecoveryAgent.this.store.create(Collection.JOURNAL, Lists.newArrayList((Object[])new UpdateOp[]{invOp, jOp}))) {
                        String msg = "Unable to create journal entries for document invalidation.";
                        throw new DocumentStoreException(msg);
                    }
                    sweepRev.set(Utils.max((Revision)sweepRev.get(), jRev));
                    LastRevRecoveryAgent.this.store.createOrUpdate(Collection.NODES, Lists.newArrayList(updates.values()));
                    LastRevRecoveryAgent.this.log.info("Sweeper updated {}", updates.keySet());
                }
            });
        }
        UnsavedModifications unsaved = new UnsavedModifications();
        UnsavedModifications unsavedParents = new UnsavedModifications();
        ConcurrentMap<String, Revision> knownLastRevOrModification = MapFactory.getInstance().create();
        final JournalEntry changes = Collection.JOURNAL.newDocument(this.store);
        long count = 0L;
        for (NodeDocument doc : suspects) {
            if (++count % 100000L == 0L) {
                this.log.info("Scanned {} suspects so far...", (Object)count);
            }
            Revision currentLastRev = doc.getLastRev().get(clusterId);
            Revision lastModifiedRev = this.determineLastModification(doc, clusterId);
            Revision lastRevForParents = Utils.max(lastModifiedRev, currentLastRev);
            if (lastRevForParents != null) {
                knownLastRevOrModification.put(doc.getPath(), lastRevForParents);
            }
            if (lastRevForParents == null) continue;
            String path = doc.getPath();
            changes.modified(path);
            while (!PathUtils.denotesRoot((String)path)) {
                path = PathUtils.getParentPath((String)path);
                unsavedParents.put(path, lastRevForParents);
            }
        }
        for (String parentPath : unsavedParents.getPaths()) {
            Revision calcLastRev = unsavedParents.get(parentPath);
            Revision knownLastRev = (Revision)knownLastRevOrModification.get(parentPath);
            if (knownLastRev == null) {
                String id = Utils.getIdFromPath(parentPath);
                NodeDocument doc = this.store.find(Collection.NODES, id);
                if (doc != null) {
                    Revision lastRev = doc.getLastRev().get(clusterId);
                    Revision lastMod = this.determineLastModification(doc, clusterId);
                    knownLastRev = Utils.max(lastRev, lastMod);
                } else {
                    this.log.warn("Unable to find document: {}", (Object)id);
                    continue;
                }
            }
            if (knownLastRev != null && calcLastRev.compareRevisionTime(knownLastRev) <= 0) continue;
            unsaved.put(parentPath, calcLastRev);
        }
        if (sweepRev.get() != null) {
            unsaved.put("/", (Revision)sweepRev.get());
        }
        final Revision lastRootRev = unsaved.get("/");
        int size = unsaved.getPaths().size();
        String updates = unsaved.toString();
        if (dryRun) {
            this.log.info("Dry run of lastRev recovery identified [{}] documents for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        } else {
            if (this.revisionContext.getClock().getTime() > deadline) {
                String msg = String.format("Cluster node %d was unable to perform lastRev recovery for clusterId %d within deadline: %s", clusterId, clusterId, Utils.timestampToString(deadline));
                throw new DocumentStoreException(msg);
            }
            unsaved.persist(this.store, new Supplier<Revision>(){

                public Revision get() {
                    return (Revision)sweepRev.get();
                }
            }, new UnsavedModifications.Snapshot(){

                @Override
                public void acquiring(Revision mostRecent) {
                    if (lastRootRev == null) {
                        return;
                    }
                    String id = JournalEntry.asId(lastRootRev);
                    JournalEntry existingEntry = LastRevRecoveryAgent.this.store.find(Collection.JOURNAL, id);
                    if (existingEntry != null) {
                        return;
                    }
                    LastRevRecoveryAgent.this.store.create(Collection.JOURNAL, Collections.singletonList(changes.asUpdateOp(lastRootRev)));
                }
            }, new ReentrantLock());
            this.log.info("Updated lastRev of [{}] documents while performing lastRev recovery for cluster node [{}]: {}", new Object[]{size, clusterId, updates});
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int recoverCandidates(ClusterNodeInfoDocument nodeInfo, long startTime, long waitUntil, String info) throws DocumentStoreException {
        ClusterNodeInfoDocument infoDoc = nodeInfo;
        int clusterId = infoDoc.getClusterId();
        while (!this.missingLastRevUtil.acquireRecoveryLock(clusterId, this.revisionContext.getClusterId())) {
            Clock clock = this.revisionContext.getClock();
            long remaining = waitUntil - clock.getTime();
            if (remaining < 0L) {
                return -1;
            }
            this.log.info("Last revision recovery already being performed by cluster node {}. Waiting at most until {} for recovery to finish ({} seconds remaining).", new Object[]{infoDoc.getRecoveryBy(), Utils.timestampToString(waitUntil), remaining / 1000L});
            long time = Math.min(waitUntil, clock.getTime() + 5000L);
            try {
                clock.waitUntil(time);
            }
            catch (InterruptedException e) {
                Thread.interrupted();
                String msg = "Interrupted while waiting for _lastRev recovery to finish.";
                throw new DocumentStoreException(msg, e);
            }
            infoDoc = this.missingLastRevUtil.getClusterNodeInfo(clusterId);
            if (infoDoc == null) {
                String msg = String.format("No cluster node info document for id %d", clusterId);
                throw new DocumentStoreException(msg);
            }
            if (infoDoc.isRecoveryNeeded(clock.getTime())) continue;
            return 0;
        }
        boolean success = false;
        try {
            int n;
            this.log.info("Recovering candidates modified after: [{}] for clusterId [{}] [{}]", new Object[]{Utils.timestampToString(startTime), clusterId, info});
            Iterable<NodeDocument> suspects = this.missingLastRevUtil.getCandidates(startTime);
            try {
                this.log.info("Performing Last Revision Recovery for clusterNodeId {}", (Object)clusterId);
                int num = this.recover(suspects, clusterId);
                success = true;
                n = num;
            }
            catch (Throwable throwable) {
                Utils.closeIfCloseable(suspects);
                throw throwable;
            }
            Utils.closeIfCloseable(suspects);
            return n;
        }
        finally {
            this.missingLastRevUtil.releaseRecoveryLock(clusterId, success);
            this.afterRecovery.accept(clusterId);
        }
    }

    @Nullable
    private Revision determineLastModification(NodeDocument doc, int clusterId) {
        ClusterPredicate cp = new ClusterPredicate(clusterId);
        Revision lastModified = null;
        block0: for (String property : Sets.filter(doc.keySet(), Utils.PROPERTY_OR_DELETED)) {
            SortedMap<Revision, String> valueMap = doc.getLocalMap(property);
            for (Map.Entry entry : Maps.filterKeys(valueMap, (Predicate)cp).entrySet()) {
                Revision rev = (Revision)entry.getKey();
                String cv = this.revisionContext.getCommitValue(rev, doc);
                if (!Utils.isCommitted(cv)) continue;
                lastModified = Utils.max(lastModified, Utils.resolveCommitRevision(rev, cv));
                continue block0;
            }
        }
        return lastModified;
    }

    public boolean isRecoveryNeeded() {
        return this.missingLastRevUtil.isRecoveryNeeded();
    }

    public void performRecoveryIfNeeded() {
        if (this.isRecoveryNeeded()) {
            Iterable<Integer> clusterIds = this.getRecoveryCandidateNodes();
            this.log.info("ClusterNodeId [{}] starting Last Revision Recovery for clusterNodeId(s) {}", (Object)this.revisionContext.getClusterId(), clusterIds);
            for (int clusterId : clusterIds) {
                if (this.recover(clusterId) != -1) continue;
                this.log.info("Last Revision Recovery for cluster node {} ongoing by other cluster node.", (Object)clusterId);
            }
        }
    }

    public Iterable<Integer> getRecoveryCandidateNodes() {
        return Iterables.transform((Iterable)Iterables.filter(this.missingLastRevUtil.getAllClusters(), (Predicate)new Predicate<ClusterNodeInfoDocument>(){

            public boolean apply(ClusterNodeInfoDocument input) {
                return LastRevRecoveryAgent.this.revisionContext.getClusterId() != input.getClusterId() && input.isRecoveryNeeded(LastRevRecoveryAgent.this.revisionContext.getClock().getTime());
            }
        }), ClusterNodeInfoDocument::getClusterId);
    }

    private static class ClusterPredicate
    implements Predicate<Revision> {
        private final int clusterId;

        private ClusterPredicate(int clusterId) {
            this.clusterId = clusterId;
        }

        public boolean apply(Revision input) {
            return this.clusterId == input.getClusterId();
        }
    }
}

