/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.handler.admin;

import java.lang.invoke.MethodHandles;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.solr.cloud.LeaderElector;
import org.apache.solr.cloud.OverseerTaskProcessor;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.Replica;
import org.apache.solr.common.cloud.Slice;
import org.apache.solr.common.cloud.ZkNodeProps;
import org.apache.solr.common.cloud.ZkStateReader;
import org.apache.solr.common.params.CollectionParams;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.handler.admin.CollectionsHandler;
import org.apache.solr.request.SolrQueryRequest;
import org.apache.solr.response.SolrQueryResponse;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class RebalanceLeaders {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    final SolrQueryRequest req;
    final SolrQueryResponse rsp;
    final CollectionsHandler collectionsHandler;
    final CoreContainer coreContainer;

    RebalanceLeaders(SolrQueryRequest req, SolrQueryResponse rsp, CollectionsHandler collectionsHandler) {
        this.req = req;
        this.rsp = rsp;
        this.collectionsHandler = collectionsHandler;
        this.coreContainer = collectionsHandler.getCoreContainer();
    }

    void execute() throws KeeperException, InterruptedException {
        this.req.getParams().required().check("collection");
        String collectionName = this.req.getParams().get("collection");
        if (StringUtils.isBlank(collectionName)) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, String.format(Locale.ROOT, "The collection is required for the Rebalance Leaders command.", new Object[0]));
        }
        this.coreContainer.getZkController().getZkStateReader().updateClusterState();
        ClusterState clusterState = this.coreContainer.getZkController().getClusterState();
        DocCollection dc = clusterState.getCollection(collectionName);
        if (dc == null) {
            throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Collection '" + collectionName + "' does not exist, no action taken.");
        }
        HashMap<String, String> currentRequests = new HashMap<String, String>();
        int max = this.req.getParams().getInt("maxAtOnce", Integer.MAX_VALUE);
        if (max <= 0) {
            max = Integer.MAX_VALUE;
        }
        int maxWaitSecs = this.req.getParams().getInt("maxWaitSeconds", 60);
        NamedList<Object> results = new NamedList<Object>();
        boolean keepGoing = true;
        for (Slice slice : dc.getSlices()) {
            this.ensurePreferredIsLeader(results, slice, currentRequests);
            if (currentRequests.size() != max) continue;
            log.info("Queued " + max + " leader reassignments, waiting for some to complete.");
            keepGoing = this.waitForLeaderChange(currentRequests, maxWaitSecs, false, results);
            if (keepGoing) continue;
            break;
        }
        if (keepGoing) {
            keepGoing = this.waitForLeaderChange(currentRequests, maxWaitSecs, true, results);
        }
        if (keepGoing) {
            log.info("All leader reassignments completed.");
        } else {
            log.warn("Exceeded specified timeout of ." + maxWaitSecs + "' all leaders may not have been reassigned");
        }
        this.rsp.getValues().addAll(results);
    }

    private void ensurePreferredIsLeader(NamedList<Object> results, Slice slice, Map<String, String> currentRequests) throws KeeperException, InterruptedException {
        String inactivePreferreds = "inactivePreferreds";
        String alreadyLeaders = "alreadyLeaders";
        String collectionName = this.req.getParams().get("collection");
        for (Replica replica : slice.getReplicas()) {
            if (!replica.getBool("property.preferredleader", false)) continue;
            if (replica.getBool("leader", false)) {
                NamedList noops = (NamedList)results.get("alreadyLeaders");
                if (noops == null) {
                    noops = new NamedList();
                    results.add("alreadyLeaders", noops);
                }
                NamedList<String> res = new NamedList<String>();
                res.add("status", "success");
                res.add("msg", "Already leader");
                res.add("shard", slice.getName());
                res.add("nodeName", replica.getNodeName());
                noops.add(replica.getName(), res);
                return;
            }
            if (replica.getState() != Replica.State.ACTIVE) {
                NamedList inactives = (NamedList)results.get("inactivePreferreds");
                if (inactives == null) {
                    inactives = new NamedList();
                    results.add("inactivePreferreds", inactives);
                }
                NamedList<String> res = new NamedList<String>();
                res.add("status", "skipped");
                res.add("msg", "Node is a referredLeader, but it's inactive. Skipping");
                res.add("shard", slice.getName());
                res.add("nodeName", replica.getNodeName());
                inactives.add(replica.getName(), res);
                return;
            }
            ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
            List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath(collectionName, slice.getName()));
            if (electionNodes.size() < 2) {
                log.info("Rebalancing leaders and slice " + slice.getName() + " has less than two elements in the leader " + "election queue, but replica " + replica.getName() + " doesn't think it's the leader.");
                return;
            }
            String firstWatcher = electionNodes.get(1);
            if (!LeaderElector.getNodeName(firstWatcher).equals(replica.getName())) {
                this.makeReplicaFirstWatcher(collectionName, slice, replica);
            }
            String coreName = slice.getReplica(LeaderElector.getNodeName(electionNodes.get(0))).getStr("core");
            this.rejoinElection(collectionName, slice, electionNodes.get(0), coreName, false);
            this.waitForNodeChange(collectionName, slice, electionNodes.get(0));
            return;
        }
    }

    void makeReplicaFirstWatcher(String collectionName, Slice slice, Replica replica) throws KeeperException, InterruptedException {
        String coreName;
        ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
        List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath(collectionName, slice.getName()));
        int newSeq = -1;
        for (String electionNode : electionNodes) {
            if (!LeaderElector.getNodeName(electionNode).equals(replica.getName())) continue;
            coreName = slice.getReplica(LeaderElector.getNodeName(electionNode)).getStr("core");
            this.rejoinElection(collectionName, slice, electionNode, coreName, true);
            newSeq = this.waitForNodeChange(collectionName, slice, electionNode);
            break;
        }
        if (newSeq == -1) {
            return;
        }
        electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath(collectionName, slice.getName()));
        for (String thisNode : electionNodes) {
            if (LeaderElector.getSeq(thisNode) > newSeq) break;
            if (LeaderElector.getNodeName(thisNode).equals(replica.getName()) || LeaderElector.getSeq(thisNode) != newSeq) continue;
            coreName = slice.getReplica(LeaderElector.getNodeName(thisNode)).getStr("core");
            this.rejoinElection(collectionName, slice, thisNode, coreName, false);
            this.waitForNodeChange(collectionName, slice, thisNode);
        }
    }

    int waitForNodeChange(String collectionName, Slice slice, String electionNode) throws InterruptedException, KeeperException {
        String nodeName = LeaderElector.getNodeName(electionNode);
        int oldSeq = LeaderElector.getSeq(electionNode);
        for (int idx = 0; idx < 600; ++idx) {
            ZkStateReader zkStateReader = this.coreContainer.getZkController().getZkStateReader();
            List<String> electionNodes = OverseerTaskProcessor.getSortedElectionNodes(zkStateReader.getZkClient(), ZkStateReader.getShardLeadersElectPath(collectionName, slice.getName()));
            for (String testNode : electionNodes) {
                if (!LeaderElector.getNodeName(testNode).equals(nodeName) || oldSeq == LeaderElector.getSeq(testNode)) continue;
                return LeaderElector.getSeq(testNode);
            }
            Thread.sleep(100L);
        }
        return -1;
    }

    private void rejoinElection(String collectionName, Slice slice, String electionNode, String core, boolean rejoinAtHead) throws KeeperException, InterruptedException {
        Replica replica = slice.getReplica(LeaderElector.getNodeName(electionNode));
        HashMap<String, Object> propMap = new HashMap<String, Object>();
        propMap.put("collection", collectionName);
        propMap.put("shard", slice.getName());
        propMap.put("operation", CollectionParams.CollectionAction.REBALANCELEADERS.toLower());
        propMap.put("core", core);
        propMap.put("core_node_name", replica.getName());
        propMap.put("base_url", replica.getProperties().get("base_url"));
        propMap.put("rejoinAtHead", Boolean.toString(rejoinAtHead));
        propMap.put("election_node", electionNode);
        String asyncId = CollectionParams.CollectionAction.REBALANCELEADERS.toLower() + "_" + core + "_" + Math.abs(System.nanoTime());
        propMap.put("async", asyncId);
        ZkNodeProps m = new ZkNodeProps(propMap);
        SolrQueryResponse rspIgnore = new SolrQueryResponse();
        this.collectionsHandler.handleResponse(CollectionParams.CollectionAction.REBALANCELEADERS.toLower(), m, rspIgnore);
    }

    private boolean waitForLeaderChange(Map<String, String> currentAsyncIds, int maxWaitSecs, Boolean waitForAll, NamedList<Object> results) throws KeeperException, InterruptedException {
        if (currentAsyncIds.size() == 0) {
            return true;
        }
        for (int idx = 0; idx < maxWaitSecs * 10; ++idx) {
            Iterator<Map.Entry<String, String>> iter = currentAsyncIds.entrySet().iterator();
            boolean foundChange = false;
            while (iter.hasNext()) {
                NamedList<String> res;
                Map.Entry<String, String> pair = iter.next();
                String asyncId = pair.getKey();
                if (this.coreContainer.getZkController().getOverseerFailureMap().contains(asyncId)) {
                    this.coreContainer.getZkController().getOverseerFailureMap().remove(asyncId);
                    NamedList fails = (NamedList)results.get("failures");
                    if (fails == null) {
                        fails = new NamedList();
                        results.add("failures", fails);
                    }
                    res = new NamedList<String>();
                    res.add("status", "failed");
                    res.add("msg", "Failed to assign '" + pair.getValue() + "' to be leader");
                    fails.add(asyncId.substring(CollectionParams.CollectionAction.REBALANCELEADERS.toLower().length()), res);
                    iter.remove();
                    foundChange = true;
                    continue;
                }
                if (!this.coreContainer.getZkController().getOverseerCompletedMap().contains(asyncId)) continue;
                this.coreContainer.getZkController().getOverseerCompletedMap().remove(asyncId);
                NamedList<NamedList<String>> successes = (NamedList<NamedList<String>>)results.get("successes");
                if (successes == null) {
                    successes = new NamedList<NamedList<String>>();
                    results.add("successes", successes);
                }
                res = new NamedList();
                res.add("status", "success");
                res.add("msg", "Assigned '" + pair.getValue() + "' to be leader");
                successes.add(asyncId.substring(CollectionParams.CollectionAction.REBALANCELEADERS.toLower().length()), res);
                iter.remove();
                foundChange = true;
            }
            if (foundChange && !waitForAll.booleanValue() || currentAsyncIds.size() == 0) {
                return true;
            }
            Thread.sleep(100L);
        }
        return false;
    }
}

