/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.segment.file;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Stopwatch;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.common.io.Closer;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.jackrabbit.oak.api.jmx.CacheStatsMBean;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.ReferenceCollector;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.segment.Compactor;
import org.apache.jackrabbit.oak.segment.RecordId;
import org.apache.jackrabbit.oak.segment.Segment;
import org.apache.jackrabbit.oak.segment.SegmentId;
import org.apache.jackrabbit.oak.segment.SegmentNodeState;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundException;
import org.apache.jackrabbit.oak.segment.SegmentNotFoundExceptionListener;
import org.apache.jackrabbit.oak.segment.SegmentWriter;
import org.apache.jackrabbit.oak.segment.SegmentWriterBuilder;
import org.apache.jackrabbit.oak.segment.WriterCacheManager;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCOptions;
import org.apache.jackrabbit.oak.segment.compaction.SegmentGCStatus;
import org.apache.jackrabbit.oak.segment.file.AbstractFileStore;
import org.apache.jackrabbit.oak.segment.file.FileReaper;
import org.apache.jackrabbit.oak.segment.file.FileStoreBuilder;
import org.apache.jackrabbit.oak.segment.file.FileStoreStats;
import org.apache.jackrabbit.oak.segment.file.GCEstimation;
import org.apache.jackrabbit.oak.segment.file.GCJournal;
import org.apache.jackrabbit.oak.segment.file.GCListener;
import org.apache.jackrabbit.oak.segment.file.GCMemoryBarrier;
import org.apache.jackrabbit.oak.segment.file.GCNodeWriteMonitor;
import org.apache.jackrabbit.oak.segment.file.InvalidFileStoreVersionException;
import org.apache.jackrabbit.oak.segment.file.Manifest;
import org.apache.jackrabbit.oak.segment.file.SafeRunnable;
import org.apache.jackrabbit.oak.segment.file.Scheduler;
import org.apache.jackrabbit.oak.segment.file.SizeDeltaGcEstimation;
import org.apache.jackrabbit.oak.segment.file.TarReader;
import org.apache.jackrabbit.oak.segment.file.TarRevisions;
import org.apache.jackrabbit.oak.segment.file.TarWriter;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileStore
extends AbstractFileStore {
    private static final Logger log = LoggerFactory.getLogger(FileStore.class);
    private static final long GC_BACKOFF = Integer.getInteger("oak.gc.backoff", 36000000).intValue();
    private static final int MB = 0x100000;
    static final String LOCK_FILE_NAME = "repo.lock";
    private static final AtomicLong GC_COUNT = new AtomicLong(0L);
    @Nonnull
    private final SegmentWriter segmentWriter;
    private final int maxFileSize;
    @Nonnull
    private final GarbageCollector garbageCollector;
    private volatile List<TarReader> readers;
    private volatile TarWriter tarWriter;
    private final RandomAccessFile lockFile;
    private final FileLock lock;
    private TarRevisions revisions;
    private final Scheduler fileStoreScheduler = new Scheduler("FileStore background tasks");
    private final FileReaper fileReaper = new FileReaper();
    private final AtomicBoolean sufficientDiskSpace = new AtomicBoolean(true);
    private final AtomicBoolean sufficientMemory = new AtomicBoolean(true);
    private volatile boolean shutdown;
    private final ReadWriteLock fileStoreLock = new ReentrantReadWriteLock();
    private final FileStoreStats stats;
    @Nonnull
    private final SegmentNotFoundExceptionListener snfeListener;

    FileStore(final FileStoreBuilder builder) throws InvalidFileStoreVersionException, IOException {
        super(builder);
        this.lockFile = new RandomAccessFile(new File(this.directory, LOCK_FILE_NAME), "rw");
        try {
            this.lock = this.lockFile.getChannel().lock();
        }
        catch (OverlappingFileLockException ex) {
            throw new IllegalStateException(this.directory.getAbsolutePath() + " is in use by another store.", ex);
        }
        this.segmentWriter = SegmentWriterBuilder.segmentWriterBuilder("sys").withGeneration(new Supplier<Integer>(){

            public Integer get() {
                return FileStore.this.getGcGeneration();
            }
        }).withWriterPool().with(builder.getCacheManager()).build(this);
        this.maxFileSize = builder.getMaxFileSize() * 0x100000;
        this.garbageCollector = new GarbageCollector(builder.getGcOptions(), builder.getGcListener(), new GCJournal(this.directory), builder.getCacheManager());
        Map<Integer, Map<Character, File>> map = FileStore.collectFiles(this.directory);
        Manifest manifest = Manifest.empty();
        if (!map.isEmpty()) {
            manifest = FileStore.checkManifest(this.openManifest());
        }
        this.saveManifest(manifest);
        this.readers = Lists.newArrayListWithCapacity((int)map.size());
        Object[] indices = map.keySet().toArray(new Integer[map.size()]);
        Arrays.sort(indices);
        for (int i = indices.length - 1; i >= 0; --i) {
            this.readers.add(TarReader.open(map.get(indices[i]), this.memoryMapping, this.recovery));
        }
        this.stats = new FileStoreStats(builder.getStatsProvider(), this, this.size());
        int writeNumber = 0;
        if (indices.length > 0) {
            writeNumber = (Integer)indices[indices.length - 1] + 1;
        }
        this.tarWriter = new TarWriter(this.directory, this.stats, writeNumber);
        this.snfeListener = builder.getSnfeListener();
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK flush [%s]", this.directory), 5L, TimeUnit.SECONDS, new Runnable(){

            @Override
            public void run() {
                if (FileStore.this.shutdown) {
                    return;
                }
                try {
                    FileStore.this.flush();
                }
                catch (IOException e) {
                    log.warn("Failed to flush the TarMK at {}", (Object)FileStore.this.directory, (Object)e);
                }
            }
        });
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK filer reaper [%s]", this.directory), 5L, TimeUnit.SECONDS, new Runnable(){

            @Override
            public void run() {
                FileStore.this.fileReaper.reap();
            }
        });
        this.fileStoreScheduler.scheduleAtFixedRate(String.format("TarMK disk space check [%s]", this.directory), 1L, TimeUnit.MINUTES, new Runnable(){
            final SegmentGCOptions gcOptions;
            {
                this.gcOptions = builder.getGcOptions();
            }

            @Override
            public void run() {
                FileStore.this.checkDiskSpace(this.gcOptions);
            }
        });
        log.info("TarMK opened: {} (mmap={})", (Object)this.directory, (Object)this.memoryMapping);
        log.debug("TarMK readers {}", this.readers);
    }

    FileStore bind(TarRevisions revisions) throws IOException {
        this.revisions = revisions;
        this.revisions.bind(this, this.initialNode());
        return this;
    }

    private void saveManifest(Manifest manifest) throws IOException {
        manifest.setStoreVersion(1);
        manifest.save(this.getManifestFile());
    }

    @Nonnull
    private Supplier<RecordId> initialNode() {
        return new Supplier<RecordId>(){

            public RecordId get() {
                try {
                    SegmentWriter writer = SegmentWriterBuilder.segmentWriterBuilder("init").build(FileStore.this);
                    NodeBuilder builder = EmptyNodeState.EMPTY_NODE.builder();
                    builder.setChildNode("root", EmptyNodeState.EMPTY_NODE);
                    SegmentNodeState node = writer.writeNode(builder.getNodeState());
                    writer.flush();
                    return node.getRecordId();
                }
                catch (IOException e) {
                    String msg = "Failed to write initial node";
                    log.error(msg, (Throwable)e);
                    throw new IllegalStateException(msg, e);
                }
            }
        };
    }

    private int getGcGeneration() {
        return this.revisions.getHead().getSegmentId().getGcGeneration();
    }

    @CheckForNull
    public CacheStatsMBean getStringDeduplicationCacheStats() {
        return this.segmentWriter.getStringCacheStats();
    }

    @CheckForNull
    public CacheStatsMBean getTemplateDeduplicationCacheStats() {
        return this.segmentWriter.getTemplateCacheStats();
    }

    @CheckForNull
    public CacheStatsMBean getNodeDeduplicationCacheStats() {
        return this.segmentWriter.getNodeCacheStats();
    }

    public Runnable getGCRunner() {
        return new SafeRunnable(String.format("TarMK revision gc [%s]", this.directory), new Runnable(){

            @Override
            public void run() {
                try {
                    FileStore.this.gc();
                }
                catch (IOException e) {
                    log.error("Error running revision garbage collection", (Throwable)e);
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long size() {
        long writeFileSnapshotSize;
        ImmutableList readersSnapshot;
        this.fileStoreLock.readLock().lock();
        try {
            readersSnapshot = ImmutableList.copyOf(this.readers);
            writeFileSnapshotSize = this.tarWriter != null ? this.tarWriter.fileLength() : 0L;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
        long size = writeFileSnapshotSize;
        for (TarReader reader : readersSnapshot) {
            size += reader.size();
        }
        return size;
    }

    public int readerCount() {
        this.fileStoreLock.readLock().lock();
        try {
            int n = this.readers.size();
            return n;
        }
        finally {
            this.fileStoreLock.readLock().unlock();
        }
    }

    public FileStoreStats getStats() {
        return this.stats;
    }

    public void flush() throws IOException {
        if (this.revisions == null) {
            return;
        }
        this.revisions.flush(new Callable<Void>(){

            @Override
            public Void call() throws Exception {
                FileStore.this.segmentWriter.flush();
                FileStore.this.tarWriter.flush();
                FileStore.this.stats.flushed();
                return null;
            }
        });
    }

    public void gc() throws IOException {
        this.garbageCollector.run();
    }

    public GCEstimation estimateCompactionGain() {
        return this.garbageCollector.estimateCompactionGain((Supplier<Boolean>)Suppliers.ofInstance((Object)false));
    }

    public boolean compact() {
        return this.garbageCollector.compact() > 0;
    }

    public void cleanup() throws IOException {
        this.garbageCollector.cleanup();
    }

    public void collectBlobReferences(ReferenceCollector collector) throws IOException {
        this.garbageCollector.collectBlobReferences(collector);
    }

    public void cancelGC() {
        this.garbageCollector.cancel();
    }

    @Override
    @Nonnull
    public SegmentWriter getWriter() {
        return this.segmentWriter;
    }

    @Override
    @Nonnull
    public TarRevisions getRevisions() {
        return this.revisions;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.shutdown = true;
        this.fileStoreScheduler.close();
        try {
            this.flush();
        }
        catch (IOException e) {
            log.warn("Unable to flush the store", (Throwable)e);
        }
        Closer closer = Closer.create();
        closer.register((Closeable)this.revisions);
        this.fileStoreLock.writeLock().lock();
        try {
            if (this.lock != null) {
                try {
                    this.lock.release();
                }
                catch (IOException e) {
                    log.warn("Unable to release the file lock", (Throwable)e);
                }
            }
            closer.register((Closeable)this.lockFile);
            List<TarReader> list = this.readers;
            this.readers = Lists.newArrayList();
            for (TarReader reader : list) {
                closer.register((Closeable)reader);
            }
            closer.register((Closeable)this.tarWriter);
        }
        finally {
            this.fileStoreLock.writeLock().unlock();
        }
        FileStore.closeAndLogOnFail((Closeable)closer);
        this.fileReaper.reap();
        System.gc();
        log.info("TarMK closed: {}", (Object)this.directory);
    }

    @Override
    public boolean containsSegment(SegmentId id) {
        long msb = id.getMostSignificantBits();
        long lsb = id.getLeastSignificantBits();
        return this.containsSegment(msb, lsb);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean containsSegment(long msb, long lsb) {
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        if (this.tarWriter != null) {
            this.fileStoreLock.readLock().lock();
            try {
                if (this.tarWriter.containsEntry(msb, lsb)) {
                    boolean bl = true;
                    return bl;
                }
            }
            finally {
                this.fileStoreLock.readLock().unlock();
            }
        }
        for (TarReader reader : this.readers) {
            if (!reader.containsEntry(msb, lsb)) continue;
            return true;
        }
        return false;
    }

    @Override
    @Nonnull
    public Segment readSegment(final SegmentId id) {
        try {
            return this.segmentCache.getSegment(id, new Callable<Segment>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 * Enabled aggressive block sorting
                 * Enabled unnecessary exception pruning
                 * Enabled aggressive exception aggregation
                 */
                @Override
                public Segment call() throws Exception {
                    ByteBuffer buffer;
                    Object reader2;
                    long msb = id.getMostSignificantBits();
                    long lsb = id.getLeastSignificantBits();
                    for (Object reader2 : FileStore.this.readers) {
                        try {
                            if (((TarReader)reader2).isClosed()) {
                                log.debug("Skipping closed tar file {}", reader2);
                                continue;
                            }
                            buffer = ((TarReader)reader2).readEntry(msb, lsb);
                            if (buffer == null) continue;
                            return new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer);
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", reader2, (Object)e);
                        }
                    }
                    if (FileStore.this.tarWriter != null) {
                        FileStore.this.fileStoreLock.readLock().lock();
                        try {
                            ByteBuffer buffer2 = FileStore.this.tarWriter.readEntry(msb, lsb);
                            if (buffer2 != null) {
                                reader2 = new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer2);
                                FileStore.this.fileStoreLock.readLock().unlock();
                                return reader2;
                            }
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", (Object)FileStore.this.tarWriter, (Object)e);
                        }
                        catch (Throwable throwable) {
                            throw throwable;
                        }
                    }
                    Iterator iterator = FileStore.this.readers.iterator();
                    while (iterator.hasNext()) {
                        reader2 = (TarReader)iterator.next();
                        try {
                            if (((TarReader)reader2).isClosed()) {
                                log.info("Skipping closed tar file {}", reader2);
                                continue;
                            }
                            buffer = ((TarReader)reader2).readEntry(msb, lsb);
                            if (buffer == null) continue;
                            return new Segment(FileStore.this, FileStore.this.segmentReader, id, buffer);
                        }
                        catch (IOException e) {
                            log.warn("Failed to read from tar file {}", reader2, (Object)e);
                        }
                    }
                    throw new SegmentNotFoundException(id);
                }
            });
        }
        catch (ExecutionException e) {
            SegmentNotFoundException snfe = e.getCause() instanceof SegmentNotFoundException ? (SegmentNotFoundException)((Object)e.getCause()) : new SegmentNotFoundException(id, e);
            this.snfeListener.notify(id, snfe);
            throw snfe;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void writeSegment(SegmentId id, byte[] buffer, int offset, int length) throws IOException {
        Segment segment = null;
        int generation = 0;
        if (id.isDataSegmentId()) {
            ByteBuffer data;
            if (offset > 4096) {
                data = ByteBuffer.allocate(length);
                data.put(buffer, offset, length);
                data.rewind();
            } else {
                data = ByteBuffer.wrap(buffer, offset, length);
            }
            segment = new Segment(this, this.segmentReader, id, data);
            generation = segment.getGcGeneration();
        }
        this.fileStoreLock.writeLock().lock();
        try {
            long size = this.tarWriter.writeEntry(id.getMostSignificantBits(), id.getLeastSignificantBits(), buffer, offset, length, generation);
            if (segment != null) {
                FileStore.populateTarGraph(segment, this.tarWriter);
                FileStore.populateTarBinaryReferences(segment, this.tarWriter);
            }
            if (size >= (long)this.maxFileSize) {
                this.newWriter();
            }
        }
        finally {
            this.fileStoreLock.writeLock().unlock();
        }
        if (segment != null) {
            this.segmentCache.putSegment(segment);
        }
    }

    private void newWriter() throws IOException {
        TarWriter newWriter = this.tarWriter.createNextGeneration();
        if (newWriter != this.tarWriter) {
            File writeFile = this.tarWriter.getFile();
            ArrayList list = Lists.newArrayListWithCapacity((int)(1 + this.readers.size()));
            list.add(TarReader.open(writeFile, this.memoryMapping));
            list.addAll(this.readers);
            this.readers = list;
            this.tarWriter = newWriter;
        }
    }

    private void checkDiskSpace(SegmentGCOptions gcOptions) {
        long availableDiskSpace;
        long repositoryDiskSpace = this.size();
        boolean updated = SegmentGCOptions.isDiskSpaceSufficient(repositoryDiskSpace, availableDiskSpace = this.directory.getFreeSpace());
        boolean previous = this.sufficientDiskSpace.getAndSet(updated);
        if (previous && !updated) {
            log.warn("Available disk space ({}) is too low, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount((long)availableDiskSpace), (Object)IOUtils.humanReadableByteCount((long)repositoryDiskSpace));
        }
        if (updated && !previous) {
            log.info("Available disk space ({}) is sufficient again for repository operations, current repository size is approx. {}", (Object)IOUtils.humanReadableByteCount((long)availableDiskSpace), (Object)IOUtils.humanReadableByteCount((long)repositoryDiskSpace));
        }
    }

    private class GarbageCollector {
        @Nonnull
        private final SegmentGCOptions gcOptions;
        @Nonnull
        private final GCListener gcListener;
        @Nonnull
        private final GCJournal gcJournal;
        @Nonnull
        private final WriterCacheManager cacheManager;
        @Nonnull
        private final GCNodeWriteMonitor compactionMonitor;
        private volatile boolean cancelled;
        private long lastSuccessfullGC;

        GarbageCollector(@Nonnull SegmentGCOptions gcOptions, @Nonnull GCListener gcListener, @Nonnull GCJournal gcJournal, WriterCacheManager cacheManager) {
            this.gcOptions = gcOptions;
            this.gcListener = gcListener;
            this.gcJournal = gcJournal;
            this.cacheManager = cacheManager;
            this.compactionMonitor = gcOptions.getGCNodeWriteMonitor();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        synchronized void run() throws IOException {
            try {
                this.gcListener.info("TarMK GC #{}: started", new Object[]{GC_COUNT.incrementAndGet()});
                long dt = System.currentTimeMillis() - this.lastSuccessfullGC;
                if (dt < GC_BACKOFF) {
                    this.gcListener.skipped("TarMK GC #{}: skipping garbage collection as it already ran less than {} hours ago ({} s).", new Object[]{GC_COUNT, GC_BACKOFF / 3600000L, dt / 1000L});
                    return;
                }
                GCMemoryBarrier gcMemoryBarrier = new GCMemoryBarrier(FileStore.this.sufficientMemory, this.gcListener, GC_COUNT.get(), this.gcOptions);
                boolean sufficientEstimatedGain = true;
                if (this.gcOptions.isEstimationDisabled()) {
                    this.gcListener.info("TarMK GC #{}: estimation skipped because it was explicitly disabled", new Object[]{GC_COUNT});
                } else if (this.gcOptions.isPaused()) {
                    this.gcListener.info("TarMK GC #{}: estimation skipped because compaction is paused", new Object[]{GC_COUNT});
                } else {
                    this.gcListener.info("TarMK GC #{}: estimation started", new Object[]{GC_COUNT});
                    this.gcListener.updateStatus(SegmentGCStatus.ESTIMATION.message());
                    Stopwatch watch = Stopwatch.createStarted();
                    CancelCompactionSupplier cancel = new CancelCompactionSupplier(FileStore.this);
                    GCEstimation estimate = this.estimateCompactionGain(cancel);
                    if (((Boolean)cancel.get()).booleanValue()) {
                        this.gcListener.warn("TarMK GC #{}: estimation interrupted: {}. Skipping garbage collection.", new Object[]{GC_COUNT, cancel});
                        gcMemoryBarrier.close();
                        return;
                    }
                    sufficientEstimatedGain = estimate.gcNeeded();
                    String gcLog = estimate.gcLog();
                    if (sufficientEstimatedGain) {
                        this.gcListener.info("TarMK GC #{}: estimation completed in {} ({} ms). {}", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), gcLog});
                    } else {
                        this.gcListener.skipped("TarMK GC #{}: estimation completed in {} ({} ms). {}", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), gcLog});
                    }
                }
                if (sufficientEstimatedGain) {
                    if (!this.gcOptions.isPaused()) {
                        this.log(FileStore.this.segmentWriter.getNodeCacheOccupancyInfo());
                        int gen = this.compact();
                        if (gen > 0) {
                            FileStore.this.fileReaper.add(this.cleanupOldGenerations(gen));
                            this.lastSuccessfullGC = System.currentTimeMillis();
                        } else if (gen < 0) {
                            this.gcListener.info("TarMK GC #{}: cleaning up after failed compaction", new Object[]{GC_COUNT});
                            FileStore.this.fileReaper.add(this.cleanupGeneration(-gen));
                        }
                        this.log(FileStore.this.segmentWriter.getNodeCacheOccupancyInfo());
                    } else {
                        this.gcListener.skipped("TarMK GC #{}: compaction paused", new Object[]{GC_COUNT});
                    }
                }
                gcMemoryBarrier.close();
            }
            finally {
                this.compactionMonitor.finished();
                this.gcListener.updateStatus(SegmentGCStatus.IDLE.message());
            }
        }

        synchronized GCEstimation estimateCompactionGain(Supplier<Boolean> stop) {
            return new SizeDeltaGcEstimation(this.gcOptions, this.gcJournal, FileStore.this.stats.getApproximateSize());
        }

        private void log(@CheckForNull String nodeCacheOccupancyInfo) {
            if (nodeCacheOccupancyInfo != null) {
                log.info("NodeCache occupancy: {}", (Object)nodeCacheOccupancyInfo);
            }
        }

        private int compactionAborted(int generation) {
            this.gcListener.compactionFailed(generation);
            return -generation;
        }

        private int compactionSucceeded(int generation) {
            this.gcListener.compactionSucceeded(generation);
            return generation;
        }

        synchronized int compact() {
            int newGeneration = FileStore.this.getGcGeneration() + 1;
            try {
                Stopwatch watch = Stopwatch.createStarted();
                this.gcListener.info("TarMK GC #{}: compaction started, gc options={}", new Object[]{GC_COUNT, this.gcOptions});
                this.gcListener.updateStatus(SegmentGCStatus.COMPACTION.message());
                GCJournal.GCJournalEntry gcEntry = this.gcJournal.read();
                long initialSize = FileStore.this.size();
                this.compactionMonitor.init(GC_COUNT.get(), gcEntry.getRepoSize(), gcEntry.getNodes(), initialSize);
                SegmentNodeState before = FileStore.this.getHead();
                CancelCompactionSupplier cancel = new CancelCompactionSupplier(FileStore.this);
                SegmentWriter writer = SegmentWriterBuilder.segmentWriterBuilder("c").with(this.cacheManager).withGeneration(newGeneration).withoutWriterPool().build(FileStore.this);
                writer.setCompactionMonitor(this.compactionMonitor);
                SegmentNodeState after = this.compact(before, writer, cancel);
                if (after == null) {
                    this.gcListener.warn("TarMK GC #{}: compaction cancelled: {}.", new Object[]{GC_COUNT, cancel});
                    return this.compactionAborted(newGeneration);
                }
                this.gcListener.info("TarMK GC #{}: compaction cycle 0 completed in {} ({} ms). Compacted {} to {}", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), before.getRecordId(), after.getRecordId()});
                int cycles = 0;
                boolean success = false;
                while (cycles < this.gcOptions.getRetryCount() && !(success = FileStore.this.revisions.setHead(before.getRecordId(), after.getRecordId(), TarRevisions.EXPEDITE_OPTION))) {
                    this.gcListener.info("TarMK GC #{}: compaction detected concurrent commits while compacting. Compacting these commits. Cycle {} of {}", new Object[]{GC_COUNT, ++cycles, this.gcOptions.getRetryCount()});
                    this.gcListener.updateStatus(SegmentGCStatus.COMPACTION_RETRY.message() + cycles);
                    Stopwatch cycleWatch = Stopwatch.createStarted();
                    SegmentNodeState head = FileStore.this.getHead();
                    after = this.compact(head, writer, cancel);
                    if (after == null) {
                        this.gcListener.warn("TarMK GC #{}: compaction cancelled: {}.", new Object[]{GC_COUNT, cancel});
                        return this.compactionAborted(newGeneration);
                    }
                    this.gcListener.info("TarMK GC #{}: compaction cycle {} completed in {} ({} ms). Compacted {} against {} to {}", new Object[]{GC_COUNT, cycles, cycleWatch, cycleWatch.elapsed(TimeUnit.MILLISECONDS), head.getRecordId(), before.getRecordId(), after.getRecordId()});
                    before = head;
                }
                if (!success) {
                    this.gcListener.info("TarMK GC #{}: compaction gave up compacting concurrent commits after {} cycles.", new Object[]{GC_COUNT, cycles});
                    int forceTimeout = this.gcOptions.getForceTimeout();
                    if (forceTimeout > 0) {
                        this.gcListener.info("TarMK GC #{}: trying to force compact remaining commits for {} seconds. Concurrent commits to the store will be blocked.", new Object[]{GC_COUNT, forceTimeout});
                        this.gcListener.updateStatus(SegmentGCStatus.COMPACTION_FORCE_COMPACT.message());
                        Stopwatch forceWatch = Stopwatch.createStarted();
                        ++cycles;
                        success = this.forceCompact(writer, this.or(cancel, this.timeOut(forceTimeout, TimeUnit.SECONDS)));
                        if (success) {
                            this.gcListener.info("TarMK GC #{}: compaction succeeded to force compact remaining commits after {} ({} ms).", new Object[]{GC_COUNT, forceWatch, forceWatch.elapsed(TimeUnit.MILLISECONDS)});
                        } else if (((Boolean)cancel.get()).booleanValue()) {
                            this.gcListener.warn("TarMK GC #{}: compaction failed to force compact remaining commits after {} ({} ms). Compaction was cancelled: {}.", new Object[]{GC_COUNT, forceWatch, forceWatch.elapsed(TimeUnit.MILLISECONDS), cancel});
                        } else {
                            this.gcListener.warn("TarMK GC #{}: compaction failed to force compact remaining commits. after {} ({} ms). Most likely compaction didn't get exclusive access to the store.", new Object[]{GC_COUNT, forceWatch, forceWatch.elapsed(TimeUnit.MILLISECONDS)});
                        }
                    }
                }
                if (success) {
                    writer.flush();
                    this.gcListener.info("TarMK GC #{}: compaction succeeded in {} ({} ms), after {} cycles", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), cycles});
                    return this.compactionSucceeded(newGeneration);
                }
                this.gcListener.info("TarMK GC #{}: compaction failed after {} ({} ms), and {} cycles", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), cycles});
                return this.compactionAborted(newGeneration);
            }
            catch (InterruptedException e) {
                this.gcListener.error("TarMK GC #" + GC_COUNT + ": compaction interrupted", e);
                Thread.currentThread().interrupt();
                return this.compactionAborted(newGeneration);
            }
            catch (IOException e) {
                this.gcListener.error("TarMK GC #" + GC_COUNT + ": compaction encountered an error", e);
                return this.compactionAborted(newGeneration);
            }
        }

        private Supplier<Boolean> timeOut(final long duration, final @Nonnull TimeUnit unit) {
            return new Supplier<Boolean>(){
                final long deadline;
                {
                    this.deadline = System.currentTimeMillis() + TimeUnit.MILLISECONDS.convert(duration, unit);
                }

                public Boolean get() {
                    return System.currentTimeMillis() > this.deadline;
                }
            };
        }

        private Supplier<Boolean> or(@Nonnull Supplier<Boolean> supplier1, @Nonnull Supplier<Boolean> supplier2) {
            if (((Boolean)supplier1.get()).booleanValue()) {
                return Suppliers.ofInstance((Object)true);
            }
            return supplier2;
        }

        private SegmentNodeState compact(NodeState head, SegmentWriter writer, Supplier<Boolean> cancel) throws IOException {
            if (this.gcOptions.isOffline()) {
                return new Compactor(FileStore.this.segmentReader, writer, FileStore.this.getBlobStore(), cancel, this.gcOptions).compact(EmptyNodeState.EMPTY_NODE, head, EmptyNodeState.EMPTY_NODE);
            }
            return writer.writeNode(head, cancel);
        }

        private boolean forceCompact(final @Nonnull SegmentWriter writer, final @Nonnull Supplier<Boolean> cancel) throws InterruptedException {
            return FileStore.this.revisions.setHead(new Function<RecordId, RecordId>(){

                @Nullable
                public RecordId apply(RecordId base) {
                    try {
                        long t0 = System.currentTimeMillis();
                        SegmentNodeState after = GarbageCollector.this.compact(FileStore.this.segmentReader.readNode(base), writer, (Supplier<Boolean>)cancel);
                        if (after == null) {
                            GarbageCollector.this.gcListener.info("TarMK GC #{}: compaction cancelled after {} seconds", new Object[]{GC_COUNT, (System.currentTimeMillis() - t0) / 1000L});
                            return null;
                        }
                        return after.getRecordId();
                    }
                    catch (IOException e) {
                        GarbageCollector.this.gcListener.error("TarMK GC #{" + GC_COUNT + "}: Error during forced compaction.", e);
                        return null;
                    }
                }
            }, TarRevisions.timeout(this.gcOptions.getForceTimeout(), TimeUnit.SECONDS));
        }

        synchronized void cleanup() throws IOException {
            FileStore.this.fileReaper.add(this.cleanupOldGenerations(FileStore.this.getGcGeneration()));
        }

        private List<File> cleanupOldGenerations(int gcGeneration) throws IOException {
            final int reclaimGeneration = gcGeneration - this.gcOptions.getRetainedGenerations();
            Predicate<Integer> reclaimPredicate = new Predicate<Integer>(){

                public boolean apply(Integer generation) {
                    return generation <= reclaimGeneration;
                }
            };
            return this.cleanup(reclaimPredicate, "gc-count=" + GC_COUNT + ",gc-status=success,store-generation=" + gcGeneration + ",reclaim-predicate=(generation<=" + reclaimGeneration + ")");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private List<File> cleanup(@Nonnull Predicate<Integer> reclaimGeneration, @Nonnull String gcInfo) throws IOException {
            Stopwatch watch = Stopwatch.createStarted();
            HashSet bulkRefs = Sets.newHashSet();
            LinkedHashMap cleaned = Maps.newLinkedHashMap();
            long initialSize = 0L;
            FileStore.this.fileStoreLock.writeLock().lock();
            try {
                this.gcListener.info("TarMK GC #{}: cleanup started.", new Object[]{GC_COUNT});
                this.gcListener.updateStatus(SegmentGCStatus.CLEANUP.message());
                FileStore.this.newWriter();
                FileStore.this.segmentCache.clear();
                System.gc();
                this.collectBulkReferences(bulkRefs);
                for (Object reader : FileStore.this.readers) {
                    cleaned.put(reader, reader);
                    initialSize += ((TarReader)reader).size();
                }
            }
            finally {
                FileStore.this.fileStoreLock.writeLock().unlock();
            }
            this.gcListener.info("TarMK GC #{}: current repository size is {} ({} bytes)", new Object[]{GC_COUNT, IOUtils.humanReadableByteCount((long)initialSize), initialSize});
            HashSet reclaim = Sets.newHashSet();
            for (Object reader : cleaned.keySet()) {
                ((TarReader)reader).mark(bulkRefs, reclaim, reclaimGeneration);
                log.info("{}: size of bulk references/reclaim set {}/{}", new Object[]{reader, bulkRefs.size(), reclaim.size()});
                if (!FileStore.this.shutdown) continue;
                this.gcListener.info("TarMK GC #{}: cleanup interrupted", new Object[]{GC_COUNT});
                break;
            }
            HashSet reclaimed = Sets.newHashSet();
            for (TarReader reader : cleaned.keySet()) {
                cleaned.put(reader, reader.sweep(reclaim, reclaimed));
                if (!FileStore.this.shutdown) continue;
                this.gcListener.info("TarMK GC #{}: cleanup interrupted", new Object[]{GC_COUNT});
                break;
            }
            long afterCleanupSize = 0L;
            ArrayList oldReaders = Lists.newArrayList();
            FileStore.this.fileStoreLock.writeLock().lock();
            try {
                ArrayList sweptReaders = Lists.newArrayList();
                for (TarReader reader : FileStore.this.readers) {
                    if (cleaned.containsKey(reader)) {
                        TarReader newReader = (TarReader)cleaned.get(reader);
                        if (newReader != null) {
                            sweptReaders.add(newReader);
                            afterCleanupSize += newReader.size();
                        }
                        if (newReader == reader) continue;
                        oldReaders.add(reader);
                        continue;
                    }
                    sweptReaders.add(reader);
                }
                FileStore.this.readers = sweptReaders;
            }
            finally {
                FileStore.this.fileStoreLock.writeLock().unlock();
            }
            FileStore.this.tracker.clearSegmentIdTables(reclaimed, gcInfo);
            LinkedList toRemove = Lists.newLinkedList();
            for (TarReader oldReader : oldReaders) {
                AbstractFileStore.closeAndLogOnFail(oldReader);
                File file = oldReader.getFile();
                toRemove.addLast(file);
            }
            this.gcListener.info("TarMK GC #{}: cleanup marking files for deletion: {}", new Object[]{GC_COUNT, this.toFileNames(toRemove)});
            long finalSize = FileStore.this.size();
            long reclaimedSize = initialSize - afterCleanupSize;
            FileStore.this.stats.reclaimed(reclaimedSize);
            this.gcJournal.persist(reclaimedSize, finalSize, FileStore.this.getGcGeneration(), this.compactionMonitor.getCompactedNodes());
            this.gcListener.cleaned(reclaimedSize, finalSize);
            this.gcListener.info("TarMK GC #{}: cleanup completed in {} ({} ms). Post cleanup size is {} ({} bytes) and space reclaimed {} ({} bytes).", new Object[]{GC_COUNT, watch, watch.elapsed(TimeUnit.MILLISECONDS), IOUtils.humanReadableByteCount((long)finalSize), finalSize, IOUtils.humanReadableByteCount((long)reclaimedSize), reclaimedSize});
            return toRemove;
        }

        private String toFileNames(@Nonnull List<File> files) {
            if (files.isEmpty()) {
                return "none";
            }
            return Joiner.on((String)",").join(files);
        }

        private void collectBulkReferences(Set<UUID> bulkRefs) {
            HashSet refs = Sets.newHashSet();
            for (SegmentId id : FileStore.this.tracker.getReferencedSegmentIds()) {
                refs.add(id.asUUID());
            }
            FileStore.this.tarWriter.collectReferences(refs);
            for (UUID ref : refs) {
                if (SegmentId.isDataSegmentId(ref.getLeastSignificantBits())) continue;
                bulkRefs.add(ref);
            }
        }

        private List<File> cleanupGeneration(final int gcGeneration) throws IOException {
            Predicate<Integer> cleanupPredicate = new Predicate<Integer>(){

                public boolean apply(Integer generation) {
                    return generation == gcGeneration;
                }
            };
            return this.cleanup(cleanupPredicate, "gc-count=" + GC_COUNT + ",gc-status=failed,store-generation=" + (gcGeneration - 1) + ",reclaim-predicate=(generation==" + gcGeneration + ")");
        }

        synchronized void collectBlobReferences(ReferenceCollector collector) throws IOException {
            FileStore.this.segmentWriter.flush();
            ArrayList tarReaders = Lists.newArrayList();
            FileStore.this.fileStoreLock.writeLock().lock();
            try {
                FileStore.this.newWriter();
                tarReaders.addAll(FileStore.this.readers);
            }
            finally {
                FileStore.this.fileStoreLock.writeLock().unlock();
            }
            int minGeneration = FileStore.this.getGcGeneration() - this.gcOptions.getRetainedGenerations() + 1;
            for (TarReader tarReader : tarReaders) {
                tarReader.collectBlobReferences(collector, minGeneration);
            }
        }

        void cancel() {
            this.cancelled = true;
        }

        private class CancelCompactionSupplier
        implements Supplier<Boolean> {
            private final FileStore store;
            private String reason;

            public CancelCompactionSupplier(FileStore store) {
                GarbageCollector.this.cancelled = false;
                this.store = store;
            }

            public Boolean get() {
                if (!this.store.sufficientDiskSpace.get()) {
                    this.reason = "Not enough disk space";
                    return true;
                }
                if (!this.store.sufficientMemory.get()) {
                    this.reason = "Not enough memory";
                    return true;
                }
                if (this.store.shutdown) {
                    this.reason = "The FileStore is shutting down";
                    return true;
                }
                if (GarbageCollector.this.cancelled) {
                    this.reason = "Cancelled by user";
                    return true;
                }
                return false;
            }

            public String toString() {
                return this.reason;
            }
        }
    }
}

