/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.jcr.repository;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.io.Closeable;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.jcr.Credentials;
import javax.jcr.LoginException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.SimpleCredentials;
import javax.jcr.Value;
import javax.jcr.ValueFactory;
import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.api.JackrabbitRepository;
import org.apache.jackrabbit.api.security.authentication.token.TokenCredentials;
import org.apache.jackrabbit.commons.SimpleValueFactory;
import org.apache.jackrabbit.oak.api.ContentRepository;
import org.apache.jackrabbit.oak.api.ContentSession;
import org.apache.jackrabbit.oak.api.jmx.SessionMBean;
import org.apache.jackrabbit.oak.jcr.delegate.SessionDelegate;
import org.apache.jackrabbit.oak.jcr.repository.JcrDescriptorsImpl;
import org.apache.jackrabbit.oak.jcr.session.RefreshStrategy;
import org.apache.jackrabbit.oak.jcr.session.SessionContext;
import org.apache.jackrabbit.oak.jcr.session.SessionStats;
import org.apache.jackrabbit.oak.plugins.observation.CommitRateLimiter;
import org.apache.jackrabbit.oak.spi.gc.DelegatingGCMonitor;
import org.apache.jackrabbit.oak.spi.gc.GCMonitor;
import org.apache.jackrabbit.oak.spi.security.SecurityProvider;
import org.apache.jackrabbit.oak.spi.whiteboard.Registration;
import org.apache.jackrabbit.oak.spi.whiteboard.Whiteboard;
import org.apache.jackrabbit.oak.spi.whiteboard.WhiteboardUtils;
import org.apache.jackrabbit.oak.stats.Clock;
import org.apache.jackrabbit.oak.stats.StatisticManager;
import org.apache.jackrabbit.oak.util.GenericDescriptors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RepositoryImpl
implements JackrabbitRepository {
    public static final String REFRESH_INTERVAL = "oak.refresh-interval";
    public static final String RELAXED_LOCKING = "oak.relaxed-locking";
    private static final Logger log = LoggerFactory.getLogger(RepositoryImpl.class);
    protected final Whiteboard whiteboard;
    protected final boolean fastQueryResultSize;
    private final GenericDescriptors descriptors;
    private final ContentRepository contentRepository;
    private final SecurityProvider securityProvider;
    private final int observationQueueLength;
    private final CommitRateLimiter commitRateLimiter;
    private final Clock.Fast clock;
    private final DelegatingGCMonitor gcMonitor = new DelegatingGCMonitor();
    private final Registration gcMonitorRegistration;
    private final ThreadLocal<Long> threadSaveCount = new ThreadLocal();
    private final ScheduledExecutorService scheduledExecutor = RepositoryImpl.createListeningScheduledExecutorService();
    private final StatisticManager statisticManager;

    public RepositoryImpl(@Nonnull ContentRepository contentRepository, @Nonnull Whiteboard whiteboard, @Nonnull SecurityProvider securityProvider, int observationQueueLength, CommitRateLimiter commitRateLimiter) {
        this(contentRepository, whiteboard, securityProvider, observationQueueLength, commitRateLimiter, false);
    }

    public RepositoryImpl(@Nonnull ContentRepository contentRepository, @Nonnull Whiteboard whiteboard, @Nonnull SecurityProvider securityProvider, int observationQueueLength, CommitRateLimiter commitRateLimiter, boolean fastQueryResultSize) {
        this.contentRepository = (ContentRepository)Preconditions.checkNotNull((Object)contentRepository);
        this.whiteboard = (Whiteboard)Preconditions.checkNotNull((Object)whiteboard);
        this.securityProvider = (SecurityProvider)Preconditions.checkNotNull((Object)securityProvider);
        this.observationQueueLength = observationQueueLength;
        this.commitRateLimiter = commitRateLimiter;
        this.descriptors = this.determineDescriptors();
        this.statisticManager = new StatisticManager(whiteboard, this.scheduledExecutor);
        this.clock = new Clock.Fast(this.scheduledExecutor);
        this.gcMonitorRegistration = whiteboard.register(GCMonitor.class, (Object)this.gcMonitor, Collections.emptyMap());
        this.fastQueryResultSize = fastQueryResultSize;
    }

    public String[] getDescriptorKeys() {
        return this.descriptors.getKeys();
    }

    public boolean isStandardDescriptor(String key) {
        return this.descriptors.isStandardDescriptor(key);
    }

    public String getDescriptor(String key) {
        try {
            Value v = this.getDescriptorValue(key);
            return v == null ? null : v.getString();
        }
        catch (RepositoryException e) {
            log.debug("Error converting value for descriptor with key {} to string", (Object)key);
            return null;
        }
    }

    public Value getDescriptorValue(String key) {
        return this.descriptors.getValue(key);
    }

    public Value[] getDescriptorValues(String key) {
        return this.descriptors.getValues(key);
    }

    public boolean isSingleValueDescriptor(String key) {
        return this.descriptors.isSingleValueDescriptor(key);
    }

    public Session login(@Nullable Credentials credentials, @Nullable String workspaceName) throws RepositoryException {
        return this.login(credentials, workspaceName, null);
    }

    public Session login() throws RepositoryException {
        return this.login(null, null, null);
    }

    public Session login(Credentials credentials) throws RepositoryException {
        return this.login(credentials, null, null);
    }

    public Session login(String workspace) throws RepositoryException {
        return this.login(null, workspace, null);
    }

    public Session login(@CheckForNull Credentials credentials, @CheckForNull String workspaceName, @CheckForNull Map<String, Object> attributes) throws RepositoryException {
        try {
            Long refreshInterval;
            if (attributes == null) {
                attributes = Collections.emptyMap();
            }
            if ((refreshInterval = RepositoryImpl.getRefreshInterval(credentials)) == null) {
                refreshInterval = RepositoryImpl.getRefreshInterval(attributes);
            } else if (attributes.containsKey(REFRESH_INTERVAL)) {
                throw new RepositoryException("Duplicate attribute 'oak.refresh-interval'.");
            }
            boolean relaxedLocking = RepositoryImpl.getRelaxedLocking(attributes);
            RefreshStrategy.Timed refreshStrategy = refreshInterval == null ? new RefreshStrategy.LogOnce(60L) : new RefreshStrategy.Timed(refreshInterval);
            ContentSession contentSession = this.contentRepository.login(credentials, workspaceName);
            SessionDelegate sessionDelegate = this.createSessionDelegate(refreshStrategy, contentSession);
            SessionContext context = this.createSessionContext(this.statisticManager, this.securityProvider, RepositoryImpl.createAttributes(refreshInterval, relaxedLocking), sessionDelegate, this.observationQueueLength, this.commitRateLimiter);
            return context.getSession();
        }
        catch (javax.security.auth.login.LoginException e) {
            throw new LoginException(e.getMessage(), (Throwable)e);
        }
    }

    private SessionDelegate createSessionDelegate(RefreshStrategy refreshStrategy, ContentSession contentSession) {
        final RefreshOnGC refreshOnGC = new RefreshOnGC(this.gcMonitor);
        refreshStrategy = RefreshStrategy.Composite.create(refreshStrategy, refreshOnGC);
        return new SessionDelegate(contentSession, this.securityProvider, refreshStrategy, this.threadSaveCount, this.statisticManager, (Clock)this.clock){
            RegistrationTask registrationTask;
            ScheduledFuture<?> scheduledTask;
            {
                super(x0, x1, x2, x3, x4, x5);
                this.registrationTask = new RegistrationTask(this.getSessionStats(), RepositoryImpl.this.whiteboard);
                this.scheduledTask = RepositoryImpl.this.scheduledExecutor.schedule(this.registrationTask, 1L, TimeUnit.MINUTES);
            }

            @Override
            public void logout() {
                refreshOnGC.close();
                this.registrationTask.cancel();
                this.scheduledTask.cancel(false);
                super.logout();
            }
        };
    }

    public void shutdown() {
        this.statisticManager.dispose();
        this.gcMonitorRegistration.unregister();
        this.clock.close();
        this.closeExecutor();
        if (this.contentRepository instanceof Closeable) {
            IOUtils.closeQuietly((Closeable)((Closeable)this.contentRepository));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeExecutor() {
        try {
            this.scheduledExecutor.shutdown();
            this.scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            log.error("Error while shutting down the executorService", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        finally {
            if (!this.scheduledExecutor.isTerminated()) {
                log.warn("executorService didn't shutdown properly. Will be forced now.");
            }
            this.scheduledExecutor.shutdownNow();
        }
    }

    protected SessionContext createSessionContext(StatisticManager statisticManager, SecurityProvider securityProvider, Map<String, Object> attributes, SessionDelegate delegate, int observationQueueLength, CommitRateLimiter commitRateLimiter) {
        return new SessionContext((Repository)this, statisticManager, securityProvider, this.whiteboard, attributes, delegate, observationQueueLength, commitRateLimiter, this.fastQueryResultSize);
    }

    protected GenericDescriptors determineDescriptors() {
        return new JcrDescriptorsImpl(this.contentRepository.getDescriptors(), (ValueFactory)new SimpleValueFactory());
    }

    protected GenericDescriptors getDescriptors() {
        return this.descriptors;
    }

    private static ScheduledExecutorService createListeningScheduledExecutorService() {
        ThreadFactory tf = new ThreadFactory(){
            private final AtomicLong counter = new AtomicLong();

            @Override
            public Thread newThread(@Nonnull Runnable r) {
                Thread t = new Thread(r, this.newName());
                t.setDaemon(true);
                return t;
            }

            private String newName() {
                return "oak-repository-executor-" + this.counter.incrementAndGet();
            }
        };
        return new ScheduledThreadPoolExecutor(1, tf){

            @Override
            public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
                this.purge();
                return super.schedule(callable, delay, unit);
            }

            @Override
            public ScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit) {
                this.purge();
                return super.schedule(command, delay, unit);
            }

            @Override
            public ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
                this.purge();
                return super.scheduleAtFixedRate(command, initialDelay, period, unit);
            }

            @Override
            public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit) {
                this.purge();
                return super.scheduleWithFixedDelay(command, initialDelay, delay, unit);
            }
        };
    }

    private static Long getRefreshInterval(Credentials credentials) {
        String value;
        if (credentials instanceof SimpleCredentials) {
            Object value2 = ((SimpleCredentials)credentials).getAttribute(REFRESH_INTERVAL);
            return RepositoryImpl.toLong(value2);
        }
        if (credentials instanceof TokenCredentials && (value = ((TokenCredentials)credentials).getAttribute(REFRESH_INTERVAL)) != null) {
            return RepositoryImpl.toLong(value);
        }
        return null;
    }

    private static Long getRefreshInterval(Map<String, Object> attributes) {
        return RepositoryImpl.toLong(attributes.get(REFRESH_INTERVAL));
    }

    private static boolean getRelaxedLocking(Map<String, Object> attributes) {
        Object value = attributes.get(RELAXED_LOCKING);
        if (value instanceof Boolean) {
            return (Boolean)value;
        }
        if (value instanceof String) {
            return Boolean.parseBoolean((String)value);
        }
        return false;
    }

    private static Long toLong(Object value) {
        if (value instanceof Long) {
            return (Long)value;
        }
        if (value instanceof Integer) {
            return ((Integer)value).longValue();
        }
        if (value instanceof String) {
            return RepositoryImpl.toLong((String)value);
        }
        return null;
    }

    private static Long toLong(String longValue) {
        try {
            return Long.valueOf(longValue);
        }
        catch (NumberFormatException e) {
            log.warn("Invalid value '" + longValue + "' for " + REFRESH_INTERVAL + ". Expected long. ", (Throwable)e);
            return null;
        }
    }

    private static Map<String, Object> createAttributes(Long refreshInterval, boolean relaxedLocking) {
        if (refreshInterval == null && !relaxedLocking) {
            return Collections.emptyMap();
        }
        if (refreshInterval == null) {
            return Collections.singletonMap(RELAXED_LOCKING, relaxedLocking);
        }
        if (!relaxedLocking) {
            return Collections.singletonMap(REFRESH_INTERVAL, refreshInterval);
        }
        return ImmutableMap.of((Object)REFRESH_INTERVAL, (Object)refreshInterval, (Object)RELAXED_LOCKING, (Object)relaxedLocking);
    }

    private static class RegistrationTask
    implements Runnable {
        private final SessionStats sessionStats;
        private final Whiteboard whiteboard;
        private boolean cancelled;
        private Registration completed;

        public RegistrationTask(SessionStats sessionStats, Whiteboard whiteboard) {
            this.sessionStats = sessionStats;
            this.whiteboard = whiteboard;
        }

        @Override
        public synchronized void run() {
            if (!this.cancelled) {
                this.completed = WhiteboardUtils.registerMBean((Whiteboard)this.whiteboard, SessionMBean.class, (Object)this.sessionStats, (String)"SessionStatistics", (String)this.sessionStats.toString());
            }
        }

        public synchronized void cancel() {
            this.cancelled = true;
            if (this.completed != null) {
                this.completed.unregister();
                this.completed = null;
            }
        }
    }

    private static class RefreshOnGC
    extends GCMonitor.Empty
    implements RefreshStrategy {
        private final Registration registration;
        private volatile boolean compacted;

        public RefreshOnGC(DelegatingGCMonitor gcMonitor) {
            this.registration = gcMonitor.registerGCMonitor((GCMonitor)this);
        }

        public void close() {
            this.registration.unregister();
        }

        public void compacted(long[] segmentCounts, long[] recordCounts, long[] compactionMapWeights) {
            this.compacted = true;
        }

        @Override
        public boolean needsRefresh(long secondsSinceLastAccess) {
            return this.compacted;
        }

        @Override
        public void refreshed() {
            this.compacted = false;
        }

        public String toString() {
            return "Refresh on revision garbage collection";
        }
    }
}

