/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob.migration;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.CommitFailedException;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.plugins.blob.BlobStoreBlob;
import org.apache.jackrabbit.oak.plugins.blob.migration.DepthFirstNodeIterator;
import org.apache.jackrabbit.oak.plugins.memory.PropertyBuilder;
import org.apache.jackrabbit.oak.spi.blob.BlobStore;
import org.apache.jackrabbit.oak.spi.blob.split.SplitBlobStore;
import org.apache.jackrabbit.oak.spi.commit.CommitInfo;
import org.apache.jackrabbit.oak.spi.commit.EmptyHook;
import org.apache.jackrabbit.oak.spi.state.ChildNodeEntry;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlobMigrator {
    private static final Logger log = LoggerFactory.getLogger(BlobMigrator.class);
    private static final int MERGE_LIMIT = 100;
    private static final int MERGE_TIMEOUT = 30;
    private final SplitBlobStore blobStore;
    private final NodeStore nodeStore;
    private final AtomicBoolean stopMigration = new AtomicBoolean(false);
    private DepthFirstNodeIterator nodeIterator;
    private NodeBuilder rootBuilder;
    private long lastCommit;
    private int migratedNodes;
    private volatile String lastPath;
    private volatile int totalMigratedNodes;

    public BlobMigrator(SplitBlobStore blobStore, NodeStore nodeStore) {
        this.blobStore = blobStore;
        this.nodeStore = nodeStore;
        this.refreshAndReset();
    }

    public boolean start() throws IOException {
        this.totalMigratedNodes = 0;
        this.refreshAndReset();
        return this.migrate();
    }

    public boolean migrate() throws IOException {
        while (true) {
            if (this.nodeIterator.hasNext()) {
                this.lastPath = this.nodeIterator.getPath();
                if (this.stopMigration.getAndSet(false)) {
                    if (this.migratedNodes > 0) {
                        this.tryCommit();
                    }
                    return false;
                }
                this.migrateNode(this.rootBuilder, this.nodeIterator);
                if (!this.timeToCommit()) continue;
                this.tryCommit();
                continue;
            }
            if (this.migratedNodes <= 0 || this.tryCommit()) break;
        }
        return true;
    }

    private boolean tryCommit() {
        try {
            this.nodeStore.merge(this.rootBuilder, EmptyHook.INSTANCE, CommitInfo.EMPTY);
            this.totalMigratedNodes += this.migratedNodes;
            log.info("{} nodes merged succesfully. Nodes migrated in this session: {}", (Object)this.migratedNodes, (Object)this.totalMigratedNodes);
            this.lastCommit = System.currentTimeMillis();
            this.migratedNodes = 0;
            return true;
        }
        catch (CommitFailedException e) {
            log.error("Can't commit. Resetting the migrator", (Throwable)e);
            this.refreshAndReset();
            return false;
        }
    }

    private boolean timeToCommit() {
        long changesMerged = (System.currentTimeMillis() - this.lastCommit) / 1000L;
        if (this.migratedNodes >= 100) {
            log.info("Migrated nodes count: {}. Merging changes.", (Object)this.migratedNodes);
            return true;
        }
        if (this.migratedNodes > 0 && changesMerged >= 30L) {
            log.info("Changes have been merged {}s ago. Merging {} nodes.", (Object)changesMerged, (Object)this.migratedNodes);
            return true;
        }
        return false;
    }

    public void stop() {
        this.stopMigration.set(true);
    }

    public String getLastProcessedPath() {
        return this.lastPath;
    }

    public int getTotalMigratedNodes() {
        return this.totalMigratedNodes;
    }

    private void refreshAndReset() {
        NodeState rootState = this.nodeStore.getRoot();
        this.rootBuilder = rootState.builder();
        this.nodeIterator = new DepthFirstNodeIterator(rootState);
        this.lastPath = null;
        this.lastCommit = System.currentTimeMillis();
        this.migratedNodes = 0;
    }

    private void migrateNode(NodeBuilder rootBuilder, DepthFirstNodeIterator iterator) throws IOException {
        ChildNodeEntry node = (ChildNodeEntry)iterator.next();
        NodeState state = node.getNodeState();
        for (PropertyState property : state.getProperties()) {
            Object newProperty = property.getType() == Type.BINARY ? this.migrateProperty(property) : (property.getType() == Type.BINARIES ? this.migrateMultiProperty(property) : null);
            if (newProperty == null) continue;
            NodeBuilder builder = iterator.getBuilder(rootBuilder);
            if (builder.exists()) {
                builder.setProperty(newProperty);
                ++this.migratedNodes;
                log.debug("Migrated property {}/{}", (Object)this.lastPath, (Object)property.getName());
                continue;
            }
            log.warn("Can't migrate blobs for a non-existing node: {}", (Object)this.lastPath);
        }
    }

    private PropertyState migrateProperty(PropertyState propertyState) throws IOException {
        Blob oldBlob = (Blob)propertyState.getValue(Type.BINARY);
        String blobId = this.getIdentity(oldBlob);
        if (this.blobStore.isMigrated(blobId)) {
            return null;
        }
        String newBlobId = this.blobStore.writeBlob(oldBlob.getNewStream());
        BlobStoreBlob newBlob = new BlobStoreBlob((BlobStore)this.blobStore, newBlobId);
        PropertyBuilder builder = new PropertyBuilder(Type.BINARY);
        builder.assignFrom(propertyState);
        builder.setValue((Object)newBlob);
        return builder.getPropertyState();
    }

    private PropertyState migrateMultiProperty(PropertyState propertyState) throws IOException {
        Iterable oldBlobs = (Iterable)propertyState.getValue(Type.BINARIES);
        ArrayList<BlobStoreBlob> newBlobs = new ArrayList<BlobStoreBlob>();
        PropertyBuilder builder = new PropertyBuilder(Type.BINARY);
        builder.assignFrom(propertyState);
        boolean blobUpdated = false;
        for (Blob oldBlob : oldBlobs) {
            String blobId = this.getIdentity(oldBlob);
            if (this.blobStore.isMigrated(blobId)) {
                newBlobs.add(new BlobStoreBlob((BlobStore)this.blobStore, blobId));
                continue;
            }
            String newBlobId = this.blobStore.writeBlob(oldBlob.getNewStream());
            BlobStoreBlob newBlob = new BlobStoreBlob((BlobStore)this.blobStore, newBlobId);
            newBlobs.add(newBlob);
            blobUpdated = true;
        }
        if (blobUpdated) {
            builder.setValues(newBlobs);
            return builder.getPropertyState();
        }
        return null;
    }

    private String getIdentity(Blob blob) throws IOException {
        String id = blob.getContentIdentity();
        if (id == null) {
            id = DigestUtils.shaHex((InputStream)blob.getNewStream());
        }
        return id;
    }
}

